"""
Use Case routes - Complex CRUD operations for usecase, details, paragraphs, and associations.
"""

from flask import Blueprint, request, jsonify
from db import get_db_connection
import json
import os
import uuid
from utils import delete_file, delete_files, get_asset_url, normalize_file_path

use_case_bp = Blueprint('use_case', __name__)

RELATIONS = {
    'keywords': {'table': 'usecase_schema.keyword_usecase', 'col': 'keyword_id'},
    'frameworks': {'table': 'usecase_schema.framework_usecase', 'col': 'framework_id'},
    'models': {'table': 'usecase_schema.model_usecase', 'col': 'model_id'},
    'hardwares': {'table': 'usecase_schema.hardware_usecase', 'col': 'hardware_id'}
}

# The diagram shows 'model_id' being used in all 4 junction tables.
# keyword_usecase: model_id FK keyword(id)
# framework_usecase: model_id FK framework(id)
# ... etc.

def _update_relations(cursor, usecase_id, type_key, new_ids):
    """Sync many-to-many relations."""
    rel = RELATIONS[type_key]
    table = rel['table']
    col = rel['col']
    
    # Delete old
    cursor.execute(f"DELETE FROM {table} WHERE usecase_id = %s", (usecase_id,))
    
    # Insert new
    if new_ids:
        # Deduplicate
        new_ids = list(set(new_ids))
        values = [(str(nid), str(usecase_id)) for nid in new_ids]
        args_str = ','.join(cursor.mogrify("(%s,%s)", x).decode('utf-8') for x in values)
        cursor.execute(f"INSERT INTO {table} ({col}, usecase_id) VALUES {args_str}")

@use_case_bp.route('/use-cases', methods=['GET'])
def get_all_use_cases():
    """Get all use cases with basic info."""
    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        cursor.execute("SELECT * FROM usecase_schema.usecase ORDER BY title")
        items = cursor.fetchall()
        
        result = []
        for item in items:
            d = dict(item)
            d['image_src'] = get_asset_url(d.get('image_src'))
            result.append(d)
            
        return jsonify(result), 200
    except Exception as e:
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()
@use_case_bp.route('/use-cases/<uuid:id>', methods=['GET'])
def get_use_case(id):
    """Get full use case details."""
    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        
        # 1. Fetch UseCase
        cursor.execute("SELECT * FROM usecase_schema.usecase WHERE id = %s", (str(id),))
        uc = cursor.fetchone()
        if not uc:
             return jsonify({'error': 'Not found'}), 404
        
        result = dict(uc)
        result['image_src'] = get_asset_url(result.get('image_src'))
        
        # 2. Fetch Detail
        if result.get('usecase_detail_id'):
            cursor.execute("SELECT * FROM usecase_schema.usecase_detail WHERE id = %s", (str(result['usecase_detail_id']),))
            detail = cursor.fetchone()
            if detail:
                d_dict = dict(detail)
                d_dict['main_image'] = get_asset_url(d_dict.get('main_image'))
                # Safe JSON load if stored as text/jsonb in DB or dict already
                # psycopg2 usually returns dict for jsonb, but if text/str:
                for col in ['publications', 'dataset', 'source_code']:
                    if isinstance(d_dict.get(col), str):
                         try:
                             d_dict[col] = json.loads(d_dict[col])
                         except:
                             d_dict[col] = [] if col == 'publications' else {}
                result['detail'] = d_dict
            else:
                result['detail'] = {}
        else:
             result['detail'] = {}

        # 3. Fetch Paragraphs
        if result.get('usecase_detail_id'):
             cursor.execute("""
                SELECT * FROM usecase_schema.paragraph 
                WHERE usecase_detail_id = %s 
                ORDER BY order_index
             """, (str(result['usecase_detail_id']),))
             paras = cursor.fetchall()
             para_list = []
             for p in paras:
                 pd = dict(p)
                 pd['image'] = get_asset_url(pd.get('image'))
                 # pd['content'] usually text[], handle it
                 # if stored as array in PG, psycopg2 returns list.
                 para_list.append(pd)
             result['paragraphs'] = para_list
        else:
             result['paragraphs'] = []

        # 4. Fetch Relations
        # keywords
        cursor.execute("SELECT keyword_id FROM usecase_schema.keyword_usecase WHERE usecase_id = %s", (str(id),))
        result['keywords'] = [str(r['keyword_id']) for r in cursor.fetchall()]
        
        # frameworks
        cursor.execute("SELECT framework_id FROM usecase_schema.framework_usecase WHERE usecase_id = %s", (str(id),))
        result['frameworks'] = [str(r['framework_id']) for r in cursor.fetchall()]

        # models
        cursor.execute("SELECT model_id FROM usecase_schema.model_usecase WHERE usecase_id = %s", (str(id),))
        result['models'] = [str(r['model_id']) for r in cursor.fetchall()]

        # hardwares
        cursor.execute("SELECT hardware_id FROM usecase_schema.hardware_usecase WHERE usecase_id = %s", (str(id),))
        result['hardwares'] = [str(r['hardware_id']) for r in cursor.fetchall()]

        return jsonify(result), 200

    except Exception as e:
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()
        
@use_case_bp.route('/use-cases', methods=['POST'])
def create_use_case():
    """Create new use case."""
    data = request.get_json()
    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        
        # 1. Create Detail
        detail_data = data.get('detail', {})
        new_detail_uuid = str(uuid.uuid4())
        
        cursor.execute("""
            INSERT INTO usecase_schema.usecase_detail 
            (id, title, caption, main_image, publications, dataset, source_code)
            VALUES (%s, %s, %s, %s, %s, %s, %s)
            RETURNING id
        """, (
            new_detail_uuid,
            detail_data.get('title', ''),
            detail_data.get('caption', ''),
            normalize_file_path(detail_data.get('main_image')),
            json.dumps(detail_data.get('publications', [])),
            json.dumps(detail_data.get('dataset', {})),
            json.dumps(detail_data.get('source_code', {}))
        ))
        detail_id = cursor.fetchone()['id']
        
        # 2. Create UseCase
        cursor.execute("""
            INSERT INTO usecase_schema.usecase 
            (title, description, image_src, usecase_detail_id)
            VALUES (%s, %s, %s, %s)
            RETURNING *
        """, (
            data.get('title'),
            data.get('description'),
            normalize_file_path(data.get('image_src')),
            str(detail_id)
        ))
        new_uc = cursor.fetchone()
        new_uc_id = new_uc['id']
        
        # 3. Create Paragraphs
        paragraphs = data.get('paragraphs', [])
        for idx, p in enumerate(paragraphs):
            cursor.execute("""
                INSERT INTO usecase_schema.paragraph
                (title, content, image, order_index, usecase_detail_id)
                VALUES (%s, %s, %s, %s, %s)
            """, (
                p.get('title'),
                p.get('content', []),
                normalize_file_path(p.get('image')),
                idx,
                str(detail_id)
            ))
            
        # 4. Relations
        _update_relations(cursor, str(new_uc_id), 'keywords', data.get('keywords', []))
        _update_relations(cursor, str(new_uc_id), 'frameworks', data.get('frameworks', []))
        _update_relations(cursor, str(new_uc_id), 'models', data.get('models', []))
        _update_relations(cursor, str(new_uc_id), 'hardwares', data.get('hardwares', []))
        
        conn.commit()
        
        # Return complete object with image URL
        res = dict(new_uc)
        res['image_src'] = get_asset_url(res.get('image_src'))
        return jsonify(res), 201

    except Exception as e:
        conn.rollback()
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()

@use_case_bp.route('/use-cases/<uuid:id>', methods=['PUT'])
def update_use_case(id):
    """Update use case."""
    data = request.get_json()
    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        
        # Fetch current usecase to check old files and detail_id
        cursor.execute("SELECT * FROM usecase_schema.usecase WHERE id = %s", (str(id),))
        current_uc = cursor.fetchone()
        if not current_uc:
            return jsonify({'error': 'Not found'}), 404
            
        current_detail_id = current_uc['usecase_detail_id']
        
        # File Cleanup: UseCase Image
        old_img = current_uc['image_src']
        new_img = normalize_file_path(data.get('image_src'))
        if old_img and old_img != new_img:
            delete_file(old_img)
        
        # 1. Update UseCase
        cursor.execute("""
            UPDATE usecase_schema.usecase 
            SET title = %s, description = %s, image_src = %s
            WHERE id = %s
        """, (
            data.get('title'),
            data.get('description'),
            new_img,
            str(id)
        ))
        
        # 2. Update/Create Detail
        detail_data = data.get('detail', {})
        new_detail_id = current_detail_id
        
        if current_detail_id:
             # Fetch current detail for file cleanup
             cursor.execute("SELECT main_image FROM usecase_schema.usecase_detail WHERE id = %s", (str(current_detail_id),))
             curr_det = cursor.fetchone()
             if curr_det:
                 old_main_img = curr_det['main_image']
                 new_main_img = normalize_file_path(detail_data.get('main_image'))
                 if old_main_img and old_main_img != new_main_img:
                     delete_file(old_main_img)
             
             # Update Detail
             cursor.execute("""
                UPDATE usecase_schema.usecase_detail
                SET title = %s, caption = %s, main_image = %s, 
                    publications = %s, dataset = %s, source_code = %s,
                    update_time = CURRENT_TIMESTAMP
                WHERE id = %s
             """, (
                 detail_data.get('title'),
                 detail_data.get('caption'),
                 normalize_file_path(detail_data.get('main_image')),
                 json.dumps(detail_data.get('publications', [])),
                 json.dumps(detail_data.get('dataset', {})),
                 json.dumps(detail_data.get('source_code', {})),
                 str(current_detail_id)
             ))
        elif detail_data:
             # Create Detail
             new_uuid = str(uuid.uuid4())
             cursor.execute("""
                INSERT INTO usecase_schema.usecase_detail 
                (id, title, caption, main_image, publications, dataset, source_code)
                VALUES (%s, %s, %s, %s, %s, %s, %s)
                RETURNING id
             """, (
                 new_uuid,
                 detail_data.get('title'),
                 detail_data.get('caption'),
                 normalize_file_path(detail_data.get('main_image')),
                 json.dumps(detail_data.get('publications', [])),
                 json.dumps(detail_data.get('dataset', {})),
                 json.dumps(detail_data.get('source_code', {}))
             ))
             new_detail_id = cursor.fetchone()['id']
             # Link to UseCase
             cursor.execute("UPDATE usecase_schema.usecase SET usecase_detail_id = %s WHERE id = %s", (str(new_detail_id), str(id)))

        # Verification: Ensure new_detail_id actually refers to an existing record.
        if new_detail_id:
             cursor.execute("SELECT 1 FROM usecase_schema.usecase_detail WHERE id = %s", (str(new_detail_id),))
             if not cursor.fetchone():
                 # Detail ID exists on UseCase but record is missing. Create it now.
                 new_uuid = str(uuid.uuid4())
                 cursor.execute("""
                    INSERT INTO usecase_schema.usecase_detail 
                    (id, title, caption, main_image, publications, dataset, source_code)
                    VALUES (%s, %s, %s, %s, %s, %s, %s)
                    RETURNING id
                 """, (
                     new_uuid,
                     detail_data.get('title', ''),
                     detail_data.get('caption', ''),
                     detail_data.get('main_image', None),
                     json.dumps(detail_data.get('publications', [])),
                     json.dumps(detail_data.get('dataset', {})),
                     json.dumps(detail_data.get('source_code', {}))
                 ))
                 new_detail_id = cursor.fetchone()['id']
                 cursor.execute("UPDATE usecase_schema.usecase SET usecase_detail_id = %s WHERE id = %s", (str(new_detail_id), str(id)))

        # 3. Handle Paragraphs
        if new_detail_id:
            paragraphs = data.get('paragraphs', [])
            
            # Fetch existing paragraphs to handle deletions and file cleanup
            cursor.execute("SELECT id, image FROM usecase_schema.paragraph WHERE usecase_detail_id = %s", (str(new_detail_id),))
            existing_paras_rows = cursor.fetchall()
            existing_map = {str(r['id']): r for r in existing_paras_rows}
            
            input_ids = [p.get('id') for p in paragraphs if p.get('id')]
            
            # Delete removed paragraphs
            for pid, p_row in existing_map.items():
                if pid not in input_ids:
                    # Cleanup file
                    if p_row['image']:
                        delete_file(p_row['image'])
                    cursor.execute("DELETE FROM usecase_schema.paragraph WHERE id = %s", (pid,))
            
            # Upsert paragraphs
            for idx, p in enumerate(paragraphs):
                p_id = p.get('id')
                if p_id and p_id in existing_map:
                    # Update
                    old_para_img = existing_map[p_id]['image']
                    new_para_img = normalize_file_path(p.get('image'))
                    if old_para_img and old_para_img != new_para_img:
                        delete_file(old_para_img)
                    
                    
                    cursor.execute("""
                        UPDATE usecase_schema.paragraph
                        SET title = %s, content = %s, image = %s, order_index = %s
                        WHERE id = %s
                    """, (
                        p.get('title'),
                        p.get('content', []),
                        new_para_img,
                        idx,
                        p_id
                    ))
                else:
                    # Insert
                    cursor.execute("""
                        INSERT INTO usecase_schema.paragraph
                        (title, content, image, order_index, usecase_detail_id)
                        VALUES (%s, %s, %s, %s, %s)
                    """, (
                        p.get('title'),
                        p.get('content', []),
                        normalize_file_path(p.get('image')),
                        idx,
                        new_detail_id
                    ))

        # 4. Relations
        _update_relations(cursor, str(id), 'keywords', data.get('keywords', []))
        _update_relations(cursor, str(id), 'frameworks', data.get('frameworks', []))
        _update_relations(cursor, str(id), 'models', data.get('models', []))
        _update_relations(cursor, str(id), 'hardwares', data.get('hardwares', []))

        conn.commit()
        return jsonify({'message': 'Updated successfully'}), 200

    except Exception as e:
        import traceback
        traceback.print_exc()
        conn.rollback()
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()

@use_case_bp.route('/use-cases/<uuid:id>', methods=['DELETE'])
def delete_use_case(id):
    """Delete use case and cleanup."""
    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        
        # Fetch info for cleanup
        cursor.execute("SELECT image_src, usecase_detail_id FROM usecase_schema.usecase WHERE id = %s", (str(id),))
        uc = cursor.fetchone()
        
        if not uc:
             return jsonify({'error': 'Not found'}), 404
             
        # Delete UseCase Relations
        for rel in RELATIONS.values():
            cursor.execute(f"DELETE FROM {rel['table']} WHERE usecase_id = %s", (str(id),))
            
        # Delete UseCase row
        cursor.execute("DELETE FROM usecase_schema.usecase WHERE id = %s", (str(id),))
        
        files_to_delete = []
        if uc['image_src']:
            files_to_delete.append(uc['image_src'])
        
        # Handle Detail
        if uc['usecase_detail_id']:
            det_id = uc['usecase_detail_id']
            
            # Fetch paragraphs for cleanup
            cursor.execute("SELECT image FROM usecase_schema.paragraph WHERE usecase_detail_id = %s", (str(det_id),))
            paras = cursor.fetchall()
            for p in paras:
                if p['image']:
                    files_to_delete.append(p['image'])
            
            # Fetch detail for cleanup
            cursor.execute("SELECT main_image FROM usecase_schema.usecase_detail WHERE id = %s", (str(det_id),))
            det = cursor.fetchone()
            if det and det['main_image']:
                files_to_delete.append(det['main_image'])
            
            # Delete paragraphs
            cursor.execute("DELETE FROM usecase_schema.paragraph WHERE usecase_detail_id = %s", (str(det_id),))
            # Delete detail
            cursor.execute("DELETE FROM usecase_schema.usecase_detail WHERE id = %s", (str(det_id),))
        
        # Batch delete all collected files
        delete_files(files_to_delete)
            
        conn.commit()
        return jsonify({'message': 'Deleted successfully'}), 200

    except Exception as e:
        conn.rollback()
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()
