"""
Tools routes - CRUD operations for tools table.
"""

from flask import Blueprint, request, jsonify
from db import get_db_connection
import os
from utils import delete_file, delete_files, get_asset_url, normalize_file_path

tools_bp = Blueprint('tools', __name__)


@tools_bp.route('/tools', methods=['GET'])
def get_all_tools():
    """Get all tools."""
    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        cursor.execute("SELECT * FROM infrastructure_schema.supported_tools ORDER BY queue ASC, created_time DESC")
        items = cursor.fetchall()
        
        result = []
        for item in items:
            d = dict(item)
            d['image'] = get_asset_url(d.get('image'))
            result.append(d)
            
        return jsonify(result), 200
    except Exception as e:
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()


@tools_bp.route('/tools/<uuid:id>', methods=['GET'])
def get_tool(id):
    """Get tool by ID."""
    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        cursor.execute("SELECT * FROM infrastructure_schema.supported_tools WHERE id = %s", (str(id),))
        item = cursor.fetchone()
        if item:
            d = dict(item)
            d['image'] = get_asset_url(d.get('image'))
            return jsonify(d), 200
        return jsonify({'error': 'Tool not found'}), 404
    except Exception as e:
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()


@tools_bp.route('/tools', methods=['POST'])
def create_tool():
    """Create new tool."""
    data = request.get_json()
    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        cursor.execute("""
            INSERT INTO infrastructure_schema.supported_tools (
                title, content, link, documentation, image, queue
            )
            VALUES (%s, %s, %s, %s, %s, %s)
            RETURNING *
        """, (
            data.get('title'),
            data.get('content'),
            data.get('link'),
            data.get('documentation'),
            normalize_file_path(data.get('image')),
            data.get('queue', 0)
        ))
        item = cursor.fetchone()
        conn.commit()
        
        d = dict(item)
        d['image'] = get_asset_url(d.get('image'))
        return jsonify(d), 201
    except Exception as e:
        conn.rollback()
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()


@tools_bp.route('/tools/<uuid:id>', methods=['PUT'])
def update_tool(id):
    """Update tool by ID."""
    data = request.get_json()
    conn = get_db_connection()
    try:
        cursor = conn.cursor()

        # Check for image change to delete old file
        cursor.execute("SELECT image FROM infrastructure_schema.supported_tools WHERE id = %s", (str(id),))
        current_item = cursor.fetchone()
        
        old_image = current_item['image'] if current_item else None
        new_image = normalize_file_path(data.get('image'))
        
        if old_image and old_image != new_image:
            delete_file(old_image)

        cursor.execute("""
            UPDATE infrastructure_schema.supported_tools 
            SET title = %s, content = %s, link = %s, documentation = %s,
                image = %s, queue = %s, updated_time = CURRENT_TIMESTAMP
            WHERE id = %s
            RETURNING *
        """, (
            data.get('title'),
            data.get('content'),
            data.get('link'),
            data.get('documentation'),
            new_image,
            data.get('queue', 0),
            str(id)
        ))
        item = cursor.fetchone()
        conn.commit()
        if item:
            d = dict(item)
            d['image'] = get_asset_url(d.get('image'))
            return jsonify(d), 200
        return jsonify({'error': 'Tool not found'}), 404
    except Exception as e:
        conn.rollback()
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()


@tools_bp.route('/tools/<uuid:id>', methods=['DELETE'])
def delete_tool(id):
    """Delete tool by ID."""
    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        
        # Get image before delete
        cursor.execute("SELECT image FROM infrastructure_schema.supported_tools WHERE id = %s", (str(id),))
        item = cursor.fetchone()

        cursor.execute("DELETE FROM infrastructure_schema.supported_tools WHERE id = %s RETURNING id", (str(id),))
        deleted = cursor.fetchone()
        conn.commit()
        
        if deleted:
            if item:
                delete_files([item['image']])

            return jsonify({'message': 'Tool deleted'}), 200
        return jsonify({'error': 'Tool not found'}), 404
    except Exception as e:
        conn.rollback()
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()
