"""
Team Member (About Us) routes - CRUD operations for members table.
"""

from flask import Blueprint, request, jsonify
from db import get_db_connection
import os
from utils import delete_file, delete_files, get_asset_url, normalize_file_path

team_member_bp = Blueprint('team_member', __name__)


@team_member_bp.route('/team-members', methods=['GET'])
def get_all_team_members():
    """Get all team members."""
    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        # Ordering by queue first (assuming it controls display order), then active, then name
        cursor.execute("SELECT * FROM aboutus_schema.members ORDER BY queue ASC, active DESC, surname, name")
        items = cursor.fetchall()
        
        result = []
        for item in items:
            d = dict(item)
            d['image_url'] = get_asset_url(d.get('image_url'))
            result.append(d)
            
        return jsonify(result), 200
    except Exception as e:
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()


@team_member_bp.route('/team-members/<uuid:id>', methods=['GET'])
def get_team_member(id):
    """Get team member by ID."""
    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        cursor.execute("SELECT * FROM aboutus_schema.members WHERE id = %s", (str(id),))
        item = cursor.fetchone()
        if item:
            d = dict(item)
            d['image_url'] = get_asset_url(d.get('image_url'))
            return jsonify(d), 200
        return jsonify({'error': 'Team member not found'}), 404
    except Exception as e:
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()


@team_member_bp.route('/team-members', methods=['POST'])
def create_team_member():
    """Create new team member."""
    data = request.get_json()
    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        cursor.execute("""
            INSERT INTO aboutus_schema.members (
                name, surname, role, image_url, linkedin_profile,
                polito_profile, iris_profile, researchgate_profile,
                googlescholar_profile, orcid_profile, active, description, queue
            )
            VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s)
            RETURNING *
        """, (
            data.get('name'),
            data.get('surname'),
            data.get('role'),
            normalize_file_path(data.get('image_url')),
            data.get('linkedin_profile'),
            data.get('polito_profile'),
            data.get('iris_profile'),
            data.get('researchgate_profile'),
            data.get('googlescholar_profile'),
            data.get('orcid_profile'),
            data.get('active', False),
            data.get('description'),
            data.get('queue', 0)
        ))
        item = cursor.fetchone()
        conn.commit()
        
        d = dict(item)
        d['image_url'] = get_asset_url(d.get('image_url'))
        return jsonify(d), 201
    except Exception as e:
        conn.rollback()
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()


@team_member_bp.route('/team-members/<uuid:id>', methods=['PUT'])
def update_team_member(id):
    """Update team member by ID."""
    data = request.get_json()
    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        
        # Check if image_url is changing so we can delete the old file
        # 1. Fetch current image_url
        cursor.execute("SELECT image_url FROM aboutus_schema.members WHERE id = %s", (str(id),))
        current_member = cursor.fetchone()
        
        old_image_url = current_member['image_url'] if current_member else None
        new_image_url = normalize_file_path(data.get('image_url'))
        
        # 2. If old image exists and is different from new one, delete old file
        if old_image_url and old_image_url != new_image_url:
            delete_file(old_image_url)

        cursor.execute("""
            UPDATE aboutus_schema.members 
            SET name = %s, surname = %s, role = %s, image_url = %s,
                linkedin_profile = %s, polito_profile = %s, iris_profile = %s,
                researchgate_profile = %s, googlescholar_profile = %s,
                orcid_profile = %s, active = %s, description = %s, queue = %s,
                updated_at = CURRENT_TIMESTAMP
            WHERE id = %s
            RETURNING *
        """, (
            data.get('name'),
            data.get('surname'),
            data.get('role'),
            new_image_url,
            data.get('linkedin_profile'),
            data.get('polito_profile'),
            data.get('iris_profile'),
            data.get('researchgate_profile'),
            data.get('googlescholar_profile'),
            data.get('orcid_profile'),
            data.get('active', False),
            data.get('description'),
            data.get('queue', 0),
            str(id)
        ))
        item = cursor.fetchone()
        conn.commit()
        if item:
            d = dict(item)
            d['image_url'] = get_asset_url(d.get('image_url'))
            return jsonify(d), 200
        return jsonify({'error': 'Team member not found'}), 404
    except Exception as e:
        conn.rollback()
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()


@team_member_bp.route('/team-members/<uuid:id>', methods=['DELETE'])
def delete_team_member(id):
    """Delete team member by ID."""
    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        
        # 1. Get image_url before deleting
        cursor.execute("SELECT image_url FROM aboutus_schema.members WHERE id = %s", (str(id),))
        member = cursor.fetchone()
        
        # 2. Delete record
        cursor.execute("DELETE FROM aboutus_schema.members WHERE id = %s RETURNING id", (str(id),))
        deleted = cursor.fetchone()
        conn.commit()
        
        if deleted:
            # 3. Delete file if exists
            if member:
                delete_files([member['image_url']])
            
            return jsonify({'message': 'Team member deleted'}), 200
        return jsonify({'error': 'Team member not found'}), 404
    except Exception as e:
        conn.rollback()
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()
