"""
Publication routes - CRUD operations for publication_schema.publications table.
"""

from flask import Blueprint, request, jsonify
from db import get_db_connection
import os
from utils import delete_file, delete_files, get_asset_url, normalize_file_path

publication_bp = Blueprint('publication', __name__)

FILE_FIELDS = [
    'audio_url', 'video_url', 'mental_map_url', 'report_url',
    'flashcard_url', 'quiz_url', 'infografica_url', 'presentation_url',
    'datatable_url', 'full_paper_url'
]

@publication_bp.route('/publications', methods=['GET'])
def get_all_publications():
    """Get all publications."""
    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        cursor.execute("""
            SELECT * FROM publication_schema.publications 
            ORDER BY 
                reviewed DESC,
                CASE WHEN state = 'in_progress' THEN 0 ELSE 1 END,
                publication_year DESC, 
                title
        """)
        items = cursor.fetchall()
        
        # Transform items to include full URLs
        result = []
        for item in items:
            d = dict(item)
            for field in FILE_FIELDS:
                d[field] = get_asset_url(d.get(field))
            result.append(d)
            
        return jsonify(result), 200
    except Exception as e:
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()


@publication_bp.route('/publications/<uuid:id>', methods=['GET'])
def get_publication(id):
    """Get publication by ID with its authors."""
    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        # Get publication
        cursor.execute("""
            SELECT * FROM publication_schema.publications WHERE id = %s
        """, (str(id),))
        item = cursor.fetchone()
        if not item:
            return jsonify({'error': 'Publication not found'}), 404
        
        # Get authors
        cursor.execute("""
            SELECT pa.*, a.name, a.surname
            FROM publication_schema.publication_authors pa
            LEFT JOIN innuce_schema.authors a ON pa.team_member_id = a.id
            WHERE pa.publication_id = %s
            ORDER BY pa.author_rank
        """, (str(id),))
        authors = cursor.fetchall()
        
        result = dict(item)
        # Add full URLs
        for field in FILE_FIELDS:
            result[field] = get_asset_url(result.get(field))
            
        result['authors'] = list(authors)
        return jsonify(result), 200
    except Exception as e:
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()


@publication_bp.route('/publications', methods=['POST'])
def create_publication():
    """Create new publication."""
    data = request.get_json()
    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        
        # Build query dynamically based on fields
        fields = [
            'handle', 'title', 'publication_year', 'venue', 'type', 'state', 'doi', 'url',
            'citation_raw', 'abstract', 'reviewed', 'keyword',
            'audio_url', 'video_url', 'mental_map_url', 'report_url',
            'flashcard_url', 'quiz_url', 'infografica_url', 'presentation_url', 'datatable_url',
            'full_paper_url', 'task_status'
        ]
        
        placeholders = ', '.join(['%s'] * len(fields))
        columns = ', '.join(fields)
        
        full_paper_path = normalize_file_path(data.get('full_paper_url'))
        # If full paper is uploaded, set status to pending to trigger auto-generation
        task_status = 'pending' if full_paper_path else None
        
        values = (
            data.get('handle', ''),
            data.get('title'),
            data.get('publication_year'),
            data.get('venue'),
            data.get('type'),
            data.get('state', 'in_progress'),
            data.get('doi'),
            data.get('url'),
            data.get('citation_raw'),
            data.get('abstract'),
            data.get('reviewed', False),
            data.get('keyword'),
            normalize_file_path(data.get('audio_url')),
            normalize_file_path(data.get('video_url')),
            normalize_file_path(data.get('mental_map_url')),
            normalize_file_path(data.get('report_url')),
            normalize_file_path(data.get('flashcard_url')),
            normalize_file_path(data.get('quiz_url')),
            normalize_file_path(data.get('infografica_url')),
            normalize_file_path(data.get('presentation_url')),
            normalize_file_path(data.get('datatable_url')),
            full_paper_path,
            task_status
        )
        
        cursor.execute(f"""
            INSERT INTO publication_schema.publications 
            ({columns})
            VALUES ({placeholders})
            RETURNING *
        """, values)
        
        item = cursor.fetchone()
        conn.commit()
        result = dict(item)
        for field in FILE_FIELDS:
            result[field] = get_asset_url(result.get(field))
        return jsonify(result), 201
    except Exception as e:
        conn.rollback()
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()


@publication_bp.route('/publications/<uuid:id>', methods=['PUT'])
def update_publication(id):
    """Update publication by ID."""
    data = request.get_json()
    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        
        # 1. Fetch current data for file clean up
        cursor.execute("SELECT * FROM publication_schema.publications WHERE id = %s", (str(id),))
        current_item = cursor.fetchone()
        
        if current_item:
            # Check each file field
            for field in FILE_FIELDS:
                old_val = current_item.get(field)
                new_val = normalize_file_path(data.get(field))
                
                # If old file exists and is different from new one, delete old file
                if old_val and old_val != new_val:
                    delete_file(old_val)

        # Determine if full paper changed to trigger regeneration
        new_full_paper = normalize_file_path(data.get('full_paper_url'))
        old_full_paper = current_item.get('full_paper_url') if current_item else None
        
        # Logic: 
        # 1. If new paper uploaded (new != old), set to pending
        # 2. If paper removed (new is None), set to None (or completed?) - likely None/cancelled
        # 3. If paper unchanged, keep existing status (do not update column or keep same)
        
        task_status_val = current_item.get('task_status')
        if new_full_paper != old_full_paper:
            if new_full_paper:
                task_status_val = 'pending' # New paper -> trigger generation
            else:
                task_status_val = None # Paper removed -> clear status

        # 2. Update record
        cursor.execute("""
            UPDATE publication_schema.publications 
            SET title = %s, publication_year = %s, venue = %s, type = %s, 
                state = %s, doi = %s, url = %s, citation_raw = %s, 
                abstract = %s, reviewed = %s, keyword = %s,
                audio_url = %s, video_url = %s, mental_map_url = %s, report_url = %s,
                flashcard_url = %s, quiz_url = %s, infografica_url = %s, 
                presentation_url = %s, datatable_url = %s, full_paper_url = %s,
                task_status = %s
            WHERE id = %s
            RETURNING *
        """, (
            data.get('title'),
            data.get('publication_year'),
            data.get('venue'),
            data.get('type'),
            data.get('state'),
            data.get('doi'),
            data.get('url'),
            data.get('citation_raw'),
            data.get('abstract'),
            data.get('reviewed'),
            data.get('keyword'),
            normalize_file_path(data.get('audio_url')),
            normalize_file_path(data.get('video_url')),
            normalize_file_path(data.get('mental_map_url')),
            normalize_file_path(data.get('report_url')),
            normalize_file_path(data.get('flashcard_url')),
            normalize_file_path(data.get('quiz_url')),
            normalize_file_path(data.get('infografica_url')),
            normalize_file_path(data.get('presentation_url')),
            normalize_file_path(data.get('datatable_url')),
            new_full_paper,
            task_status_val,
            str(id)
        ))
        item = cursor.fetchone()
        conn.commit()
        if item:
            result = dict(item)
            for field in FILE_FIELDS:
                result[field] = get_asset_url(result.get(field))
            return jsonify(result), 200
        return jsonify({'error': 'Publication not found'}), 404
    except Exception as e:
        conn.rollback()
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()


@publication_bp.route('/publications/<uuid:id>', methods=['DELETE'])
def delete_publication(id):
    """Delete publication by ID."""
    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        
        # 1. Fetch current item for file deletion
        cursor.execute("SELECT * FROM publication_schema.publications WHERE id = %s", (str(id),))
        item = cursor.fetchone()

        # 2. Delete
        cursor.execute("""
            DELETE FROM publication_schema.publications 
            WHERE id = %s RETURNING id
        """, (str(id),))
        deleted = cursor.fetchone()
        conn.commit()
        
        if deleted:
            # 3. Clean up files
            if item:
                file_list = [item.get(field) for field in FILE_FIELDS if item.get(field)]
                delete_files(file_list)

            return jsonify({'message': 'Publication deleted'}), 200
        return jsonify({'error': 'Publication not found'}), 404
    except Exception as e:
        conn.rollback()
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()


# Publication-Author relationship routes
@publication_bp.route('/publications/<uuid:pub_id>/authors/<uuid:author_id>', methods=['POST'])
def add_author_to_publication(pub_id, author_id):
    """Add author to publication."""
    data = request.get_json() or {}
    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        cursor.execute("""
            INSERT INTO publication_schema.publication_authors 
            (publication_id, team_member_id, full_name, author_rank, is_corresponding, is_internal)
            VALUES (%s, %s, %s, %s, %s, %s)
            RETURNING *
        """, (
            str(pub_id),
            str(author_id),
            data.get('full_name', ''),
            data.get('author_rank', 1),
            data.get('is_corresponding', False),
            data.get('is_internal', True)
        ))
        item = cursor.fetchone()
        conn.commit()
        return jsonify(dict(item)), 201
    except Exception as e:
        conn.rollback()
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()


@publication_bp.route('/publications/<uuid:pub_id>/authors/<uuid:author_id>', methods=['DELETE'])
def remove_author_from_publication(pub_id, author_id):
    """Remove author from publication."""
    conn = get_db_connection()
    try:
        cursor = conn.cursor()
        cursor.execute("""
            DELETE FROM publication_schema.publication_authors 
            WHERE publication_id = %s AND team_member_id = %s 
            RETURNING id
        """, (str(pub_id), str(author_id)))
        deleted = cursor.fetchone()
        conn.commit()
        if deleted:
            return jsonify({'message': 'Author removed from publication'}), 200
        return jsonify({'error': 'Relationship not found'}), 404
    except Exception as e:
        conn.rollback()
        return jsonify({'error': str(e)}), 500
    finally:
        cursor.close()
        conn.close()
