"""
Main Flask application for admin website backend.
Registers all blueprints and runs on port 5001.
"""

import sys
import os

# Add src directory to path for imports
sys.path.insert(0, os.path.dirname(os.path.abspath(__file__)))

from flask import Flask, jsonify

# Import blueprints
from routes.hardware import hardware_bp
from routes.tools import tools_bp
from routes.team_member import team_member_bp
from routes.use_case import use_case_bp
from routes.publication import publication_bp
from routes.upload import upload_bp
from routes.use_case_attributes import use_case_attributes_bp


def create_app():
    """Create and configure the Flask application."""
    app = Flask(__name__)
    app.config['MAX_CONTENT_LENGTH'] = 50 * 1024 * 1024 # 50MB limit
    
    # Register blueprints with /api prefix
    app.register_blueprint(hardware_bp, url_prefix='/api')
    app.register_blueprint(tools_bp, url_prefix='/api')
    app.register_blueprint(team_member_bp, url_prefix='/api')
    app.register_blueprint(use_case_bp, url_prefix='/api')
    app.register_blueprint(publication_bp, url_prefix='/api')
    app.register_blueprint(upload_bp, url_prefix='/api')
    app.register_blueprint(use_case_attributes_bp, url_prefix='/api')
    
    # Health check endpoint
    @app.route('/health')
    def health_check():
        return jsonify({'status': 'healthy'}), 200
    
    # Root endpoint
    @app.route('/')
    def index():
        return jsonify({
            'service': 'Admin Website Backend API',
            'version': '1.0.0',
            'endpoints': {
                'hardware': '/api/hardware',
                'tools': '/api/tools',
                'team_members': '/api/team-members',
                'use_cases': '/api/use-cases',
                'publications': '/api/publications'
            }
        }), 200
    
    return app

if __name__ == '__main__':
    app = create_app()
