/* eslint-disable @typescript-eslint/no-explicit-any */
import config from '../config';

// Generic types
export interface ApiResponse {
    message?: string;
    error?: string;
}

// Generic API helper functions
async function fetchApi<T>(endpoint: string, options?: RequestInit): Promise<T> {
    const response = await fetch(`${config.apiBaseUrl}${endpoint}`, {
        ...options,
        headers: {
            'Content-Type': 'application/json',
            ...options?.headers,
        },
    });

    if (!response.ok) {
        const error = await response.json().catch(() => ({ message: 'Request failed' }));
        throw new Error(error.error || error.message || 'Request failed');
    }

    return response.json();
}

// Upload API
export const uploadApi = {
    upload: async (file: File) => {
        const formData = new FormData();
        formData.append('file', file);
        // folder param removed as backend flattens structure

        const response = await fetch(`${config.apiBaseUrl}/upload`, {
            method: 'POST',
            body: formData,
            // Don't set Content-Type header manually, let browser set it with boundary
        });

        if (!response.ok) {
            const error = await response.json().catch(() => ({ message: 'Upload failed' }));
            throw new Error(error.error || error.message || 'Upload failed');
        }

        return response.json();
    }
};

// Hardware API
export const hardwareApi = {
    getAll: <T = any[]>() => fetchApi<T>('/hardware'),
    getById: <T = any>(id: string) => fetchApi<T>(`/hardware/${id}`),
    create: <T = any>(data: any) => fetchApi<T>('/hardware', {
        method: 'POST',
        body: JSON.stringify(data),
    }),
    update: <T = any>(id: string, data: any) => fetchApi<T>(`/hardware/${id}`, {
        method: 'PUT',
        body: JSON.stringify(data),
    }),
    delete: (id: string) => fetchApi<ApiResponse>(`/hardware/${id}`, {
        method: 'DELETE',
    }),
};

// Tools API
export const toolsApi = {
    getAll: <T = any[]>() => fetchApi<T>('/tools'),
    getById: <T = any>(id: string) => fetchApi<T>(`/tools/${id}`),
    create: <T = any>(data: any) => fetchApi<T>('/tools', {
        method: 'POST',
        body: JSON.stringify(data),
    }),
    update: <T = any>(id: string, data: any) => fetchApi<T>(`/tools/${id}`, {
        method: 'PUT',
        body: JSON.stringify(data),
    }),
    delete: (id: string) => fetchApi<ApiResponse>(`/tools/${id}`, {
        method: 'DELETE',
    }),
};

// Team Members API
export const teamMembersApi = {
    getAll: <T = any[]>() => fetchApi<T>('/team-members'),
    getById: <T = any>(id: string) => fetchApi<T>(`/team-members/${id}`),
    create: <T = any>(data: any) => fetchApi<T>('/team-members', {
        method: 'POST',
        body: JSON.stringify(data),
    }),
    update: <T = any>(id: string, data: any) => fetchApi<T>(`/team-members/${id}`, {
        method: 'PUT',
        body: JSON.stringify(data),
    }),
    delete: (id: string) => fetchApi<ApiResponse>(`/team-members/${id}`, {
        method: 'DELETE',
    }),
};

// Use Cases API
// Use Cases API
export const useCasesApi = {
    getAll: <T = any[]>() => fetchApi<T>('/use-cases'),
    getById: <T = any>(id: string) => fetchApi<T>(`/use-cases/${id}`),
    create: <T = any>(data: any) => fetchApi<T>('/use-cases', {
        method: 'POST',
        body: JSON.stringify(data),
    }),
    update: <T = any>(id: string, data: any) => fetchApi<T>(`/use-cases/${id}`, {
        method: 'PUT',
        body: JSON.stringify(data),
    }),
    delete: (id: string) => fetchApi<ApiResponse>(`/use-cases/${id}`, {
        method: 'DELETE',
    }),
};

// Use Case Attributes API
export const useCaseAttributesApi = {
    getAll: <T = any[]>(type: 'keywords' | 'frameworks' | 'models' | 'hardwares') => fetchApi<T>(`/use-case-attributes/${type}`),
    create: <T = any>(type: 'keywords' | 'frameworks' | 'models' | 'hardwares', data: any) => fetchApi<T>(`/use-case-attributes/${type}`, {
        method: 'POST',
        body: JSON.stringify(data),
    }),
    update: <T = any>(type: 'keywords' | 'frameworks' | 'models' | 'hardwares', id: string, data: any) => fetchApi<T>(`/use-case-attributes/${type}/${id}`, {
        method: 'PUT',
        body: JSON.stringify(data),
    }),
    delete: (type: 'keywords' | 'frameworks' | 'models' | 'hardwares', id: string) => fetchApi<ApiResponse>(`/use-case-attributes/${type}/${id}`, {
        method: 'DELETE',
    }),
};

// Publications API
export const publicationsApi = {
    getAll: <T = any[]>() => fetchApi<T>('/publications'),
    getById: <T = any>(id: string) => fetchApi<T>(`/publications/${id}`),
    create: <T = any>(data: any) => fetchApi<T>('/publications', {
        method: 'POST',
        body: JSON.stringify(data),
    }),
    update: <T = any>(id: string, data: any) => fetchApi<T>(`/publications/${id}`, {
        method: 'PUT',
        body: JSON.stringify(data),
    }),
    delete: (id: string) => fetchApi<ApiResponse>(`/publications/${id}`, {
        method: 'DELETE',
    }),
};
