/* eslint-disable @typescript-eslint/no-explicit-any */
import React, { useState, useEffect, useCallback } from 'react';
import { useNavigate } from 'react-router-dom';
import DataTable, { type Column } from '../../components/DataTable';
import FormModal, { type FormField } from '../../components/FormModal';
import { useCaseAttributesApi } from '../../services/api';
import styles from './UseCasesFilters.module.css';

type AttributeType = 'keywords' | 'frameworks' | 'models' | 'hardwares';

interface AttributeItem {
    id: string;
    name: string;
}

const TABS: { id: AttributeType; label: string }[] = [
    { id: 'keywords', label: 'Keywords' },
    { id: 'frameworks', label: 'Frameworks' },
    { id: 'models', label: 'Models' },
    { id: 'hardwares', label: 'Hardwares' },
];

const columns: Column<AttributeItem>[] = [
    { key: 'name', title: 'Name' },
];

const formFields: FormField[] = [
    { key: 'name', label: 'Name', required: true },
];

const UseCasesFilters: React.FC = () => {
    const navigate = useNavigate();
    const [activeTab, setActiveTab] = useState<AttributeType>('keywords');
    const [data, setData] = useState<AttributeItem[]>([]);
    const [loading, setLoading] = useState(false);
    const [modalOpen, setModalOpen] = useState(false);
    const [editingItem, setEditingItem] = useState<AttributeItem | null>(null);

    const fetchData = useCallback(async () => {
        setLoading(true);
        try {
            const items = await useCaseAttributesApi.getAll<AttributeItem[]>(activeTab);
            setData(items);
        } catch (error) {
            console.error(`Failed to fetch ${activeTab}:`, error);
        } finally {
            setLoading(false);
        }
    }, [activeTab]);

    useEffect(() => {
        fetchData();
    }, [fetchData]);

    const handleAdd = () => {
        setEditingItem(null);
        setModalOpen(true);
    };

    const handleEdit = (item: AttributeItem) => {
        setEditingItem(item);
        setModalOpen(true);
    };

    const handleDelete = async (item: AttributeItem) => {
        if (!confirm(`Are you sure you want to delete "${item.name}"?`)) return;
        try {
            await useCaseAttributesApi.delete(activeTab, item.id);
            fetchData();
        } catch (error) {
            console.error('Failed to delete:', error);
            alert('Failed to delete item');
        }
    };

    const handleSubmit = async (formData: Record<string, any>) => {
        if (editingItem) {
            await useCaseAttributesApi.update(activeTab, editingItem.id, formData);
        } else {
            await useCaseAttributesApi.create(activeTab, formData);
        }
        fetchData();
    };

    return (
        <div className={styles.container}>
            <div className={styles.header}>
                <button
                    className={styles.backBtn}
                    onClick={() => navigate('/use-cases')}
                >
                    &larr; Back to Use Cases
                </button>
            </div>
            <div className={styles.tabs}>
                {TABS.map(tab => (
                    <button
                        key={tab.id}
                        className={`${styles.tab} ${activeTab === tab.id ? styles.active : ''}`}
                        onClick={() => setActiveTab(tab.id)}
                    >
                        {tab.label}
                    </button>
                ))}
            </div>

            <div className={styles.content}>
                <DataTable
                    title={`${TABS.find(t => t.id === activeTab)?.label} Management`}
                    columns={columns}
                    data={data}
                    loading={loading}
                    onAdd={handleAdd}
                    onEdit={handleEdit}
                    onDelete={handleDelete}
                />
            </div>

            <FormModal
                isOpen={modalOpen}
                title={editingItem ? `Edit ${activeTab.slice(0, -1)}` : `Add ${activeTab.slice(0, -1)}`}
                fields={formFields}
                initialData={editingItem || {}}
                onClose={() => setModalOpen(false)}
                onSubmit={handleSubmit}
            />
        </div>
    );
};

export default UseCasesFilters;
