/* eslint-disable @typescript-eslint/no-explicit-any */
import React, { useState, useEffect, useCallback } from 'react';
import { useNavigate } from 'react-router-dom';
import DataTable, { type Column } from '../../components/DataTable';
import UseCaseFormModal from './UseCaseFormModal';
import { useCasesApi } from '../../services/api';
import styles from './UseCases.module.css';

interface UseCaseItem {
    id: string;
    title: string;
    description: string;
    image_src: string | null;
    tags?: string[]; // simplified
}

const columns: Column<UseCaseItem>[] = [
    {
        key: 'title',
        title: 'Title',
        width: '250px',
        render: (val) => <span className={styles.title}>{val}</span>
    },
    {
        key: 'description',
        title: 'Description',
        render: (val) => (
            <span title={val} className={styles.description}>
                {val?.length > 100 ? val.substring(0, 100) + '...' : val}
            </span>
        )
    },
    {
        key: 'image_src',
        title: 'Image',
        width: '100px',
        render: (val) => {
            if (!val) return '-';
            return <img src={val} alt="Thumbnail" className={styles.thumbnail} />;
        }
    }
];

const UseCases: React.FC = () => {
    const navigate = useNavigate();
    const [data, setData] = useState<UseCaseItem[]>([]);
    const [loading, setLoading] = useState(true);
    const [modalOpen, setModalOpen] = useState(false);
    const [editingItem, setEditingItem] = useState<UseCaseItem | null>(null);

    const fetchData = useCallback(async () => {
        setLoading(true);
        try {
            const items = await useCasesApi.getAll();
            setData(items);
        } catch (error) {
            console.error('Failed to fetch use cases:', error);
        } finally {
            setLoading(false);
        }
    }, []);

    useEffect(() => {
        fetchData();
    }, [fetchData]);

    const handleAdd = () => {
        setEditingItem(null);
        setModalOpen(true);
    };

    const handleEdit = async (item: UseCaseItem) => {
        // Fetch full details
        try {
            const fullItem = await useCasesApi.getById(item.id);
            setEditingItem(fullItem);
            setModalOpen(true);
        } catch (error) {
            console.error('Failed to fetch details:', error);
            alert('Failed to load details');
        }
    };

    const handleDelete = async (item: UseCaseItem) => {
        if (!confirm(`Are you sure you want to delete "${item.title}"?`)) return;
        try {
            await useCasesApi.delete(item.id);
            fetchData();
        } catch (error) {
            console.error('Failed to delete:', error);
            alert('Failed to delete item');
        }
    };

    const handleSubmit = async (formData: any) => {
        if (editingItem && editingItem.id) {
            await useCasesApi.update(editingItem.id, formData);
        } else {
            await useCasesApi.create(formData);
        }
        fetchData();
    };

    return (
        <div className={styles.page}>
            <div className={styles.header}>
                <button className={styles.filterBtn} onClick={() => navigate('/use-cases/filters')}>
                    Manage Filters (Keywords, etc.)
                </button>
            </div>

            <DataTable
                title="Use Cases Management"
                columns={columns}
                data={data}
                loading={loading}
                onAdd={handleAdd}
                onEdit={handleEdit}
                onDelete={handleDelete}
            />

            <UseCaseFormModal
                isOpen={modalOpen}
                initialData={editingItem}
                onClose={() => setModalOpen(false)}
                onSubmit={handleSubmit}
            />
        </div>
    );
};

export default UseCases;
