/* eslint-disable @typescript-eslint/no-explicit-any */
import React, { useState, useEffect } from 'react';
import FileUpload from '../../components/FileUpload/FileUpload';
import { useCaseAttributesApi, uploadApi } from '../../services/api';
import styles from './UseCaseFormModal.module.css';

interface UseCaseFormModalProps {
    isOpen: boolean;
    initialData?: any;
    onClose: () => void;
    onSubmit: (data: any) => Promise<void>;
}

const UseCaseFormModal: React.FC<UseCaseFormModalProps> = ({ isOpen, initialData, onClose, onSubmit }) => {
    const [formData, setFormData] = useState<any>({
        title: '',
        description: '',
        image_src: null,
        detail: {
            title: '',
            caption: '',
            main_image: null,
            publications: [],
            dataset: {},
            source_code: {}
        },
        paragraphs: [],
        keywords: [],
        frameworks: [],
        models: [],
        hardwares: []
    });

    const [attributes, setAttributes] = useState<{
        keywords: any[];
        frameworks: any[];
        models: any[];
        hardwares: any[];
    }>({
        keywords: [],
        frameworks: [],
        models: [],
        hardwares: []
    });

    const [loading, setLoading] = useState(false);
    const [error, setError] = useState<string | null>(null);
    const [activeTab, setActiveTab] = useState('general');

    useEffect(() => {
        if (isOpen) {
            // Load attributes
            const loadAttrs = async () => {
                try {
                    const [k, f, m, h] = await Promise.all([
                        useCaseAttributesApi.getAll('keywords'),
                        useCaseAttributesApi.getAll('frameworks'),
                        useCaseAttributesApi.getAll('models'),
                        useCaseAttributesApi.getAll('hardwares')
                    ]);
                    setAttributes({
                        keywords: k as any[],
                        frameworks: f as any[],
                        models: m as any[],
                        hardwares: h as any[]
                    });
                } catch (e) {
                    console.error("Failed to load attributes", e);
                }
            };
            loadAttrs();

            // Set initial data
            if (initialData) {
                // deep copy to avoid mutations
                setFormData(JSON.parse(JSON.stringify(initialData)));
            } else {
                setFormData({
                    title: '',
                    description: '',
                    image_src: null,
                    detail: {
                        title: '',
                        caption: '',
                        main_image: null,
                        publications: [],
                        dataset: {},
                        source_code: {}
                    },
                    paragraphs: [],
                    keywords: [],
                    frameworks: [],
                    models: [],
                    hardwares: []
                });
            }
            setActiveTab('general');
            setError(null);
        }
    }, [isOpen, initialData]);

    const handleInputChange = (field: string, value: any, parent: string | null = null) => {
        if (parent === 'detail') {
            setFormData((prev: any) => ({
                ...prev,
                detail: { ...prev.detail, [field]: value }
            }));
        } else {
            setFormData((prev: any) => ({ ...prev, [field]: value }));
        }
    };

    const handleTagToggle = (type: string, id: string) => {
        setFormData((prev: any) => {
            const list = prev[type] || [];
            if (list.includes(id)) {
                return { ...prev, [type]: list.filter((x: string) => x !== id) };
            }
            return { ...prev, [type]: [...list, id] };
        });
    };

    const handleParagraphChange = (index: number, field: string, value: any) => {
        const newParas = [...(formData.paragraphs || [])];
        if (!newParas[index]) return;

        // Special handle for content (text[])
        if (field === 'content') {
            // Assuming textarea input for now, split by newline? Or just one string?
            // Backend expects json list. Let's keep it simple: array of strings.
            // If input is string, wrap in array or split?
            // User input from textarea is string.
            newParas[index][field] = [value];
        } else {
            newParas[index][field] = value;
        }
        setFormData((prev: any) => ({ ...prev, paragraphs: newParas }));
    };

    // Helper to get textarea value from content array
    const getParagraphContent = (para: any) => {
        if (Array.isArray(para.content)) return para.content.join('\n');
        return para.content || '';
    };

    const addParagraph = () => {
        setFormData((prev: any) => ({
            ...prev,
            paragraphs: [...(prev.paragraphs || []), { title: '', content: [], image: null, order_index: prev.paragraphs?.length || 0 }]
        }));
    };

    const removeParagraph = (index: number) => {
        setFormData((prev: any) => ({
            ...prev,
            paragraphs: (prev.paragraphs || []).filter((_: any, i: number) => i !== index)
        }));
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        setLoading(true);
        setError(null);

        try {
            const dataToSubmit = { ...formData }; // shallow copy

            // Upload UseCase Image
            if (dataToSubmit.image_src instanceof File) {
                const res = await uploadApi.upload(dataToSubmit.image_src);
                dataToSubmit.image_src = res.url;
            }

            // Upload Detail Image
            if (dataToSubmit.detail?.main_image instanceof File) {
                const res = await uploadApi.upload(dataToSubmit.detail.main_image);
                dataToSubmit.detail.main_image = res.url;
            }

            // Upload Paragraph Images
            if (dataToSubmit.paragraphs) {
                for (const p of dataToSubmit.paragraphs) {
                    if (p.image instanceof File) {
                        const res = await uploadApi.upload(p.image);
                        p.image = res.url;
                    }
                }
            }

            // Clean up: ensure content is array
            dataToSubmit.paragraphs = dataToSubmit.paragraphs.map((p: any) => ({
                ...p,
                content: Array.isArray(p.content) ? p.content : [p.content]
            }));

            await onSubmit(dataToSubmit);
            onClose();
        } catch (err: any) {
            console.error(err);
            setError(err.message || 'Failed to save');
        } finally {
            setLoading(false);
        }
    };

    if (!isOpen) return null;

    return (
        <div className={styles.overlay}>
            <div className={styles.modal}>
                <div className={styles.header}>
                    <h2>{initialData ? 'Edit Use Case' : 'New Use Case'}</h2>
                    <button onClick={onClose} className={styles.closeBtn}>×</button>
                </div>

                <div className={styles.tabs}>
                    <button className={activeTab === 'general' ? styles.active : ''} onClick={() => setActiveTab('general')}>General</button>
                    <button className={activeTab === 'detail' ? styles.active : ''} onClick={() => setActiveTab('detail')}>Detail</button>
                    <button className={activeTab === 'tags' ? styles.active : ''} onClick={() => setActiveTab('tags')}>Tags</button>
                    <button className={activeTab === 'paragraphs' ? styles.active : ''} onClick={() => setActiveTab('paragraphs')}>Paragraphs ({formData.paragraphs?.length || 0})</button>
                </div>

                <div className={styles.body}>
                    {error && <div className={styles.error}>{error}</div>}

                    {activeTab === 'general' && (
                        <div className={styles.section}>
                            <div className={styles.field}>
                                <label>Title*</label>
                                <input
                                    value={formData.title}
                                    onChange={(e) => handleInputChange('title', e.target.value)}
                                    required
                                />
                            </div>
                            <div className={styles.field}>
                                <label>Description</label>
                                <textarea
                                    value={formData.description || ''}
                                    onChange={(e) => handleInputChange('description', e.target.value)}
                                    rows={4}
                                />
                            </div>
                            <div className={styles.field}>
                                <label>Image</label>
                                <FileUpload
                                    value={formData.image_src}
                                    onChange={(f) => handleInputChange('image_src', f)}
                                    accept="image/*"
                                />
                            </div>
                        </div>
                    )}

                    {activeTab === 'detail' && (
                        <div className={styles.section}>
                            <div className={styles.field}>
                                <label>Detail Title</label>
                                <textarea
                                    value={formData.detail.title || ''}
                                    onChange={(e) => handleInputChange('title', e.target.value, 'detail')}
                                />
                            </div>
                            <div className={styles.field}>
                                <label>Caption</label>
                                <textarea
                                    value={formData.detail.caption || ''}
                                    onChange={(e) => handleInputChange('caption', e.target.value, 'detail')}
                                />
                            </div>
                            <div className={styles.field}>
                                <label>Main Content Image</label>
                                <FileUpload
                                    value={formData.detail.main_image}
                                    onChange={(f) => handleInputChange('main_image', f, 'detail')}
                                    accept="image/*"
                                />
                            </div>
                        </div>
                    )}

                    {activeTab === 'tags' && (
                        <div className={styles.grid}>
                            {(['keywords', 'frameworks', 'models', 'hardwares'] as const).map(type => (
                                <div key={type} className={styles.tagGroup}>
                                    <h4>{type.charAt(0).toUpperCase() + type.slice(1)}</h4>
                                    <div className={styles.tagList}>
                                        {attributes[type].map(item => (
                                            <label key={item.id} className={styles.checkbox}>
                                                <input
                                                    type="checkbox"
                                                    checked={formData[type]?.includes(item.id)}
                                                    onChange={() => handleTagToggle(type, item.id)}
                                                />
                                                {item.name}
                                            </label>
                                        ))}
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}

                    {activeTab === 'paragraphs' && (
                        <div className={styles.section}>
                            {formData.paragraphs?.map((para: any, index: number) => (
                                <div key={index} className={styles.paragraphCard}>
                                    <div className={styles.paraHeader}>
                                        <span>Paragraph {index + 1}</span>
                                        <button type="button" onClick={() => removeParagraph(index)} className={styles.removeBtn}>Remove</button>
                                    </div>
                                    <div className={styles.field}>
                                        <label>Title</label>
                                        <input
                                            value={para.title || ''}
                                            onChange={(e) => handleParagraphChange(index, 'title', e.target.value)}
                                        />
                                    </div>
                                    <div className={styles.field}>
                                        <label>Content</label>
                                        <textarea
                                            value={getParagraphContent(para)}
                                            onChange={(e) => handleParagraphChange(index, 'content', e.target.value)}
                                            rows={4}
                                        />
                                    </div>
                                    <div className={styles.field}>
                                        <label>Image</label>
                                        <FileUpload
                                            value={para.image}
                                            onChange={(f) => handleParagraphChange(index, 'image', f)}
                                            accept="image/*"
                                        />
                                    </div>
                                </div>
                            ))}
                            <button type="button" onClick={addParagraph} className={styles.addBtn}>+ Add Paragraph</button>
                        </div>
                    )}
                </div>

                <div className={styles.footer}>
                    <button onClick={onClose} disabled={loading} className={styles.cancelBtn}>Cancel</button>
                    <button onClick={handleSubmit} disabled={loading} className={styles.submitBtn}>
                        {loading ? 'Saving...' : 'Save'}
                    </button>
                </div>
            </div>
        </div>
    );
};

export default UseCaseFormModal;
