/* eslint-disable @typescript-eslint/no-explicit-any */
import React, { useState, useEffect, useCallback } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { useCasesApi } from '../../services/api';
import FormModal, { type FormField } from '../../components/FormModal';
import styles from './UseCaseDetail.module.css';

interface Paragraph {
    id: string;
    use_case_id: string;
    subtitle: string;
    content: string;
    order_index: number;
    image_url: string | null;
    image_caption: string | null;
}

interface UseCaseWithParagraphs {
    id: string;
    title: string;
    description: string;
    created_at: string | null;
    updated_at: string | null;
    paragraphs: Paragraph[];
}

const paragraphFields: FormField[] = [
    { key: 'subtitle', label: 'Subtitle', required: true, placeholder: 'Enter paragraph subtitle' },
    { key: 'content', label: 'Content', type: 'textarea', required: true, placeholder: 'Enter paragraph content' },
    { key: 'order_index', label: 'Order', type: 'number', placeholder: 'Display order (1, 2, 3...)' },
    { key: 'image_url', label: 'Image URL', placeholder: 'Enter image URL (optional)' },
    { key: 'image_caption', label: 'Image Caption', placeholder: 'Enter image caption (optional)' },
];

const UseCaseDetail: React.FC = () => {
    const { id } = useParams<{ id: string }>();
    const navigate = useNavigate();
    const [data, setData] = useState<UseCaseWithParagraphs | null>(null);
    const [loading, setLoading] = useState(true);
    const [modalOpen, setModalOpen] = useState(false);
    const [editingParagraph, setEditingParagraph] = useState<Paragraph | null>(null);

    const fetchData = useCallback(async () => {
        if (!id) return;
        setLoading(true);
        try {
            const item = await useCasesApi.getById<UseCaseWithParagraphs>(id);
            setData(item);
        } catch (error) {
            console.error('Failed to fetch use case:', error);
        } finally {
            setLoading(false);
        }
    }, [id]);

    useEffect(() => {
        fetchData();
    }, [fetchData]);

    const handleBack = () => {
        navigate('/use-case');
    };

    const handleAddParagraph = () => {
        setEditingParagraph(null);
        setModalOpen(true);
    };

    const handleEditParagraph = (paragraph: Paragraph) => {
        setEditingParagraph(paragraph);
        setModalOpen(true);
    };

    const handleDeleteParagraph = async (paragraph: Paragraph) => {
        if (!id) return;
        if (!confirm(`Are you sure you want to delete "${paragraph.subtitle}"?`)) return;
        try {
            await useCasesApi.deleteParagraph(id, paragraph.id);
            fetchData();
        } catch (error) {
            console.error('Failed to delete paragraph:', error);
            alert('Failed to delete paragraph');
        }
    };

    const handleSubmitParagraph = async (formData: Record<string, any>) => {
        if (!id) return;
        if (editingParagraph) {
            await useCasesApi.updateParagraph(id, editingParagraph.id, formData);
        } else {
            await useCasesApi.createParagraph(id, formData);
        }
        fetchData();
    };

    if (loading) {
        return (
            <div className={styles.page}>
                <div className={styles.loading}>Loading...</div>
            </div>
        );
    }

    if (!data) {
        return (
            <div className={styles.page}>
                <div className={styles.error}>Use case not found</div>
                <button className={styles.backBtn} onClick={handleBack}>← Back to List</button>
            </div>
        );
    }

    return (
        <div className={styles.page}>
            <div className={styles.header}>
                <button className={styles.backBtn} onClick={handleBack}>
                    ← Back to List
                </button>
                <h1 className={styles.title}>{data.title}</h1>
            </div>

            <div className={styles.infoCard}>
                <h3>Description</h3>
                <p>{data.description || 'No description'}</p>
                <div className={styles.meta}>
                    <span>Created: {data.created_at ? new Date(data.created_at).toLocaleDateString() : '-'}</span>
                    <span>Updated: {data.updated_at ? new Date(data.updated_at).toLocaleDateString() : '-'}</span>
                </div>
            </div>

            <div className={styles.section}>
                <div className={styles.sectionHeader}>
                    <h2>Paragraphs ({data.paragraphs?.length || 0})</h2>
                    <button className={styles.addBtn} onClick={handleAddParagraph}>
                        + Add Paragraph
                    </button>
                </div>

                {data.paragraphs?.length === 0 ? (
                    <div className={styles.empty}>No paragraphs yet. Click "Add Paragraph" to create one.</div>
                ) : (
                    <div className={styles.paragraphList}>
                        {data.paragraphs?.sort((a, b) => a.order_index - b.order_index).map((paragraph) => (
                            <div key={paragraph.id} className={styles.paragraphCard}>
                                <div className={styles.paragraphHeader}>
                                    <span className={styles.queue}>#{paragraph.order_index}</span>
                                    <h4>{paragraph.subtitle}</h4>
                                    <div className={styles.actions}>
                                        <button className={styles.editBtn} onClick={() => handleEditParagraph(paragraph)}>
                                            Edit
                                        </button>
                                        <button className={styles.deleteBtn} onClick={() => handleDeleteParagraph(paragraph)}>
                                            Delete
                                        </button>
                                    </div>
                                </div>
                                <p className={styles.content}>{paragraph.content}</p>
                                {paragraph.image_url && (
                                    <div className={styles.imagePreview}>
                                        <img src={paragraph.image_url} alt={paragraph.image_caption || ''} />
                                        {paragraph.image_caption && <span>{paragraph.image_caption}</span>}
                                    </div>
                                )}
                            </div>
                        ))}
                    </div>
                )}
            </div>

            <FormModal
                isOpen={modalOpen}
                title={editingParagraph ? 'Edit Paragraph' : 'Add Paragraph'}
                fields={paragraphFields}
                initialData={editingParagraph || { order_index: (data.paragraphs?.length || 0) + 1 }}
                onClose={() => setModalOpen(false)}
                onSubmit={handleSubmitParagraph}
            />
        </div>
    );
};

export default UseCaseDetail;
