/* eslint-disable @typescript-eslint/no-explicit-any */
import React, { useState, useEffect, useCallback, useMemo } from 'react';
import DataTable, { type Column } from '../../components/DataTable';
import FormModal, { type FormField } from '../../components/FormModal';
import { toolsApi, uploadApi } from '../../services/api';
import styles from './Tools.module.css';

interface ToolItem {
    id: string;
    title: string;
    content: string;
    link: string | null;
    documentation: string | null;
    image: string | null;
    queue: number;
    created_time: string;
    updated_time: string;
}

const columns: Column<ToolItem>[] = [
    { key: 'queue', title: 'Queue', width: '70px' },
    { key: 'title', title: 'Title', width: '200px' },
    {
        key: 'content',
        title: 'Description',
        width: '250px',
        render: (value) => (
            <span className={styles.truncate} title={value}>{value?.length > 50 ? value.substring(0, 50) + '...' : value || '-'}</span>
        )
    },
    {
        key: 'image',
        title: 'Image',
        width: '100px',
        render: (value) => {
            if (!value) return '-';
            return (
                <div style={{ display: 'flex', alignItems: 'center', gap: '8px' }}>
                    <img
                        src={value}
                        alt=""
                        style={{ width: '40px', height: '40px', objectFit: 'cover', borderRadius: '4px' }}
                        onError={(e) => { (e.target as HTMLImageElement).src = 'https://via.placeholder.com/40'; }}
                    />
                </div>
            );
        }
    },
    {
        key: 'link',
        title: 'Link',
        width: '100px',
        render: (value) => value ? (
            <a href={value} target="_blank" rel="noopener noreferrer" className={styles.link}>
                Open
            </a>
        ) : '-'
    },
    {
        key: 'documentation',
        title: 'Docs',
        width: '100px',
        render: (value) => value ? (
            <a href={value} target="_blank" rel="noopener noreferrer" className={styles.link}>
                View
            </a>
        ) : '-'
    },
    { key: 'updated_time', title: 'Updated', width: '150px', render: (val) => new Date(val).toLocaleDateString() },
];

const formFields: FormField[] = [
    { key: 'queue', label: 'Queue Order', type: 'number', required: true },
    { key: 'title', label: 'Title', required: true, placeholder: 'Enter title' },
    { key: 'content', label: 'Description', type: 'textarea', placeholder: 'Enter description' },
    { key: 'image', label: 'Image', type: 'file' },
    { key: 'link', label: 'External Link', placeholder: 'https://...' },
    { key: 'documentation', label: 'Documentation URL', placeholder: 'https://...' },
];

const Tools: React.FC = () => {
    const [data, setData] = useState<ToolItem[]>([]);
    const [loading, setLoading] = useState(true);
    const [modalOpen, setModalOpen] = useState(false);
    const [editingItem, setEditingItem] = useState<ToolItem | null>(null);

    const fetchData = useCallback(async () => {
        setLoading(true);
        try {
            const items = await toolsApi.getAll();
            setData(items);
        } catch (error) {
            console.error('Failed to fetch tools:', error);
        } finally {
            setLoading(false);
        }
    }, []);

    useEffect(() => {
        fetchData();
    }, [fetchData]);

    const handleAdd = () => {
        setEditingItem(null);
        setModalOpen(true);
    };

    const handleEdit = (item: ToolItem) => {
        setEditingItem(item);
        setModalOpen(true);
    };

    const handleDelete = async (item: ToolItem) => {
        if (!confirm(`Are you sure you want to delete "${item.title}"?`)) return;
        try {
            await toolsApi.delete(item.id);
            fetchData();
        } catch (error) {
            console.error('Failed to delete:', error);
            alert('Failed to delete item');
        }
    };

    const handleSubmit = async (formData: Record<string, any>) => {
        // Handle file upload
        if (formData.image instanceof File) {
            const uploadRes = await uploadApi.upload(formData.image);
            formData.image = uploadRes.url;
        }

        if (editingItem) {
            await toolsApi.update(editingItem.id, formData);
        } else {
            await toolsApi.create(formData);
        }
        fetchData();
    };

    const currentFields = useMemo(() => {
        return formFields.map(field => {
            if (field.key === 'image') {
                return {
                    ...field,
                    label: editingItem ? 'Modify Image' : 'Upload Image'
                };
            }
            return field;
        });
    }, [editingItem]);

    return (
        <div className={styles.page}>
            <DataTable
                title="Tools Management"
                columns={columns}
                data={data}
                loading={loading}
                onAdd={handleAdd}
                onEdit={handleEdit}
                onDelete={handleDelete}
            />

            <FormModal
                isOpen={modalOpen}
                title={editingItem ? 'Edit Tool' : 'Add Tool'}
                fields={currentFields}
                initialData={editingItem || {}}
                onClose={() => setModalOpen(false)}
                onSubmit={handleSubmit}
            />
        </div>
    );
};

export default Tools;
