/* eslint-disable @typescript-eslint/no-explicit-any */
import React, { useState, useEffect, useCallback, useMemo } from 'react';
import DataTable, { type Column } from '../../components/DataTable';
import FormModal, { type FormField } from '../../components/FormModal';
import { teamMembersApi, uploadApi } from '../../services/api';
import styles from './TeamMember.module.css';

const TeamMember: React.FC = () => {
    const [data, setData] = useState<TeamMemberItem[]>([]);
    const [loading, setLoading] = useState(true);
    const [modalOpen, setModalOpen] = useState(false);
    const [editingItem, setEditingItem] = useState<TeamMemberItem | null>(null);

    const fetchData = useCallback(async () => {
        setLoading(true);
        try {
            const items = await teamMembersApi.getAll();
            setData(items);
        } catch (error) {
            console.error('Failed to fetch team members:', error);
        } finally {
            setLoading(false);
        }
    }, []);

    useEffect(() => {
        fetchData();
    }, [fetchData]);

    const handleAdd = () => {
        setEditingItem(null);
        setModalOpen(true);
    };

    const handleEdit = (item: TeamMemberItem) => {
        setEditingItem(item);
        setModalOpen(true);
    };

    const handleDelete = async (item: TeamMemberItem) => {
        if (!confirm(`Are you sure you want to delete "${item.name} ${item.surname}"?`)) return;
        try {
            await teamMembersApi.delete(item.id);
            fetchData();
        } catch (error) {
            console.error('Failed to delete:', error);
            alert('Failed to delete item');
        }
    };

    const handleSubmit = async (formData: Record<string, any>) => {
        // Handle file upload
        if (formData.image_url instanceof File) {
            const uploadRes = await uploadApi.upload(formData.image_url);
            formData.image_url = uploadRes.url;
        }

        if (editingItem) {
            await teamMembersApi.update(editingItem.id, formData);
        } else {
            await teamMembersApi.create(formData);
        }
        fetchData();
    };

    const currentFields = useMemo(() => {
        return formFields.map(field => {
            if (field.key === 'image_url') {
                return {
                    ...field,
                    label: editingItem ? 'Modify Image' : 'Upload Image'
                };
            }
            return field;
        });
    }, [editingItem]);

    return (
        <div className={styles.page}>
            <DataTable
                title="Team Members Management"
                columns={columns}
                data={data}
                loading={loading}
                onAdd={handleAdd}
                onEdit={handleEdit}
                onDelete={handleDelete}
            />

            <FormModal
                isOpen={modalOpen}
                title={editingItem ? 'Edit Team Member' : 'Add Team Member'}
                fields={currentFields}
                initialData={editingItem || {}}
                onClose={() => setModalOpen(false)}
                onSubmit={handleSubmit}
            />
        </div>
    );
};

interface TeamMemberItem {
    id: string;
    name: string;
    surname: string;
    role: string | null;
    image_url: string | null;
    linkedin_profile: string | null;
    googlescholar_profile: string | null;
    iris_profile: string | null;
    orcid_profile: string | null;
    polito_profile: string | null;
    researchgate_profile: string | null;
    description: string | null;
    queue: number;
    active: boolean;
    created_at: string;
    updated_at: string;
}

const columns: Column<TeamMemberItem>[] = [
    { key: 'queue', title: 'Queue', width: '70px' },
    {
        key: 'name',
        title: 'Name',
        width: '150px',
        render: (_, record) => `${record.name} ${record.surname}`
    },
    { key: 'role', title: 'Role', width: '150px' },
    {
        key: 'active',
        title: 'Status',
        width: '80px',
        render: (value) => (
            <span className={value ? styles.active : styles.inactive}>
                {value ? 'Active' : 'Inactive'}
            </span>
        )
    },
    {
        key: 'image_url',
        title: 'Image',
        width: '200px',
        render: (value) => {
            if (!value) return '-';
            return (
                <div style={{ display: 'flex', alignItems: 'center', gap: '8px' }}>
                    <img
                        src={value}
                        alt="Profile"
                        style={{ width: '30px', height: '30px', borderRadius: '50%', objectFit: 'cover' }}
                        onError={(e) => { (e.target as HTMLImageElement).src = 'https://via.placeholder.com/30'; }}
                    />
                </div>
            );
        }
    },
    {
        key: 'linkedin_profile',
        title: 'LinkedIn',
        width: '100px',
        render: (value) => value ? <a href={value} target="_blank" rel="noopener noreferrer" className={styles.link}>Link</a> : '-'
    },
    {
        key: 'polito_profile',
        title: 'Polito',
        width: '100px',
        render: (value) => value ? <a href={value} target="_blank" rel="noopener noreferrer" className={styles.link}>Link</a> : '-'
    },
    {
        key: 'iris_profile',
        title: 'IRIS',
        width: '100px',
        render: (value) => value ? <a href={value} target="_blank" rel="noopener noreferrer" className={styles.link}>Link</a> : '-'
    },
    {
        key: 'googlescholar_profile',
        title: 'Scholar',
        width: '100px',
        render: (value) => value ? <a href={value} target="_blank" rel="noopener noreferrer" className={styles.link}>Link</a> : '-'
    },
    {
        key: 'orcid_profile',
        title: 'ORCID',
        width: '100px',
        render: (value) => value ? <a href={value} target="_blank" rel="noopener noreferrer" className={styles.link}>Link</a> : '-'
    },
    {
        key: 'researchgate_profile',
        title: 'R.Gate',
        width: '100px',
        render: (value) => value ? <a href={value} target="_blank" rel="noopener noreferrer" className={styles.link}>Link</a> : '-'
    },
    { key: 'description', title: 'Description', width: '200px', render: (val) => val ? (val.length > 50 ? val.substring(0, 50) + '...' : val) : '-' },
    { key: 'created_at', title: 'Created', width: '150px', render: (val) => new Date(val).toLocaleDateString() },
    { key: 'updated_at', title: 'Updated', width: '150px', render: (val) => new Date(val).toLocaleDateString() },
];

const formFields: FormField[] = [
    { key: 'queue', label: 'Queue Order', type: 'number', required: true },
    { key: 'name', label: 'First Name', required: true, placeholder: 'Enter first name' },
    { key: 'surname', label: 'Last Name', required: true, placeholder: 'Enter last name' },
    { key: 'role', label: 'Role', placeholder: 'Enter role (e.g., Professor)' },
    { key: 'description', label: 'Description', type: 'textarea', placeholder: 'Enter description' },
    { key: 'image_url', label: 'Image', type: 'file' },
    { key: 'linkedin_profile', label: 'LinkedIn Profile', placeholder: 'https://...' },
    { key: 'polito_profile', label: 'Polito Profile', placeholder: 'https://...' },
    { key: 'iris_profile', label: 'IRIS Profile', placeholder: 'https://...' },
    { key: 'googlescholar_profile', label: 'Google Scholar', placeholder: 'https://...' },
    { key: 'orcid_profile', label: 'ORCID Profile', placeholder: 'https://...' },
    { key: 'researchgate_profile', label: 'ResearchGate', placeholder: 'https://...' },
    { key: 'active', label: 'Active', type: 'checkbox' },
];

export default TeamMember;
