/* eslint-disable @typescript-eslint/no-explicit-any */
import React, { useState, useEffect, useCallback, useMemo } from 'react';
import DataTable, { type Column } from '../../components/DataTable';
import FormModal, { type FormField } from '../../components/FormModal';
import { publicationsApi, uploadApi } from '../../services/api';
import styles from './Publication.module.css';

interface PublicationItem {
    id: string;
    handle: string;
    title: string;
    type: string | null;
    publication_year: string | null;
    venue: string | null;
    state: string | null;
    doi: string | null;
    url: string | null;
    reviewed: boolean;
    citation_raw: string | null;
    abstract: string | null;
    keyword: string | null;
    audio_url: string | null;
    video_url: string | null;
    mental_map_url: string | null;
    report_url: string | null;
    flashcard_url: string | null;
    quiz_url: string | null;
    infografica_url: string | null;
    presentation_url: string | null;
    datatable_url: string | null;
    full_paper_url: string | null;
    created_at: string;
}

const columns: Column<PublicationItem>[] = [
    {
        key: 'title',
        title: 'Title',
        width: '350px',
        render: (value) => (
            <span style={{ whiteSpace: 'normal' }}>{value || '-'}</span>
        )
    },
    // {
    //     key: 'keyword',
    //     title: 'Keyword',
    //     width: '120px',
    //     render: (value) => (
    //         <span className={styles.truncate} title={value}>{value?.length > 15 ? value.substring(0, 15) + '...' : value || '-'}</span>
    //     )
    // },
    { key: 'publication_year', title: 'Year', width: '60px' },
    {
        key: 'state',
        title: 'State',
        width: '80px',
        render: (value) => (
            <span className={styles[`state_${value}`] || styles.state_default}>
                {value || '-'}
            </span>
        )
    },
    {
        key: 'reviewed',
        title: 'Review',
        width: '40px',
        render: (value) => (
            <span className={value ? styles.reviewed : styles.pending}>
                {value ? 'Y' : 'N'}
            </span>
        )
    },
    {
        key: 'full_paper_url',
        title: 'Full Text',
        width: '100px',
        render: (value) => value ? (
            <a href={value} target="_blank" rel="noopener noreferrer" className={styles.link}>
                uploaded
            </a>
        ) : (
            <span style={{ color: '#999', fontSize: '14px' }}>pending</span>
        )
    },
    // 9 Resources
    {
        key: 'audio_url',
        title: 'Audio',
        width: '45px',
        render: (v) => v ? <a href={v} target="_blank" rel="noopener noreferrer" className={styles.link}>link</a> : '-'
    },
    {
        key: 'video_url',
        title: 'Video',
        width: '45px',
        render: (v) => v ? <a href={v} target="_blank" rel="noopener noreferrer" className={styles.link}>link</a> : '-'
    },
    {
        key: 'mental_map_url',
        title: 'Metal Map',
        width: '45px',
        render: (v) => v ? <a href={v} target="_blank" rel="noopener noreferrer" className={styles.link}>link</a> : '-'
    },
    {
        key: 'report_url',
        title: 'Report',
        width: '45px',
        render: (v) => v ? <a href={v} target="_blank" rel="noopener noreferrer" className={styles.link}>link</a> : '-'
    },
    {
        key: 'flashcard_url',
        title: 'Flashcard',
        width: '45px',
        render: (v) => v ? <a href={v} target="_blank" rel="noopener noreferrer" className={styles.link}>link</a> : '-'
    },
    {
        key: 'quiz_url',
        title: 'Quiz',
        width: '45px',
        render: (v) => v ? <a href={v} target="_blank" rel="noopener noreferrer" className={styles.link}>link</a> : '-'
    },
    {
        key: 'infografica_url',
        title: 'Infographic',
        width: '45px',
        render: (v) => v ? <a href={v} target="_blank" rel="noopener noreferrer" className={styles.link}>link</a> : '-'
    },
    {
        key: 'presentation_url',
        title: 'Presentation',
        width: '45px',
        render: (v) => v ? <a href={v} target="_blank" rel="noopener noreferrer" className={styles.link}>link</a> : '-'
    },
    {
        key: 'datatable_url',
        title: 'Datatable',
        width: '45px',
        render: (v) => v ? <a href={v} target="_blank" rel="noopener noreferrer" className={styles.link}>link</a> : '-'
    },

];

const formFields: FormField[] = [
    { key: 'handle', label: 'Handle', required: true, placeholder: 'Enter handle' },
    { key: 'title', label: 'Title', required: true },
    { key: 'publication_year', label: 'Year', placeholder: 'e.g., 2024' },
    { key: 'venue', label: 'Venue' },
    { key: 'type', label: 'Type' },
    {
        key: 'state',
        label: 'State',
        type: 'select',
        options: [
            { value: 'archived', label: 'Archived' },
            { value: 'in_progress', label: 'In Progress' },
            { value: 'published', label: 'Published' },
        ]
    },
    { key: 'doi', label: 'DOI' },
    { key: 'url', label: 'URL' },
    // { key: 'keyword', label: 'Keyword' },
    { key: 'abstract', label: 'Abstract', type: 'textarea' },
    { key: 'reviewed', label: 'Reviewed', type: 'checkbox' },
    // File uploads
    { key: 'full_paper_url', label: 'Full Text', type: 'file', accept: '.pdf,.doc,.docx,application/pdf,application/msword,application/vnd.openxmlformats-officedocument.wordprocessingml.document' },
    { key: 'audio_url', label: 'Audio', type: 'file' },
    { key: 'video_url', label: 'Video', type: 'file' },
    { key: 'mental_map_url', label: 'Mental Map', type: 'file' },
    { key: 'report_url', label: 'Report', type: 'file' },
    { key: 'flashcard_url', label: 'Flashcard', type: 'file' },
    { key: 'quiz_url', label: 'Quiz URL', placeholder: 'Enter Quiz URL' },
    { key: 'infografica_url', label: 'Infographic', type: 'file' },
    { key: 'presentation_url', label: 'Presentation', type: 'file' },
    { key: 'datatable_url', label: 'Datatable', type: 'file' },
];

const Publication: React.FC = () => {
    const [data, setData] = useState<PublicationItem[]>([]);
    const [loading, setLoading] = useState(true);
    const [modalOpen, setModalOpen] = useState(false);
    const [editingItem, setEditingItem] = useState<PublicationItem | null>(null);

    const fetchData = useCallback(async () => {
        setLoading(true);
        try {
            const items = await publicationsApi.getAll();
            setData(items);
        } catch (error) {
            console.error('Failed to fetch publications:', error);
        } finally {
            setLoading(false);
        }
    }, []);

    useEffect(() => {
        fetchData();
    }, [fetchData]);

    const handleAdd = () => {
        setEditingItem(null);
        setModalOpen(true);
    };

    const handleEdit = (item: PublicationItem) => {
        setEditingItem(item);
        setModalOpen(true);
    };

    const handleDelete = async (item: PublicationItem) => {
        if (!confirm(`Are you sure you want to delete "${item.title}"?`)) return;
        try {
            await publicationsApi.delete(item.id);
            fetchData();
        } catch (error) {
            console.error('Failed to delete:', error);
            alert('Failed to delete item');
        }
    };

    const handleSubmit = async (formData: Record<string, any>) => {
        // Define file fields
        const fileFields = [
            'full_paper_url',
            'audio_url',
            'video_url',
            'mental_map_url',
            'report_url',
            'flashcard_url',
            // quiz_url is text, so skipped here
            'infografica_url',
            'presentation_url',
            'datatable_url',
        ];

        // Process uploads sequentially
        for (const key of fileFields) {
            if (formData[key] instanceof File) {
                try {
                    // Start of Selection
                    const uploadRes = await uploadApi.upload(formData[key]); // Folder 'publications' is ignored by backend now but required by TS signature if not optional
                    formData[key] = uploadRes.url;
                } catch (e) {
                    console.error(`Failed to upload ${key}`, e);
                    alert(`Failed to upload ${key}`);
                    return; // Stop on upload failure
                }
            }
        }

        if (editingItem) {
            await publicationsApi.update(editingItem.id, formData);
        } else {
            await publicationsApi.create(formData);
        }
        fetchData();
    };

    const currentFields = useMemo(() => {
        return formFields.map(field => {
            if (field.type === 'file') {
                return {
                    ...field,
                    label: editingItem ? `Modify ${field.label}` : `Upload ${field.label}`
                };
            }
            return field;
        });
    }, [editingItem]);

    return (
        <div className={styles.page}>
            <DataTable
                title="Publications Management"
                columns={columns}
                data={data}
                loading={loading}
                onAdd={handleAdd}
                onEdit={handleEdit}
                onDelete={handleDelete}
            />

            <FormModal
                isOpen={modalOpen}
                title={editingItem ? 'Edit Publication' : 'Add Publication'}
                fields={currentFields}
                initialData={editingItem || {}}
                onClose={() => setModalOpen(false)}
                onSubmit={handleSubmit}
            />
        </div>
    );
};

export default Publication;
