import React, { useState, useEffect } from 'react';
import FileUpload from '../FileUpload/FileUpload';
import styles from './FormModal.module.css';

export interface FormField {
    key: string;
    label: string;
    type?: 'text' | 'textarea' | 'number' | 'checkbox' | 'select' | 'file';
    required?: boolean;
    placeholder?: string;
    options?: { value: string; label: string }[];
    accept?: string;
}

interface FormModalProps {
    isOpen: boolean;
    title: string;
    fields: FormField[];
    initialData?: Record<string, any>;
    onClose: () => void;
    onSubmit: (data: Record<string, any>) => Promise<void>;
}

const FormModal: React.FC<FormModalProps> = ({
    isOpen,
    title,
    fields,
    initialData = {},
    onClose,
    onSubmit,
}) => {
    const [formData, setFormData] = useState<Record<string, any>>({});
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState<string | null>(null);

    useEffect(() => {
        if (isOpen) {
            const initial: Record<string, any> = {};
            fields.forEach((field) => {
                initial[field.key] = initialData[field.key] ?? (field.type === 'checkbox' ? false : '');
            });
            setFormData(initial);
            setError(null);
            // Disable body scroll when modal is open
            document.body.style.overflow = 'hidden';
        } else {
            // Re-enable body scroll when modal is closed
            document.body.style.overflow = '';
        }

        return () => {
            document.body.style.overflow = '';
        };
    }, [isOpen, initialData, fields]);

    const handleChange = (key: string, value: any) => {
        setFormData((prev) => ({ ...prev, [key]: value }));
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        setLoading(true);
        setError(null);
        try {
            await onSubmit(formData);
            onClose();
        } catch (err: any) {
            setError(err.message || 'An error occurred');
        } finally {
            setLoading(false);
        }
    };

    if (!isOpen) return null;

    return (
        <div className={styles.overlay} onClick={onClose}>
            <div className={styles.modal} onClick={(e) => e.stopPropagation()}>
                <div className={styles.header}>
                    <h2 className={styles.title}>{title}</h2>
                    <button className={styles.closeBtn} onClick={onClose}>
                        <svg viewBox="0 0 24 24" width="20" height="20">
                            <path d="M18 6L6 18M6 6l12 12" stroke="currentColor" strokeWidth="2" strokeLinecap="round" />
                        </svg>
                    </button>
                </div>

                <form className={styles.form} onSubmit={handleSubmit}>
                    <div className={styles.body}>
                        {error && <div className={styles.error}>{error}</div>}

                        {fields.map((field) => (
                            <div key={field.key} className={styles.field}>
                                <label className={styles.label}>
                                    {field.label}
                                    {field.required && <span className={styles.required}>*</span>}
                                </label>

                                {field.type === 'textarea' ? (
                                    <textarea
                                        className={styles.textarea}
                                        value={formData[field.key] || ''}
                                        onChange={(e) => handleChange(field.key, e.target.value)}
                                        placeholder={field.placeholder}
                                        required={field.required}
                                        rows={4}
                                    />
                                ) : field.type === 'checkbox' ? (
                                    <label className={styles.checkbox}>
                                        <input
                                            type="checkbox"
                                            checked={formData[field.key] || false}
                                            onChange={(e) => handleChange(field.key, e.target.checked)}
                                        />
                                        <span>Enabled</span>
                                    </label>
                                ) : field.type === 'select' ? (
                                    <select
                                        className={styles.select}
                                        value={formData[field.key] || ''}
                                        onChange={(e) => handleChange(field.key, e.target.value)}
                                        required={field.required}
                                    >
                                        <option value="">Select...</option>
                                        {field.options?.map((opt) => (
                                            <option key={opt.value} value={opt.value}>
                                                {opt.label}
                                            </option>
                                        ))}
                                    </select>
                                ) : field.type === 'file' ? (
                                    <FileUpload
                                        value={formData[field.key]}
                                        onChange={(file) => handleChange(field.key, file)}
                                        accept={field.accept}
                                    />
                                ) : (
                                    <input
                                        type={field.type || 'text'}
                                        className={styles.input}
                                        value={formData[field.key] || ''}
                                        onChange={(e) => handleChange(field.key, field.type === 'number' ? Number(e.target.value) : e.target.value)}
                                        placeholder={field.placeholder}
                                        required={field.required}
                                    />
                                )}
                            </div>
                        ))}
                    </div>

                    <div className={styles.footer}>
                        <button type="button" className={styles.cancelBtn} onClick={onClose} disabled={loading}>
                            Cancel
                        </button>
                        <button type="submit" className={styles.submitBtn} disabled={loading}>
                            {loading ? 'Saving...' : 'Save'}
                        </button>
                    </div>
                </form>
            </div>
        </div>
    );
};

export default FormModal;
