import React, { useRef } from 'react';
import styles from './FileUpload.module.css';

interface FileUploadProps {
    value: any;
    onChange: (file: File | null) => void;
    accept?: string;
}

const FileUpload: React.FC<FileUploadProps> = ({ value, onChange, accept }) => {
    const inputRef = useRef<HTMLInputElement>(null);

    const handleClick = () => {
        inputRef.current?.click();
    };

    const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        const file = e.target.files?.[0];
        if (file) {
            onChange(file);
        }
    };

    const [previewUrl, setPreviewUrl] = React.useState<string | null>(null);

    React.useEffect(() => {
        if (!value) {
            setPreviewUrl(null);
            return;
        }

        if (value instanceof File) {
            const url = URL.createObjectURL(value);
            setPreviewUrl(url);
            return () => URL.revokeObjectURL(url);
        } else {
            setPreviewUrl(value as string);
        }
    }, [value]);

    const isFile = value instanceof File;

    const isImage = () => {
        if (!value) return false;
        if (value instanceof File) {
            return value.type.startsWith('image/');
        }
        // Check extension for string URLs
        const ext = (value as string).split('.').pop()?.toLowerCase();
        return ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg', 'bmp'].includes(ext || '');
    };

    const showImage = isImage();

    return (
        <div className={styles.container}>
            <div
                className={`${styles.uploadArea} ${value ? styles.hasFile : ''}`}
                onClick={handleClick}
            >
                <input
                    ref={inputRef}
                    type="file"
                    className={styles.hiddenInput}
                    onChange={handleFileChange}
                    accept={accept}
                />

                {value ? (
                    <div className={styles.preview}>
                        {showImage && previewUrl ? (
                            <img src={previewUrl} alt="Preview" className={styles.thumbnail} />
                        ) : (
                            <div className={styles.fileIcon}>
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" width="40" height="40">
                                    <path d="M13 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V9z"></path>
                                    <polyline points="13 2 13 9 20 9"></polyline>
                                </svg>
                            </div>
                        )}
                        <div className={styles.fileInfo}>
                            <span className={styles.fileName}>
                                {isFile ? (value as File).name : (value as string).split('/').pop()}
                            </span>
                            {isFile && (
                                <span className={styles.fileSize}>
                                    {((value as File).size / 1024).toFixed(1)} KB
                                </span>
                            )}
                        </div>
                        <span className={styles.changeBtn}>Change</span>
                    </div>
                ) : (
                    <div className={styles.placeholder}>
                        <svg className={styles.icon} viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2">
                            <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4" />
                            <polyline points="17 8 12 3 7 8" />
                            <line x1="12" y1="3" x2="12" y2="15" />
                        </svg>
                        <span className={styles.text}>Upload File</span>
                        <span className={styles.subText}>Supports all formats</span>
                    </div>
                )}
            </div>
        </div>
    );
};

export default FileUpload;
