/* eslint-disable @typescript-eslint/no-explicit-any */
import React, { useState, useMemo } from 'react';
import styles from './DataTable.module.css';

export interface Column<T> {
    key: string;
    title: string;
    width?: string;
    render?: (value: any, record: T) => React.ReactNode;
}

interface DataTableProps<T> {
    title: string;
    columns: Column<T>[];
    data: T[];
    loading?: boolean;
    idKey?: string;
    pageSize?: number;
    onAdd?: () => void;
    onEdit?: (record: T) => void;
    onDelete?: (record: T) => void;
}

function DataTable<T extends Record<string, any>>({
    title,
    columns,
    data,
    loading = false,
    idKey = 'id',
    pageSize = 20,
    onAdd,
    onEdit,
    onDelete,
}: DataTableProps<T>) {
    const [currentPage, setCurrentPage] = useState(1);

    const totalPages = Math.ceil(data.length / pageSize);

    const paginatedData = useMemo(() => {
        const start = (currentPage - 1) * pageSize;
        return data.slice(start, start + pageSize);
    }, [data, currentPage, pageSize]);

    // Reset to page 1 when data changes
    React.useEffect(() => {
        setCurrentPage(1);
    }, [data.length]);

    const handlePrevPage = () => {
        setCurrentPage((prev) => Math.max(prev - 1, 1));
    };

    const handleNextPage = () => {
        setCurrentPage((prev) => Math.min(prev + 1, totalPages));
    };

    return (
        <div className={styles.container}>
            <div className={styles.header}>
                <h1 className={styles.title}>{title}</h1>
                {onAdd && (
                    <button className={styles.addBtn} onClick={onAdd}>
                        <svg viewBox="0 0 24 24" width="16" height="16">
                            <path d="M12 5v14M5 12h14" stroke="currentColor" strokeWidth="2" strokeLinecap="round" />
                        </svg>
                        Add New
                    </button>
                )}
            </div>

            <div className={styles.tableWrapper}>
                {loading ? (
                    <div className={styles.loading}>
                        <div className={styles.spinner} />
                        <span>Loading...</span>
                    </div>
                ) : (
                    <table className={styles.table}>
                        <thead>
                            <tr>
                                {columns.map((col) => (
                                    <th key={col.key} style={{ width: col.width }}>
                                        {col.title}
                                    </th>
                                ))}
                                {(onEdit || onDelete) && <th style={{ width: '120px' }} className={styles.stickyColumn}>Actions</th>}
                            </tr>
                        </thead>
                        <tbody>
                            {paginatedData.length === 0 ? (
                                <tr>
                                    <td colSpan={columns.length + (onEdit || onDelete ? 1 : 0)} className={styles.empty}>
                                        No data available
                                    </td>
                                </tr>
                            ) : (
                                paginatedData.map((record) => (
                                    <tr key={record[idKey]}>
                                        {columns.map((col) => (
                                            <td key={col.key}>
                                                {col.render ? col.render(record[col.key], record) : String(record[col.key] ?? '-')}
                                            </td>
                                        ))}
                                        {(onEdit || onDelete) && (
                                            <td className={styles.stickyColumn}>
                                                <div className={styles.actions}>
                                                    {onEdit && (
                                                        <button className={styles.editBtn} onClick={() => onEdit(record)} title="Edit">
                                                            <svg viewBox="0 0 24 24" width="16" height="16">
                                                                <path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7" />
                                                                <path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z" />
                                                            </svg>
                                                        </button>
                                                    )}
                                                    {onDelete && (
                                                        <button className={styles.deleteBtn} onClick={() => onDelete(record)} title="Delete">
                                                            <svg viewBox="0 0 24 24" width="16" height="16">
                                                                <polyline points="3 6 5 6 21 6" />
                                                                <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2" />
                                                            </svg>
                                                        </button>
                                                    )}
                                                </div>
                                            </td>
                                        )}
                                    </tr>
                                ))
                            )}
                        </tbody>
                    </table>
                )}
            </div>

            <div className={styles.footer}>
                <span className={styles.count}>Total: {data.length} items</span>
                {totalPages > 1 && (
                    <div className={styles.pagination}>
                        <button
                            className={styles.pageBtn}
                            onClick={handlePrevPage}
                            disabled={currentPage === 1}
                        >
                            ‹ Prev
                        </button>
                        <span className={styles.pageInfo}>
                            Page {currentPage} of {totalPages}
                        </span>
                        <button
                            className={styles.pageBtn}
                            onClick={handleNextPage}
                            disabled={currentPage === totalPages}
                        >
                            Next ›
                        </button>
                    </div>
                )}
            </div>
        </div>
    );
}

export default DataTable;
