#!/usr/bin/env python3
"""
EA-RAG 多专家版本 v3 - 启动脚本

对应架构图: EA-RAG Multi-Expert Architecture

8 Core Standards:
- GB 50016 (Fire), GB 55019 (Fire)
- GB 55015 (Energy), GB 50189 (Energy)
- GB 50096 (Residential)
- GB 50352 (General)
- GB 50099 (Schools)
- GB 50763 (Accessibility)

6 Expert Domains:
- Fire Safety, Energy, Residential, Schools, General, Accessibility

使用方法:
    python main.py --init       # 创建目录结构（空文件夹）
    python main.py              # 交互模式
    python main.py -q "问题"    # 单次查询
    python main.py --rebuild    # 重建索引
    python main.py --explain "问题"  # 解释路由决策
"""

import argparse
from pathlib import Path


def create_directories():
    """
    创建知识库目录结构（空文件夹）

    6 Expert Domains + 8 Core Standards
    """
    KNOWLEDGE_BASE_DIR = Path("knowledge_base")

    # 6个专家配置
    EXPERTS = {
        "fire_safety": {"name": "消防专家", "icon": "🔥", "standards": "GB 50016, GB 55019"},
        "accessibility": {"name": "无障碍专家", "icon": "♿", "standards": "GB 50763"},
        "energy": {"name": "节能专家", "icon": "⚡", "standards": "GB 55015, GB 50189"},
        "residential": {"name": "住宅专家", "icon": "🏠", "standards": "GB 50096"},
        "schools": {"name": "学校专家", "icon": "🏫", "standards": "GB 50099"},
        "general": {"name": "通用专家", "icon": "📐", "standards": "GB 50352"},
    }

    print("📁 创建知识库目录结构...")
    print("   6 Expert Domains + 8 Core Standards")
    print()

    for expert_id, config in EXPERTS.items():
        expert_dir = KNOWLEDGE_BASE_DIR / expert_id
        code_dir = expert_dir / "code"
        exp_dir = expert_dir / "experience"

        code_dir.mkdir(parents=True, exist_ok=True)
        exp_dir.mkdir(parents=True, exist_ok=True)

        print(f"   {config['icon']} {config['name']} ({config['standards']})")
        print(f"      ├── code/")
        print(f"      └── experience/")

    print("\n✅ 目录结构创建完成")
    print_directory_structure()
    print("\n💡 请将规范文档放入对应的 code/ 文件夹")
    print("💡 请将经验文档放入对应的 experience/ 文件夹")


def print_directory_structure():
    """打印目录结构说明"""
    print("\n" + "=" * 70)
    print("📂 知识库目录结构 (6 Expert Domains)")
    print("=" * 70)
    print("""
knowledge_base/
├── fire_safety/              # 🔥 Fire Safety Expert
│   ├── code/                 # Code Layer (GB 50016, GB 55019)
│   └── experience/           # Exp. Layer
│
├── accessibility/            # ♿ Accessibility Expert  
│   ├── code/                 # Code Layer (GB 50763)
│   └── experience/           # Exp. Layer
│
├── energy/                   # ⚡ Energy Expert
│   ├── code/                 # Code Layer (GB 55015, GB 50189)
│   └── experience/           # Exp. Layer
│
├── residential/              # 🏠 Residential Expert
│   ├── code/                 # Code Layer (GB 50096)
│   └── experience/           # Exp. Layer
│
├── schools/                  # 🏫 Schools Expert
│   ├── code/                 # Code Layer (GB 50099)
│   └── experience/           # Exp. Layer
│
└── general/                  # 📐 General Expert
    ├── code/                 # Code Layer (GB 50352)
    └── experience/           # Exp. Layer
""")
    print("=" * 70)
    print("\n📋 8 Core Standards Mapping:")
    print("   🔥 Fire:         GB 50016, GB 55019")
    print("   ⚡ Energy:       GB 55015, GB 50189")
    print("   🏠 Residential:  GB 50096")
    print("   📐 General:      GB 50352")
    print("   🏫 Schools:      GB 50099")
    print("   ♿ Accessibility: GB 50763")
    print()
    print("📋 示例查询:")
    print('   python main.py -q "商业建筑走廊宽度要求？"')
    print('   python main.py -q "住宅卧室面积最小是多少？"')
    print('   python main.py -q "学校教室采光系数要求？"')


def interactive_mode(system):
    """交互模式"""
    print("\n" + "=" * 70)
    print("EA-RAG 多专家建筑规范咨询系统 v3")
    print("6 Experts: Fire | Accessibility | Energy | Residential | Schools | General")
    print("输入问题查询，输入 'quit' 退出")
    print("=" * 70)

    while True:
        try:
            question = input("\n🙋 请输入问题: ").strip()

            if question.lower() in ['quit', 'exit', 'q']:
                print("👋 再见！")
                break

            if not question:
                continue

            result = system.query(question)
            print(result)

        except KeyboardInterrupt:
            print("\n👋 再见！")
            break
        except Exception as e:
            print(f"❌ 错误: {e}")
            import traceback
            traceback.print_exc()


def main():
    parser = argparse.ArgumentParser(description="EA-RAG 多专家建筑规范咨询系统 v3")
    parser.add_argument("--init", action="store_true", help="创建目录结构（空文件夹）")
    parser.add_argument("--rebuild", action="store_true", help="重建索引")
    parser.add_argument("--query", "-q", type=str, help="单次查询")
    parser.add_argument("--api-key", type=str, help="Google API Key")
    parser.add_argument("--structure", action="store_true", help="显示目录结构")
    parser.add_argument("--explain", type=str, help="解释路由决策")

    args = parser.parse_args()

    if args.structure:
        print_directory_structure()
        return

    if args.init:
        create_directories()
        return

    from ea_rag import EARAGSystem

    system = EARAGSystem(api_key=args.api_key)
    system.load_knowledge_base(force_rebuild=args.rebuild)

    if args.explain:
        print(system.explain_routing(args.explain))
        return

    if args.query:
        print(system.query(args.query))
    else:
        interactive_mode(system)


if __name__ == "__main__":
    main()