"""
EA-RAG 主系统 - 多专家版本 v3

实现架构图的完整流程:
User Query → Query Router → [Experts] → Multi-Expert Fusion → Structured Response

向量存储: ChromaDB
"""

import os
from typing import Optional, Dict

from .config import EXPERTS, KNOWLEDGE_BASE_DIR
from .models import RetrievalResult, StructuredResponse
from .embeddings import EmbeddingService
from .vector_store import VectorStore
from .expert import Expert, QueryRouter
from .generator import Generator


class EARAGSystem:
    """
    EA-RAG 多专家系统 v3

    对应架构图: EA-RAG Multi-Expert Architecture

    流程:
    1. User Query: 接收用户查询
    2. Query Router: 识别相关领域，决定激活哪些专家
    3. Expert Modules: 在激活的专家中检索 (Code Layer + Exp. Layer)
    4. Multi-Expert Fusion: 结果聚合、冲突检测、来源标注
    5. Structured Response: 生成结构化回答

    向量存储: ChromaDB (knowledge_base/chroma_db/)
    """

    def __init__(self, api_key: Optional[str] = None):
        """初始化系统"""
        # 配置 API Key
        self.api_key = api_key or os.environ.get("GOOGLE_API_KEY")

        # 初始化服务
        self.embedding_service = EmbeddingService(self.api_key)
        self.vector_store = VectorStore()  # ChromaDB
        self.generator = Generator(self.api_key)

        # 初始化专家模块 - 对应架构图中的各个 Expert
        self.experts: Dict[str, Expert] = {}
        for expert_id, config in EXPERTS.items():
            self.experts[expert_id] = Expert(
                expert_id,
                config,
                self.embedding_service,
                self.vector_store  # 传入 VectorStore
            )

        # 初始化路由器 - 对应架构图中的 Query Router
        self.router = QueryRouter(self.experts)

        print("✅ EA-RAG 多专家系统 v3 初始化完成")
        print(f"   向量数据库: ChromaDB ({self.vector_store.persist_dir})")
        print(f"   已配置 {len(self.experts)} 个专家模块:")
        for expert in self.experts.values():
            codes_str = ", ".join(expert.codes[:3])
            if len(expert.codes) > 3:
                codes_str += "..."
            print(f"   {expert.icon} {expert.name}: {codes_str}")

    def setup_directories(self):
        """创建所有专家目录"""
        print("\n📁 创建知识库目录结构...")

        for expert in self.experts.values():
            expert.setup_directories()
            print(f"   {expert.icon} {expert.name}: {expert.base_dir}/")
            print(f"      ├── code/       <- Code Layer (规范文档)")
            print(f"      └── experience/ <- Exp. Layer (经验文档)")

        print(f"\n✅ 目录结构已创建: {KNOWLEDGE_BASE_DIR}/")

    def load_knowledge_base(self, force_rebuild: bool = False):
        """加载知识库"""
        print("\n📚 加载多专家知识库...")
        print(f"   ChromaDB 路径: {self.vector_store.persist_dir}")

        for expert in self.experts.values():
            if not force_rebuild and expert.index_exists():
                stats = expert.get_stats()
                print(f"   {expert.icon} {expert.name}: "
                      f"Code {stats['code_chunks']}, Exp {stats['experience_chunks']} (已存在)")
            else:
                expert.build_index(force_rebuild=force_rebuild)

        self._print_stats()

    def _print_stats(self):
        """打印统计"""
        print("\n" + "=" * 60)
        print("📊 知识库统计 (Knowledge Base Stats)")
        print("=" * 60)

        total_code = 0
        total_exp = 0

        for expert in self.experts.values():
            stats = expert.get_stats()
            code_n = stats['code_chunks']
            exp_n = stats['experience_chunks']
            total_code += code_n
            total_exp += exp_n

            status = "✓" if (code_n > 0 or exp_n > 0) else "○"
            codes_str = ", ".join(stats['codes'][:2])
            print(f"   {status} {expert.icon} {expert.name}")
            print(f"      Codes: {codes_str}")
            print(f"      Code Layer: {code_n} chunks, Exp Layer: {exp_n} chunks")

        print("-" * 60)
        print(f"   总计: Code {total_code}, Experience {total_exp}")
        print(f"   Collections: {len(self.vector_store.list_collections())}")
        print("=" * 60)

    def query(self, question: str) -> str:
        """
        查询接口 - 实现架构图的完整流程

        Args:
            question: 用户问题 (User Query)

        Returns:
            格式化的结构化响应 (Structured Response)
        """
        print(f"\n🔍 查询: {question}")

        # ====== Step 1: Query Router ======
        # 对应架构图: Query Router - Identify Relevant Domains
        routing_result = self.router.route(question)

        activated = routing_result["activated"]
        not_activated = routing_result["not_activated"]

        activated_names = [
            f"{self.experts[eid].icon}{self.experts[eid].name}"
            for eid, _ in activated
        ]
        print(f"   Query Router → Activated: {', '.join(activated_names)}")

        if not_activated:
            not_activated_names = [
                f"{self.experts[eid].icon}{self.experts[eid].name}"
                for eid in not_activated
            ]
            print(f"                 Not Activated: {', '.join(not_activated_names)}")

        # ====== Step 2: Expert Retrieval ======
        # 对应架构图: 各 Expert 模块的 Code Layer + Exp. Layer 检索
        query_embedding = self.embedding_service.get_embedding(question)

        expert_results = {}
        for expert_id, score in activated:
            expert = self.experts[expert_id]
            result = expert.search(query_embedding, activation_score=score)
            expert_results[expert_id] = result

            code_n = len(result.code_chunks)
            exp_n = len(result.experience_chunks)
            print(f"   {expert.icon} {expert.name}: Code {code_n}, Exp {exp_n}")

        retrieval_result = RetrievalResult(
            query=question,
            activated_experts=[eid for eid, _ in activated],
            not_activated_experts=not_activated,
            expert_results=expert_results
        )

        # ====== Step 3 & 4: Multi-Expert Fusion + Structured Response ======
        # 对应架构图: Multi-Expert Fusion → Structured Response
        print("   Multi-Expert Fusion → Generating Structured Response...")

        structured_response = self.generator.generate(question, retrieval_result)

        # ====== 返回格式化输出 ======
        return structured_response.format_display()

    def query_with_details(self, question: str) -> StructuredResponse:
        """
        查询接口 - 返回结构化响应对象（用于程序化访问）
        """
        print(f"\n🔍 查询: {question}")

        # Query Router
        routing_result = self.router.route(question)
        activated = routing_result["activated"]
        not_activated = routing_result["not_activated"]

        # Expert Retrieval
        query_embedding = self.embedding_service.get_embedding(question)

        expert_results = {}
        for expert_id, score in activated:
            expert = self.experts[expert_id]
            result = expert.search(query_embedding, activation_score=score)
            expert_results[expert_id] = result

        retrieval_result = RetrievalResult(
            query=question,
            activated_experts=[eid for eid, _ in activated],
            not_activated_experts=not_activated,
            expert_results=expert_results
        )

        # Generate
        return self.generator.generate(question, retrieval_result)

    def explain_routing(self, question: str) -> str:
        """解释路由决策（调试用）"""
        return self.router.explain_routing(question)

    def clear_vector_store(self):
        """清空向量数据库"""
        print("⚠️ 清空所有向量数据...")
        self.vector_store.clear_all()
        print("✅ 向量数据库已清空")