"""
EA-RAG 数据模型 - 多专家版本 v3
对应架构图中的各个数据结构
"""

from dataclasses import dataclass, field
from typing import List, Dict, Optional, Tuple


@dataclass
class Chunk:
    """文档分块"""
    content: str
    metadata: Dict
    doc_type: str  # "code" 或 "experience"
    expert_id: str  # 所属专家ID
    embedding: Optional[List[float]] = None
    
    def to_dict(self) -> Dict:
        return {
            "content": self.content,
            "metadata": self.metadata,
            "doc_type": self.doc_type,
            "expert_id": self.expert_id,
            "embedding": self.embedding
        }
    
    @classmethod
    def from_dict(cls, data: Dict) -> "Chunk":
        return cls(
            content=data["content"],
            metadata=data["metadata"],
            doc_type=data["doc_type"],
            expert_id=data["expert_id"],
            embedding=data.get("embedding")
        )


@dataclass
class ExpertResult:
    """单个专家的检索结果 - 对应架构图中每个Expert模块的输出"""
    expert_id: str
    expert_name: str
    expert_icon: str
    code_chunks: List[Tuple[Chunk, float]]       # Code Layer 检索结果
    experience_chunks: List[Tuple[Chunk, float]] # Exp. Layer 检索结果
    activation_score: float = 0.0                # 激活分数


@dataclass
class RetrievalResult:
    """多专家检索结果 - Multi-Expert Fusion 的输入"""
    query: str                                    # 原始查询
    activated_experts: List[str]                  # 被激活的专家ID列表 (Activated)
    not_activated_experts: List[str]              # 未激活的专家ID列表 (Not Activated)
    expert_results: Dict[str, ExpertResult]       # expert_id -> ExpertResult


@dataclass
class CodeRequirement:
    """
    规范要求 - 对应架构图 Structured Response 中的 Regulatory Requirements
    
    示例: [Fire] GB 50016 §5.5.18: Evacuation corridor ≥1.4m
    """
    expert_id: str          # 来源专家
    expert_icon: str        # 专家图标
    expert_name: str        # 专家名称 (如 "Fire", "Accessibility")
    code_id: str            # 规范编号 (如 "GB 50016")
    clause: str             # 条款号 (如 "§5.5.18")
    content: str            # 条款内容
    source: str             # 来源文件


@dataclass
class StructuredResponse:
    """
    结构化响应 - 对应架构图中的 Structured Response
    
    包含:
    - Regulatory Requirements: 📋 规范要求
    - Exp. Interpretation: 💡 经验解读
    - Risk Tip: ⚠️ 风险提示
    - Liability Boundary: ⚖️ 责任边界
    """
    
    # 激活的专家信息
    activated_experts: List[Dict]  # [{id, name, icon}]
    
    # 📋 Regulatory Requirements - 规范要求
    # 对应图中: [Fire] GB 50016 §5.5.18: Evacuation corridor ≥1.4m
    #          [Accessibility] GB 55019 §4.2.3: Accessibility corridor ≥1.8m
    code_requirements: List[CodeRequirement]
    
    # 💡 Exp. Interpretation - 经验解读
    # 对应图中: Underground commercial suggested uniform 1.8m design
    expert_interpretation: str
    
    # ⚠️ Risk Tip - 风险提示
    # 对应图中: Common error: Satisfying only fire, ignoring accessibility
    risk_alerts: List[str]
    
    # ⚖️ Liability Boundary - 责任边界
    # 对应图中: Subject to drawing review agency's opinion
    professional_boundaries: str
    
    # 冲突检测结果 - Multi-Expert Fusion 的 Conflict Detection
    conflicts: List[Dict] = field(default_factory=list)
    
    # 来源标注 - Multi-Expert Fusion 的 Source Attribution
    sources: List[Dict] = field(default_factory=list)
    
    # 原始LLM响应
    raw_answer: str = ""
    
    def format_display(self) -> str:
        """直接显示 LLM 原始回答"""
        lines = []

        # 显示激活的专家
        if self.activated_experts:
            expert_str = " + ".join([f"{e['icon']}{e['name']}" for e in self.activated_experts])
            lines.append(f"🎯 激活专家: {expert_str}")
            lines.append("")

        lines.append("=" * 60)

        # 直接输出 LLM 的回答（新格式）
        lines.append(self.raw_answer)

        lines.append("=" * 60)
        
        return "\n".join(lines)


# 兼容旧版本的 Response 类
@dataclass 
class Response:
    """系统响应 (兼容旧版本)"""
    activated_experts: List[str]
    expert_answers: Dict[str, Dict]
    raw_answer: str
    sources: List[Dict]
