"""
EA-RAG 生成器 - 精简版 v3
"""

import re
import time
from typing import List, Dict
from google import genai

from .config import LLM_MODEL, EXPERTS
from .models import RetrievalResult, StructuredResponse, CodeRequirement
from .fusion import MultiExpertFusion


class Generator:
    def __init__(self, api_key: str = None):
        self.client = genai.Client(api_key=api_key) if api_key else genai.Client()
        self.fusion = MultiExpertFusion()

    def generate(self, query: str, retrieval_result: RetrievalResult) -> StructuredResponse:
        fusion_result = self.fusion.fuse(retrieval_result)
        context = self.fusion.build_context_for_llm(fusion_result)
        prompt = self._build_prompt(query, context, fusion_result)

        # 调用 LLM（带重试）
        for attempt in range(3):
            try:
                response = self.client.models.generate_content(model=LLM_MODEL, contents=prompt)
                raw_answer = response.text
                break
            except Exception as e:
                if "503" in str(e) or "overloaded" in str(e).lower():
                    if attempt < 2:
                        print(f"      ⏳ 服务器繁忙，等待10s...")
                        time.sleep(10)
                    else:
                        raise e
                else:
                    raise e

        return self._build_structured_response(query, raw_answer, fusion_result)

    def _build_prompt(self, query: str, context: str, fusion_result: Dict) -> str:
        experts = ", ".join([f"{e['icon']}{e['name']}" for e in fusion_result["activated_experts"]])

        # 检测语言
        is_en = len(re.findall(r'[\u4e00-\u9fff]', query)) < len(query) * 0.1

        # 冲突提示
        conflicts = ""
        if fusion_result["conflicts"]:
            conflicts = "⚠️跨专业冲突：" + "；".join([c["description"] for c in fusion_result["conflicts"]])

        if is_en:
            return self._en_prompt(query, context, experts, conflicts)
        return self._zh_prompt(query, context, experts, conflicts)

    def _zh_prompt(self, query: str, context: str, experts: str, conflicts: str) -> str:
        return f"""建筑规范专家系统。激活专家：{experts}
{conflicts}

{context}

问题：{query}

按此格式回答：

### 📋 规范引用
- 【规范名称 + 编号 + 条文号】：条文原文内容
- 示例：【《建筑环境通用规范》GB 55016-2021 第5.5.18条】：高层公共建筑的疏散走道净宽度不应小于1.40m

### 📖 规范理解
用通俗语言解释条文含义，并举例说明：
- 条文解读：这条规范的核心要求是什么
- 举例说明：用具体场景举例，如"例如：一个500㎡的商场，按此条规定需要..."
- 常见误区：设计师容易误解的地方

### 🎯 方案建议
A-底线方案：参数+适用场景+风险
B-推荐方案⭐：参数+适用场景+优势
C-进阶方案：参数+适用场景+收益

### 💡 经验解读
规范协调关系、审图关注点、常见做法（2-3句）

### ⚠️ 风险提示
常见错误、易忽视条款（列表，3条内）

### ⚖️ 责任边界
适用范围（1句）

原则：条文标注【编号-年份 条号】；方案有具体数值；举例要有具体数字；不编造条文"""

    def _en_prompt(self, query: str, context: str, experts: str, conflicts: str) -> str:
        return f"""Building code expert system. Experts: {experts}
{conflicts}

{context}

Question: {query}

Response format:

### 📋 Code References
- 【Standard Title + Number + Clause Number】: Original text of the clause
- Example: 【General Specification for Building Environment GB 55016-2021 Clause 5.5.18】: The clear width of evacuation corridors in high-rise public buildings shall not be less than 1.40m.

### 📖 Code Interpretation
Explain provisions in plain language with examples:
- Interpretation: Core requirement of this provision
- Example: Concrete scenario, e.g. "For a 500㎡ mall, this provision requires..."
- Common misconceptions: Where designers often misunderstand

### 🎯 Recommendations
A-Baseline: params+scenario+risk
B-Recommended⭐: params+scenario+advantage
C-Premium: params+scenario+benefit

### 💡 Interpretation
Code coordination, review focus, common practice (2-3 sentences)

### ⚠️ Risk Alerts
Common errors, overlooked clauses (list, max 3)

### ⚖️ Disclaimer
Scope (1 sentence)

Rules: Cite【Code-Year Article】; specific values; examples with numbers; no fabrication"""

    def _build_structured_response(self, query: str, raw_answer: str, fusion_result: Dict) -> StructuredResponse:
        sections = self._parse_sections(raw_answer)
        code_requirements = self._extract_code_requirements(fusion_result)
        risk_alerts = self._extract_risk_alerts(sections.get("风险提示", sections.get("Risk Alerts", "")))

        return StructuredResponse(
            activated_experts=fusion_result["activated_experts"],
            code_requirements=code_requirements,
            expert_interpretation=sections.get("经验解读", sections.get("Interpretation", "")),
            risk_alerts=risk_alerts,
            professional_boundaries=sections.get("责任边界", sections.get("Disclaimer", "以审图机构意见为准")),
            conflicts=fusion_result["conflicts"],
            sources=fusion_result["sources"],
            raw_answer=raw_answer
        )

    def _parse_sections(self, text: str) -> Dict[str, str]:
        sections = {}
        markers = [
            ("规范引用", ["📋 规范引用", "📋 Code References"]),
            ("规范理解", ["📖 规范理解", "📖 Code Interpretation"]),
            ("方案建议", ["🎯 方案建议", "🎯 Recommendations"]),
            ("决策逻辑", ["🔀 决策逻辑", "🔀 Decision Logic"]),
            ("经验解读", ["💡 经验解读", "💡 Interpretation"]),
            ("风险提示", ["⚠️ 风险提示", "⚠️ Risk Alerts"]),
            ("责任边界", ["⚖️ 责任边界", "⚖️ Disclaimer"]),
        ]

        positions = []
        for name, marks in markers:
            for m in marks:
                pos = text.find(m)
                if pos != -1:
                    positions.append((pos, name))
                    break

        positions.sort(key=lambda x: x[0])

        for i, (pos, name) in enumerate(positions):
            end = positions[i + 1][0] if i + 1 < len(positions) else len(text)
            content = text[pos:end]
            lines = content.split('\n')
            sections[name] = '\n'.join(lines[1:]).strip()

        return sections

    def _extract_code_requirements(self, fusion_result: Dict) -> List[CodeRequirement]:
        requirements = []
        for item in fusion_result.get("aggregated_code", []):
            clause = item.get("clause", "")
            if not clause:
                match = re.search(r'[§第]?(\d+(?:\.\d+)+)', item["content"])
                if match:
                    clause = f"第{match.group(1)}条"

            source = item.get("source", "")
            code_id = ""
            code_match = re.search(r'(GB\s*\d+)', source, re.IGNORECASE)
            if code_match:
                code_id = code_match.group(1).upper()

            content = item["content"].split('\n')[0]
            if len(content) > 100:
                content = content[:100] + "..."

            requirements.append(CodeRequirement(
                expert_id=item["expert_id"],
                expert_icon=item["expert_icon"],
                expert_name=item["expert_name"],
                code_id=code_id,
                clause=clause,
                content=content,
                source=source
            ))
        return requirements

    def _extract_risk_alerts(self, text: str) -> List[str]:
        alerts = []
        for line in text.split('\n'):
            line = line.strip()
            if line.startswith(('-', '•', '*', '·')):
                alert = line.lstrip('-•*· ').strip()
                if alert:
                    alerts.append(alert)
        if not alerts and text.strip():
            alerts.append(text.strip())
        return alerts