"""
EA-RAG 嵌入向量服务 - 带速率限制
"""

import time
from typing import List
from google import genai

from .config import EMBEDDING_MODEL, EMBEDDING_BATCH_SIZE

# 速率限制配置
REQUESTS_PER_MINUTE = 90  # 留一点余量，免费层限制是100
REQUEST_INTERVAL = 60.0 / REQUESTS_PER_MINUTE  # 约0.67秒/请求


class EmbeddingService:
    """嵌入向量服务 - 带速率限制"""

    def __init__(self, api_key: str = None):
        self.client = genai.Client(api_key=api_key) if api_key else genai.Client()
        self.last_request_time = 0

    def _wait_for_rate_limit(self):
        """等待以遵守速率限制"""
        elapsed = time.time() - self.last_request_time
        if elapsed < REQUEST_INTERVAL:
            time.sleep(REQUEST_INTERVAL - elapsed)
        self.last_request_time = time.time()

    def get_embedding(self, text: str, max_retries: int = 3) -> List[float]:
        """获取单个文本的嵌入向量（带重试）"""
        for attempt in range(max_retries):
            try:
                self._wait_for_rate_limit()
                result = self.client.models.embed_content(
                    model=EMBEDDING_MODEL,
                    contents=text
                )
                return result.embeddings[0].values
            except Exception as e:
                if "429" in str(e) or "RESOURCE_EXHAUSTED" in str(e):
                    # 配额超限，等待后重试
                    wait_time = 30 * (attempt + 1)  # 30s, 60s, 90s
                    print(f"      ⏳ API配额限制，等待 {wait_time}s 后重试...")
                    time.sleep(wait_time)
                else:
                    raise e
        raise Exception(f"重试 {max_retries} 次后仍失败")

    def get_embeddings_batch(self, texts: List[str]) -> List[List[float]]:
        """批量获取嵌入向量（带进度显示）"""
        embeddings = []
        total = len(texts)

        for i, text in enumerate(texts):
            emb = self.get_embedding(text)
            embeddings.append(emb)

            # 每50个显示进度
            if (i + 1) % 50 == 0 or (i + 1) == total:
                print(f"         进度: {i + 1}/{total} ({100 * (i + 1) // total}%)")

        return embeddings