import json
from pathlib import Path
from datetime import datetime, timezone

# ================== CONFIGURAZIONE ==================
# Modifica questa costante per cambiare la data minima accettata.
# Tutti gli eventi con "time" (timestamp lato client) precedente a questa
# data verranno SCARTATI.

# Today's date at 1 AM UTC
_today = datetime.now(timezone.utc).replace(hour=1, minute=0, second=0, microsecond=0)
# Fissa la data minima a 15 novembre 2025, ore 00:00 UTC
MIN_DATE_ISO = "2025-11-15T00:00:00Z"

INPUT_FILE = "original_events.jsonl"  # File di input (formato JSON Lines)
OUTPUT_FILE = "json_preprocess_usefuldata.json"  # File di output

# ====================================================


def _iso_to_unix(iso_str: str) -> float:
    """
    Converte una stringa ISO 8601 (es. '2025-01-01T00:00:00Z')
    in timestamp Unix (float, secondi).
    """
    # Gestisce la 'Z' finale come UTC
    if iso_str.endswith("Z"):
        dt = datetime.fromisoformat(iso_str.replace("Z", "+00:00"))
    else:
        dt = datetime.fromisoformat(iso_str)
        if dt.tzinfo is None:
            # Se manca timezone, assumiamo UTC
            dt = dt.replace(tzinfo=timezone.utc)

    return dt.timestamp()


def preprocess_telemetry():
    script_dir = Path(__file__).parent
    input_file = script_dir / INPUT_FILE
    if not input_file.exists():
        raise FileNotFoundError(f"File non trovato: {input_file}")

    # Calcoliamo il timestamp minimo a partire dalla costante
    min_time_unix = _iso_to_unix(MIN_DATE_ISO)
    print(f"Filtraggio eventi con time >= {MIN_DATE_ISO} (unix: {min_time_unix})")

    # Output file nella cartella processed
    processed_dir = script_dir / "processed"
    processed_dir.mkdir(exist_ok=True)
    output_file = processed_dir / OUTPUT_FILE

    processed = []
    scartati_senza_time = 0
    scartati_per_data = 0
    eventi_totali = 0

    # Il file ora è in formato JSON Lines: un oggetto JSON per riga
    with input_file.open("r", encoding="utf-8") as f:
        for line in f:
            line = line.strip()
            if not line:
                continue

            try:
                ev = json.loads(line)
            except json.JSONDecodeError:
                # Salta eventuali righe malformate
                continue

            eventi_totali += 1
            props = ev.get("properties", {})

            client_time = props.get("time")

            # Se non c'è il campo time, lo scartiamo (non possiamo confrontare la data)
            if client_time is None:
                scartati_senza_time += 1
                continue

            # Escludi gli eventi con time precedente alla data minima
            if client_time < min_time_unix:
                scartati_per_data += 1
                continue

            # Converti timestamp a data leggibile
            test_date = datetime.fromtimestamp(client_time, tz=timezone.utc).strftime("%Y-%m-%d")
            
            # Costruiamo un nuovo oggetto ridotto
            minimal_event = {
                "event": ev.get("event"),
                "distinct_id": props.get("distinct_id"),
                "user_id": props.get("$user_id"),
                "insert_id": props.get("$insert_id"),

                # Timestamp lato client (epoch)
                "time": client_time,
                
                # Data del test (YYYY-MM-DD)
                "test_date": test_date,

                # Timestamp interno del tuo gioco
                "timestamp_game": props.get("timestamp"),

                # Action info
                "actionName": props.get("actionName"),
                "actionType": props.get("actionType"),

                # Extra opzionali
                "value": props.get("value"),
                "eta": props.get("eta"),
            }

            processed.append(minimal_event)

    # Salva il nuovo file
    with output_file.open("w", encoding="utf-8") as f:
        json.dump(processed, f, indent=2)

    print(f"Preprocess completato. Salvato in: {output_file}")
    print(f"Eventi totali in input (righe valide): {eventi_totali}")
    print(f"Eventi mantenuti: {len(processed)}")
    print(f"Eventi scartati (senza time): {scartati_senza_time}")
    print(f"Eventi scartati (prima di {MIN_DATE_ISO}): {scartati_per_data}")

    return output_file


if __name__ == "__main__":
    preprocess_telemetry()