@kwdef mutable struct Prosumer <: AbstractAgent id::Int end

function make_model(;
    case_name,
    n_prosumers,
    seed,
    regulation,
    start_price,
    perc_regulated,
    kwargs...
)
    net = net_load(case_name)
    rng = Xoshiro(seed)
    is_regulated = sample(rng, [true, false], Weights([perc_regulated, 1 - perc_regulated]), n_prosumers)
    is_unregulated = .!is_regulated
    properties = Dict(
        :net=>net,
        :violation_sum=>0.,
        :n_blackouts=>0,
        :n_prosumers=>n_prosumers,
        :is_regulated=>is_regulated,
        :is_unregulated=>is_unregulated,
        :regulated=>findall(identity, is_regulated),
        :unregulated=>findall(identity, is_unregulated),
        :buses=>rand(rng, net[:prosumer_buses], n_prosumers),
        :excess=>zeros(n_prosumers),
        :net_excess=>zeros(n_prosumers),
        :abs_net_excess=>zeros(n_prosumers),
        :injection=>zeros(net[:n_buses]),
        :is_seller=>[false for _ in 1:n_prosumers],
        :is_buyer=>[false for _ in 1:n_prosumers],
        :sell_price=>ones(n_prosumers) .* start_price,
        :buy_price=>ones(n_prosumers) .* start_price,
        :regulation=>regulation,
        :exchanges=>[],
        :capital_reg=>0.,
        :capital_unreg=>0.,
        :capital_diff=>0.,
        :prosumer_violation=>zeros(n_prosumers),
        :prosumer_price_sell=>zeros(n_prosumers),
        :prosumer_power_sell=>zeros(n_prosumers),
        :prosumer_price_buy=>zeros(n_prosumers),
        :prosumer_power_buy=>zeros(n_prosumers),
        kwargs...
    )
    model = StandardABM(
        Prosumer;
        rng,
        model_step!,
        properties
    )
    model
end

function model_step!(model)
    met = (model.excess .- model.net_excess) ./ model.excess
    change = abs.(randn(abmrng(model), model.n_prosumers)) .* model.price_change
    change .*= (met .- model.power_met_thres) ./ (1 - model.power_met_thres)
    model.sell_price[model.is_seller] .+= change[model.is_seller]
    model.buy_price[model.is_buyer] .-= change[model.is_buyer]

    model.injection .= 0.
    model.excess = randn(abmrng(model), model.n_prosumers) .* model.power_std .+ model.power_offset
    model.net_excess = copy(model.excess)
    model.abs_net_excess = abs.(model.net_excess)
    model.is_seller = model.excess .> 0.
    model.is_buyer = model.excess .< 0.

    model.violation_sum = 0.
    model.n_blackouts = 0
    model.prosumer_violation = zeros(model.n_prosumers)
    model.exchanges = []

    market_clearing!(model)

    reg_sell = price_reg_sell(model)
    reg_buy = price_reg_buy(model)
    reg_profit = reg_sell - reg_buy
    unreg_sell = price_unreg_sell(model)
    unreg_buy = price_unreg_buy(model)
    unreg_profit = unreg_sell - unreg_buy
    model.capital_reg += reg_profit
    model.capital_unreg += unreg_profit
    model.capital_diff = model.capital_reg - model.capital_unreg
end

function push_exchange!(model, exchange)
    model.prosumer_price_sell[exchange[:seller]] += exchange[:price]
    model.prosumer_power_sell[exchange[:seller]] += exchange[:power]
    model.prosumer_price_buy[exchange[:buyer]] += exchange[:price]
    model.prosumer_power_buy[exchange[:buyer]] += exchange[:power]
    model.injection[model.buses[exchange[:seller]]] += exchange[:power]
    model.injection[model.buses[exchange[:buyer]]] -= exchange[:power]
    model.net_excess[exchange[:seller]] -= exchange[:power]
    model.net_excess[exchange[:buyer]] += exchange[:power]
    model.abs_net_excess[exchange[:seller]] -= exchange[:power]
    model.abs_net_excess[exchange[:buyer]] -= exchange[:power]
    if exchange[:violation_sum] == Inf
        model.n_blackouts += 1
    end
    model.violation_sum = exchange[:violation_sum]
    model.prosumer_violation[[exchange[:seller], exchange[:buyer]]] .+= exchange[:violation_sum]
    push!(model.exchanges, exchange)
end
