function mean_price(model, filter)
    price = 0.
    power = 0.
    for exc in model.exchanges
        if filter(exc)
            price += exc[:price]
            power += exc[:power]
        end
    end
    if power > 0.
        price / power
    else 0. end
end
function price(model)
    mean_price(model, _->true) end
function price_reg(model)
    mean_price(model, exc->(model.is_regulated[exc[:seller]] || model.is_regulated[exc[:buyer]])) end
function price_unreg(model)
    mean_price(model, exc->(model.is_unregulated[exc[:seller]] || model.is_unregulated[exc[:buyer]])) end
function price_reg_sell(model)
    mean_price(model, exc->model.is_regulated[exc[:seller]]) end
function price_reg_buy(model)
    mean_price(model, exc->model.is_regulated[exc[:buyer]]) end
function price_unreg_sell(model)
    mean_price(model, exc->model.is_unregulated[exc[:seller]]) end
function price_unreg_buy(model)
    mean_price(model, exc->model.is_unregulated[exc[:buyer]]) end

function price_reg_intra(model)
    mean_price(model, exc->(model.is_regulated[exc[:seller]] && model.is_regulated[exc[:buyer]])) end
function price_unreg_intra(model)
    mean_price(model, exc->(model.is_unregulated[exc[:seller]] && model.is_unregulated[exc[:buyer]])) end
function price_inter(model)
    mean_price(model, exc->(model.is_regulated[exc[:seller]] ⊻ model.is_regulated[exc[:buyer]])) end

function price_reg_sell_unreg(model)
    mean_price(model, exc->(model.is_regulated[exc[:seller]] && model.is_unregulated[exc[:buyer]])) end
function price_unreg_sell_reg(model)
    mean_price(model, exc->(model.is_unregulated[exc[:seller]] && model.is_regulated[exc[:buyer]])) end

function price_flow(model)
    to_reg = 0.
    power = 0.
    for exc in model.exchanges
        if (exc[:seller] in model.regulated) && (exc[:buyer] in model.unregulated)
            to_reg += exc[:price]
            power += exc[:power]
        elseif (exc[:buyer] in model.regulated) && (exc[:seller] in model.unregulated)
            to_reg -= exc[:price]
            power -= exc[:power]
        end
    end
    if power > 0.
        to_reg / power
    else 0. end
end

function power_met(model, mask)
    mean((model.excess[mask] .- model.net_excess[mask]) ./ model.excess[mask]) end
function power_met_reg(model)
    power_met(model, model.regulated) end
function power_met_unreg(model)
    power_met(model, model.unregulated) end

function capital(model, filter)
    capital = 0.
    for exc in model.exchanges
        if filter(exc[:buyer]) capital -= exc[:price] end
        if filter(exc[:seller]) capital += exc[:price] end
    end
    capital
end
function capital_reg(model)
    capital(model, p->in(p, model.regulated)) end
function capital_unreg(model)
    capital(model, p->in(p, model.unregulated)) end

function violation(model, mask)
    mean(model.prosumer_violation[mask]) end
function violation_all(model)
    violation(model, 1:model.n_prosumers) end
function violation_reg(model)
    violation(model, model.regulated) end
function violation_unreg(model)
    violation(model, model.unregulated) end

function sell_price(model, mask)
    mean(model.sell_price[mask]) end
function reg_sell_price(model)
    sell_price(model, model.regulated) end
function unreg_sell_price(model)
    sell_price(model, model.unregulated) end

function buy_price(model, mask)
    mean(model.buy_price[mask]) end
function reg_buy_price(model)
    buy_price(model, model.regulated) end
function unreg_buy_price(model)
    buy_price(model, model.unregulated) end

function profit_diff(model)
    sell = model.prosumer_price_sell ./ model.prosumer_power_sell
    buy = model.prosumer_price_buy ./ model.prosumer_power_buy
    profit_reg = (sell[model.regulated] .- buy[model.regulated]) .* abmtime(model)
    profit_unreg = (sell[model.unregulated] .- buy[model.unregulated]) .* abmtime(model)
    mean(profit_reg) - mean(profit_unreg)
end

function simple_profit(model)
    profit = model.prosumer_price_sell .- model.prosumer_price_buy
    mean(profit[model.regulated]) - mean(profit[model.unregulated])
end

function run_single(properties; steps, regulation=true, seed)
    properties = Dict(
        Symbol(key)=>value
        for (key, value) in pairs(properties)
    )

    model = make_model(; regulation, seed, properties...)
    _, df_model = run!(model, steps; mdata=[
        price, price_reg, price_unreg, price_reg_sell, price_reg_buy, price_unreg_sell, price_unreg_buy,
        price_reg_intra, price_unreg_intra, price_inter,
        price_reg_sell_unreg, price_unreg_sell_reg,
        reg_buy_price, reg_sell_price, unreg_buy_price, unreg_sell_price,
        power_met_reg, power_met_unreg,
        :capital_reg, :capital_unreg, :capital_diff,
        violation_reg, violation_unreg, :violation_sum,
        profit_diff, simple_profit,
    ], init=false)
    df_model
end

function run_experiment(; name, save_path, steps, runs, experiment, regulation=true)
    exp = JSON.parsefile("experiments/$experiment.json")
    @info "Running Experiment $name ($runs times, $steps steps)"
    seeds = rand(1:99999, runs)
    results = @showprogress pmap(seed->run_single(exp; steps, regulation, seed), seeds)

    @info "Collecting Data $name"

    timestamp = Dates.format(now(), "dd-mm-yyyy HH:MM:SS")

    description = Dict(
        "name"=>name,
        "steps"=>steps,
        "runs"=>runs,
        "timestamp"=>timestamp,
        "seeds"=>seeds,
        exp...
    )

    mean_df = DataFrame(Dict(col=>[] for col in names(results[1]))...)
    for i in 1:steps
        push!(mean_df, Dict(
            col=>mean([
                result[i, col]
                for result in results
            ])
            for col in names(results[1])
        ))
    end
    std_df = DataFrame(Dict(col=>[] for col in names(results[1]))...)
    for i in 1:steps
        push!(std_df, Dict(
            col=>std([
                result[i, col]
                for result in results
            ])
            for col in names(results[1])
        ))
    end

    @info "Saving Experiment $name at $timestamp"
    full_save_path = "$save_path/$(name)"
    if isdir(full_save_path) rm(full_save_path, recursive=true) end
    mkdir(full_save_path)

    open("$full_save_path/description.json", "w") do f
        JSON.print(f, description)
    end
    CSV.write("$full_save_path/data.csv", mean_df)
    CSV.write("$full_save_path/std.csv", std_df)

    nothing
end

function run_single_old(properties; steps, regulation=true, seed)
    properties = Dict(
        Symbol(key)=>value
        for (key, value) in pairs(properties)
    )

    model = make_model(; regulation, seed, properties...)
    _, df_model = run!(model, steps; mdata=[:exchanges], init=false)
    df_model
end

function run_experiment_old(; name, save_path, steps, runs, experiment, regulation=true)
    @info "Running Experiment $name ($runs times, $steps steps)"
    exp = JSON.parsefile("experiments/$experiment.json")
    seeds = rand(1:99999, runs)
    results = @showprogress pmap(seed->run_single(exp; steps, regulation, seed), seeds)

    timestamp = Dates.format(now(), "dd-mm-yyyy HH:MM:SS")

    @info "Saving Experiment $name at $timestamp"
    full_save_path = "$save_path/$name"
    if isdir(full_save_path) rm(full_save_path, recursive=true) end
    mkdir(full_save_path)
    for (seed, result) in zip(seeds, results)
        data = DataFrame(Iterators.flatten(result[!, :exchanges]))
        CSV.write("$full_save_path/$seed.csv", data)
    end

    description = Dict(
        "name"=>name,
        "steps"=>steps,
        "runs"=>runs,
        "timestamp"=>timestamp,
        "seeds"=>seeds,
        exp...
    )
    open("$full_save_path/description.json", "w") do f
        JSON.print(f, description)
    end

    nothing
end
