import os
import numpy as np
import cv2
import seaborn as sns
import matplotlib.pyplot as plt


def reconstruct_contour(fd):
    fd = np.asarray(fd, dtype=np.complex128)
    contour = np.fft.ifft(fd)
    contour = np.stack((contour.real, contour.imag), axis=-1)
    contour = np.round(contour).astype(np.float32)
    return contour

folder = os.path.dirname(os.getcwd())
ste_fd_path = os.path.join(folder, 'Output_fd', 'fourier_descriptors_steatosis.npy')
hpt_fd_path = os.path.join(folder, 'Output_fd', 'fourier_descriptors_hepatocytenuclei.npy')
oth_fd_path = os.path.join(folder, 'Output_fd', 'fourier_descriptors_othernuclei.npy')
fat_fd_path = os.path.join(folder, 'Output_fd', 'fourier_descriptors_fat.npy')


ste_areas = []
hpt_areas = []
oth_areas = []
fat_areas = []


ste_fd_array = np.load(ste_fd_path, allow_pickle=True)
hpt_fd_array = np.load(hpt_fd_path, allow_pickle=True)
oth_fd_array = np.load(oth_fd_path, allow_pickle=True)
fat_fd_array = np.load(fat_fd_path, allow_pickle=True)


print("--- Processing Steatosis---")
for ste_fd in ste_fd_array:
    ste_contour = reconstruct_contour(ste_fd)
    ste_contour = np.vstack([ste_contour, ste_contour[0]])
    ste_area = cv2.contourArea(ste_contour.astype(np.float32))
    if ste_area == 0:
        continue
    ste_areas.append(ste_area)

print("--- Processing Hepatocyte Nuclei ---")
for hpt_fd in hpt_fd_array:
    hpt_contour = reconstruct_contour(hpt_fd)
    hpt_contour = np.vstack([hpt_contour, hpt_contour[0]])
    hpt_area = cv2.contourArea(hpt_contour.astype(np.float32))
    if hpt_area == 0:
        continue
    hpt_areas.append(hpt_area)

print("--- Processing Other Nuclei ---")
for oth_fd in oth_fd_array:
    oth_contour = reconstruct_contour(oth_fd)
    oth_contour = np.vstack([oth_contour, oth_contour[0]])
    oth_area = cv2.contourArea(oth_contour.astype(np.float32))
    if oth_area == 0:
        continue
    oth_areas.append(oth_area)

print("--- Processing Fat ---")
for fat_fd in fat_fd_array:
    fat_contour = reconstruct_contour(fat_fd)
    fat_contour = np.vstack([fat_contour, fat_contour[0]])
    fat_area = cv2.contourArea(fat_contour.astype(np.float32))
    if fat_area == 0:
        continue
    fat_areas.append(fat_area)

hpt_5th = np.percentile(hpt_areas, 5)
ste_5th = np.percentile(ste_areas, 5)
oth_5th = np.percentile(oth_areas, 5)
fat_5th = np.percentile(fat_areas, 5)


print(f"steatosis, 5° percentile: {ste_5th}")
print(f"hepatocyte_nuclei, 5° percentile: {hpt_5th}")
print(f"other_nuclei, 5° percentile: {oth_5th}")
print(f"fat, 5° percentile: {fat_5th}")



## FOCUS ON STEATOSIS DISTRIBUTION
plt.figure(figsize=(9,6))
sns.histplot(ste_areas, bins=40,kde=True,color="lightblue", edgecolor='black')
plt.title("STEATOSIS Area Distributions", fontsize=14)
plt.xlabel("Areas", fontsize=12)
plt.xlim(0,4000)
plt.ylabel("Number of cells", fontsize=12)
# plt.legend(title="Bands", loc='upper right')
plt.grid(True, linestyle='--', alpha=0.5)
plt.tight_layout()
plt.show()
min_thresh = 10
max_thresh = 4000
below_min = sum(a < min_thresh for a in ste_areas)
above_max = sum(a > max_thresh for a in ste_areas)
print("\n [STEATOSIS]:")
print(f"Numero di cellule con area < {min_thresh}: {below_min}; ({below_min/len(ste_areas)*100:.2f}%)")
print(f"Numero di cellule con area > {max_thresh}: {above_max}; ({above_max/len(ste_areas)*100:.2f}%)")
print(f"Totale cellule analizzate: {len(ste_areas)}")


## FILTERING SMALL STEATOSIS FROM NPY
input_file = os.path.join(folder, "Output_Fd/fourier_descriptors_steatosis.npy")
output_file = os.path.join(folder, "Output_Fd/fourier_descriptors_steatosisfilt.npy")
coeffs = np.load(input_file, allow_pickle=True)
small_area = 10
to_cut = 1500
areas = []
for fd in coeffs:
    contour = reconstruct_contour(fd)
    contour = np.vstack([contour, contour[0]])
    area = cv2.contourArea(contour.astype(np.float32))
    areas.append(area)
areas = np.array(areas)
low_area_indices = np.where(areas < small_area)[0]
exclude_indices = low_area_indices[:to_cut]

mask = np.ones(len(coeffs), dtype=bool)
mask[exclude_indices] = False
filtered_coeffs = coeffs[mask]
np.save(output_file, filtered_coeffs)
print(f"Total number of contours to be excluded: {len(low_area_indices)}/{len(coeffs)}")
print(f"Excluded contours: {len(exclude_indices)}")
print(f"Remaining contours ({len(filtered_coeffs)}) saved in: {output_file}")