function [y, state_duration_vec, MA_vec, SigmaA_vec, MP_vec, L_trans_vec, average_power] = coef_time_series_long_pedestrian(f_c, envi, elev_deg, Lch)

% Generates a long sequence of channel samples from the land mobile
% satellite channel, according to the model described in ITU-R P.681-10,
% Sect. 6. 

% INPUTS
% f_c:      carrier frequency [Hz]
% envi:     environment ('Urban', 'Suburban', 'Village', 'Rural wooded',
%   'Residential'
% elev_deg: quantized elevation [deg] (20, 30, 45, 60 or 70)

% OUTPUTS
% y: row vector containining the complex channel coefficients, sampled
%   every Ls metres (see below) for a total plath length of Lmax metres (see below)
% state_duration_vec: row vector of the sequence of state durations [m]
% MA_vec: row vector of the sequence of values of Loo parameter M_A
% SigmaA_vec: row vector of the sequence of values of Loo parameter Sigma_A
% MP_vec: row vector of the sequence of values of Loo parameter MP
% L_trans_vec: row vector of the sequence of transition lengths [m]

if(f_c >= 1.5e9 && f_c <= 3e9)
    switch envi
        case 'Urban'
            if(elev_deg == 20)
                mu = [2.0042, 3.689]; sigma = [1.2049, 0.9796]; 
                dur_min = [3.9889, 10.3114]; 
                mu_MA = [-3.3681, -18.1771]; sigma_MA = [3.3226, 3.2672];  
                h1 = [0.1739, 1.1411]; h2 = [-11.5966, 4.0581]; 
                g1 = [0.0036, -0.2502]; g2 = [1.3230, -1.2528]; 
                Lcorr = [0.9680, 0.9680]; 
                f1 = 0.0870; f2 = 2.8469;
                pBmin = 0.1; pBmax = 0.9;
            elseif(elev_deg == 30)
                mu = [2.7332, 2.7582]; sigma = [1.1030, 1.2210]; 
                dur_min = [7.3174, 5.7276]; 
                mu_MA = [-2.3773, -17.4276]; sigma_MA = [2.1222, 3.9532];  
                h1 = [0.0941, 0.9175]; h2 = [-13.1679, -0.8009]; 
                g1 = [-0.2811, -0.1484]; g2 = [0.9323, 0.5910]; 
                Lcorr = [1.4731, 1.4731]; 
                f1 = 0.1378; f2 = 3.3733;
                pBmin = 0.1; pBmax = 0.9;
            elseif(elev_deg == 45)
                mu = [3.0639, 2.9108]; 
                sigma = [1.6980, 1.2602]; 
                dur_min = [10.0, 6.0]; 
                mu_MA = [-1.8225, -15.4844]; 
                sigma_MA = [1.1317, 3.3245];  
                h1 = [-0.0481, 0.9434]; 
                h2 = [-14.7450, -1.7555]; 
                g1 = [-0.4643, -0.0798]; 
                g2 = [0.3334, 2.8101]; 
                Lcorr = [1.7910, 1.7910]; 
                f1 = 0.0744; 
                f2 = 2.1423; 
                pBmin = 0.1; 
                pBmax = 0.9;
            elseif(elev_deg == 60)
                mu = [2.8135, 2.0211]; 
                sigma = [1.5962,  0.6568]; 
                dur_min = [10.0, 1.9126]; 
                mu_MA = [-1.5872, -14.1435]; 
                sigma_MA = [1.2446, 3.2706 ];  
                h1 = [-0.5168, 0.6975]; 
                h2 = [-17.4060, -7.5383]; 
                g1 = [-0.1953, 0.0422]; 
                g2 = [0.5353, 3.2030]; 
                Lcorr = [1.7977, 1.7977]; 
                f1 = -0.1285; 
                f2 = 5.4991; 
                pBmin = 0.1; 
                pBmax = 0.9;
            elseif(elev_deg == 70)
                mu = [4.2919, 2.1012]; 
                sigma = [2.4703, 1.0341]; 
                dur_min = [118.3312, 4.8569]; 
                mu_MA = [-1.8434, -12.9383]; 
                sigma_MA = [0.5370, 1.7588];  
                h1 = [-4.7301, 2.5318]; 
                h2 = [-26.5687, 16.8468]; 
                g1 = [0.5192, 0.3768]; 
                g2 = [1.9583, 8.4377]; 
                Lcorr = [2.0963, 2.0963]; 
                f1 = -0.0826; 
                f2 = 2.8824; 
                pBmin = 0.1; 
                pBmax = 0.9;
            else
                error('Elevation not supported!')
            end
        case 'Suburban'
            if(elev_deg == 20)
                mu = [2.2201 2.2657];
                sigma = [1.2767 1.3812];
                dur_min = [2.2914 2.5585];
                mu_MA = [-2.7191 -13.8808];
                sigma_MA = [1.3840 2.5830];
                h1 = [-0.3037 1.0136];
                h2 = [-13.0719 0.5158];
                g1 = [-0.1254 -0.1441];
                g2 = [0.7894 0.7757];
                Lcorr = [0.9290 0.9290];
                f1 = 0.2904;
                f2 = 1.0324;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 30)
                mu = [3.0138, 2.4521]; sigma = [1.4161, 0.7637];
                dur_min = [8.3214, 5.9087];
                mu_MA = [-0.7018, -11.9823]; sigma_MA = [1.2107, 3.4728]; 
                h1 = [-0.6543, 0.6200]; h2 = [-14.6457, -7.5485];
                g1 = [-0.1333, -0.1644]; g2 = [0.8992, 0.2762];
                Lcorr = [1.7135, 1.7135];
                f1 = 0.1091; f2 = 3.3000;
                pBmin = 0.1; pBmax = 0.9;
            elseif(elev_deg == 45)
                mu = [4.5857, 2.2414]; sigma = [1.3918, 0.7884];
                dur_min = [126.8375, 4.3132];
                mu_MA = [-1.1496, -10.3806]; sigma_MA = [1.0369, 2.3543]; 
                h1 = [0.2148, 0.0344]; h2 = [-17.8462, -14.2087];
                g1 = [0.0729, 0.0662]; g2 = [1.0303, 3.5043];
                Lcorr = [3.2293, 3.2293];
                f1 = 0.5766; f2 = 0.7163;
                pBmin = 0.1; pBmax = 0.9;
            elseif(elev_deg == 60)                
                mu = [3.4124, 1.9922];
                sigma = [1.4331, 0.7132];
                dur_min = [19.5431, 3.1213];
                mu_MA = [-0.7811, -12.1436];
                sigma_MA = [0.7979, 3.1798];   
                h1 = [-2.1102, 0.4372];
                h2 = [-19.7954, -8.3651];      
                g1 = [-0.2284, -0.2903];
                g2 = [0.2796, -0.6001];        
                Lcorr = [2.0215, 2.0215];      
                f1 = -0.4097;
                f2 = 8.7440;                   
                pBmin = 0.1;                                  
                pBmax = 0.9;
            elseif(elev_deg == 70)
                mu = [4.2919, 2.1012];
                sigma = [2.4703, 1.0341];
                dur_min = [118.3312, 4.8569];
                mu_MA = [-1.8434, -12.9383];
                sigma_MA = [0.5370, 1.7588];
                h1 = [-4.7301, 2.5318];
                h2 = [-26.5687, 16.8468];
                g1 = [0.5192, 0.3768];
                g2 = [1.9583, 8.4377];
                Lcorr = [2.0963, 2.0963];
                f1 = -0.0826;
                f2 = 2.8824;
                pBmin = 0.1;
                pBmax = 0.9;
            else
                error('Elevation not supported!')
            end
        case 'Village'
            if(elev_deg == 20)
                mu = [2.7663 2.2328];
                sigma = [1.1211 1.3788];
                dur_min = [6.5373 2.8174];
                mu_MA = [-2.5017 -15.2300];
                sigma_MA = [2.3059 5.0919];
                h1 = [0.0238 0.9971];
                h2 = [-11.4824 0.8970];
                g1 = [-0.2735 -0.0568];
                g2 = [1.3898 1.9253];
                Lcorr = [0.8574 0.8574];
                f1 = 0.0644;
                f2 = 2.6740;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 30)
                mu = [2.4246 1.8980];
                sigma = [1.3025 1.0505];
                dur_min = [5.4326 2.4696];
                h1 = [-2.2284 -15.1583];
                h2 = [1.4984 4.0987];
                g1 = [-0.3431 0.9614];
                g2 = [-14.0798 0.3719];
                mu_MA = [-0.2215 -0.0961];
                sigma_MA = [1.0077 1.3123];
                Lcorr = [0.8264 0.8264];
                f1 = -0.0576;
                f2 = 3.3977;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 45)
                mu = [2.8402 1.8509];
                sigma = [1.4563 0.8736];
                dur_min = [10.4906 2.6515];
                mu_MA = [-1.2871 -12.6718];
                sigma_MA = [0.6346 3.1722];
                h1 = [-0.0222 0.8329];
                h2 = [-16.7316 -3.9947];
                g1 = [-0.3905 -0.0980];
                g2 = [0.4880 1.3381];
                Lcorr = [1.4256 1.4256];
                f1 = -0.0493;
                f2 = 5.3952;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 60)
                mu = [3.7630 1.7192];
                sigma = [1.2854 1.1420];
                dur_min = [17.6726 2.5981];
                mu_MA = [-0.5364 -9.5399];
                sigma_MA = [0.6115 2.0732];
                h1 = [-0.1418 -0.4454];
                h2 = [-17.8032 -16.8201];
                g1 = [-0.2120 0.0609];
                g2 = [0.7819 2.5925];
                Lcorr = [0.8830 0.8830];
                f1 = -0.8818;
                f2 = 10.1610;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 70)
                mu = [4.0717 1.5673];
                sigma = [1.2475 0.5948];
                dur_min = [30.8829 2.1609];
                mu_MA = [-0.3340 -8.3686];
                sigma_MA = [0.6279 2.5603];
                h1 = [-1.6253 0.1788];
                h2 = [-19.7558 -9.5153];
                g1 = [-0.4438 -0.0779];
                g2 = [0.6355 1.1209];
                Lcorr = [1.5633 1.5633];
                f1 = -0.3483;
                f2 = 5.1244;
                pBmin = 0.1;
                pBmax = 0.9;
            else
                error('Elevation not supported!')
            end
        case 'Rural wooded'
            if(elev_deg == 20)
                mu = [2.1597 1.9587];
                sigma = [1.3766 1.5465];
                dur_min = [2.0744 1.3934];
                mu_MA = [-0.8065 -10.6615];
                sigma_MA = [1.5635 2.6170];
                h1 = [-0.9170 0.8440];
                h2 = [-12.1228 -1.4804];
                g1 = [-0.0348 -0.1069];
                g2 = [0.9571 1.6141];
                Lcorr = [0.8845 0.8845];
                f1 = 0.0550;
                f2 = 2.6383;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 30)
                mu = [2.5579 2.3791];
                sigma = [1.2444 1.1778];
                dur_min = [3.5947 2.2800];
                mu_MA = [-1.3214 -10.4240];
                sigma_MA = [1.6645 2.4446];
                h1 = [-1.0445 0.6278];
                h2 = [-14.3176 -4.8146];
                g1 = [-0.1656 -0.0451];
                g2 = [0.7180 2.2327];
                Lcorr = [1.0942 1.0942];
                f1 = 0.0256;
                f2 = 3.8527;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 45)
                mu = [3.1803 2.5382];
                sigma = [1.3427 1.1291];
                dur_min = [6.7673 3.3683];
                mu_MA = [-0.9902 -10.2891];
                sigma_MA = [1.0348 2.3090];
                h1 = [-0.4235 0.3386];
                h2 = [-16.8380 -9.7118];
                g1 = [-0.1095 -0.0460];
                g2 = [0.6893 2.1310];
                Lcorr = [2.3956 2.3956];
                f1 = 0.2803;
                f2 = 4.0004;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 60)
                mu = [2.9322, 2.1955]; sigma = [1.3234, 1.1115];
                dur_min = [5.7209, 1.6512];
                mu_MA = [-0.6153, -9.9595]; sigma_MA = [1.1723, 2.2188]; 
                h1 = [-1.4024, 0.2666]; h2 = [-16.9664, -9.0046];
                g1 = [-0.2516, -0.0907]; g2 = [0.5353, 1.4730];
                Lcorr = [1.7586, 1.7586];
                f1 = 0.1099; f2 = 4.2183;
                pBmin = 0.1; pBmax = 0.9;
            elseif(elev_deg == 70)
                mu = [3.8768 1.8445];
                sigma = [1.4738 0.8874];
                dur_min = [16.0855 2.9629];
                mu_MA = [-0.7818 -6.7769];
                sigma_MA = [0.7044 2.1339];
                h1 = [-2.9566 -0.3723];
                h2 = [-20.0326 -14.9638];
                g1 = [-0.2874 -0.1822];
                g2 = [0.4050 0.1163];
                Lcorr = [1.6546 1.6546];
                f1 = -0.3914;
                f2 = 6.6931;
                pBmin = 0.1;
                pBmax = 0.9;
            else
                error('Elevation not supported!')
            end
        case 'Residential'
            if(elev_deg == 20)
                mu = [2.5818 1.7136];
                sigma = [1.7310 1.1421];
                dur_min = [9.2291 1.6385];
                mu_MA = [-0.8449 -10.8315];
                sigma_MA = [1.3050 2.2642];
                h1 = [-0.3977 0.8589];
                h2 = [-12.3714 -2.4054];
                g1 = [0.0984 -0.1804];
                g2 = [1.3138 0.8553];
                Lcorr = [1.1578 1.1578];
                f1 = 0.0994;
                f2 = 2.4200;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 30)
                mu = [3.2810 1.8414];
                sigma = [1.4200 0.9697];
                dur_min = [14.4825 2.7681];
                mu_MA = [-1.3799 -11.1669];
                sigma_MA = [1.0010 2.4724];
                h1 = [-0.8893 -0.1030];
                h2 = [-16.4615 -13.7102];
                g1 = [-0.2432 -0.1025];
                g2 = [0.6519 1.7671];
                Lcorr = [1.9053 1.9053];
                f1 = 0.0196;
                f2 = 3.9374;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 60)
                mu = [3.255 3.277];
                sigma = [1.287 1.260];
                dur_min = [6.47 7.81];
                mu_MA = [0 -2.32];
                sigma_MA = [0.30 2.06];
                h1 = [-2.024 -1.496];
                h2 = [-19.454 -22.894];
                g1 = [0.273 -0.361];
                g2 = [0.403 -0.119];
                Lcorr = [3.84 3.84];
                f1 = -1.591;
                f2 = 12.274;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 70)
                mu = [4.3291 3.4534];
                sigma = [0.7249 0.9763];
                dur_min = [27.3637 8.9481];
                mu_MA = [-0.1625 -1.6084];
                sigma_MA = [0.3249 0.5817];
                h1 = [0.6321 -0.3976];
                h2 = [-21.5594 -22.7905];
                g1 = [0.1764 -0.0796];
                g2 = [0.4135 0.1939];
                Lcorr = [1.6854 1.6854];
                f1 = 3.0127;
                f2 = 6.2345;
                pBmin = 0.1;
                pBmax = 0.9;
            else
                error('Elevation not supported for this environment!')
            end
        otherwise
            error('Environment not supported!')
    end
elseif(f_c > 3e9 && f_c <= 5e9)
    switch envi
        case 'Urban'
            if(elev_deg == 20)
                mu = [2.5467 3.6890];
                sigma = [1.0431 0.9796];
                dur_min = [5.2610 10.3114];
                mu_MA = [-2.7844 -19.4022];
                sigma_MA = [2.6841 3.2428];
                h1 = [0.1757 0.9638];
                h2 = [-12.9417 -0.9382];
                g1 = [-0.2044 0.0537];
                g2 = [1.5866 4.5670];
                Lcorr = [1.4243 1.4243];
                f1 = 0.1073;
                f2 = 1.9199;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 30)
                mu = [2.0158 2.2627];
                sigma = [1.2348 1.4901];
                dur_min = [4.5491 2.0749];
                mu_MA = [-3.7749 -17.9098];
                sigma_MA = [2.2381 2.9828];
                h1 = [-0.1564 0.8250];
                h2 = [-15.1531 -2.5833];
                g1 = [-0.0343 -0.0741];
                g2 = [1.0602 2.1406];
                Lcorr = [0.8999 0.8999];
                f1 = 0.2707;
                f2 = -0.0287;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 45)
                mu = [2.3005 2.6314];
                sigma = [1.6960 1.1210];
                dur_min = [10.0 6.0];
                mu_MA = [-1.4466 -15.3926];
                sigma_MA = [1.1472 3.2527];
                h1 = [0.1550 0.9509];
                h2 = [-13.6861 -1.2462];
                g1 = [0.1666 0.0363];
                g2 = [1.2558 4.4356];
                Lcorr = [1.6424 1.6424];
                f1 = 0.2517;
                f2 = -0.3512;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 60)
                mu = [2.4546 1.8892];
                sigma = [1.9595 0.8982];
                dur_min = [10.0 1.9126];
                mu_MA = [-1.6655 -14.4922];
                sigma_MA = [0.8244 3.4941];
                h1 = [-0.4887 0.4501];
                h2 = [-17.2505 -9.6935];
                g1 = [-0.3373 0.1202];
                g2 = [0.3285 4.8329];
                Lcorr = [2.3036 2.3036];
                f1 = 0.0025;
                f2 = 1.4949;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 70)
                mu = [2.8354 1.5170];
                sigma = [2.4631 1.1057];
                dur_min = [67.5721 3.6673];
                mu_MA = [-1.0455 -14.2294];
                sigma_MA = [0.2934 5.4444];
                h1 = [-3.0973 0.0908];
                h2 = [-20.7862 -15.8022];
                g1 = [0.0808 0.0065];
                g2 = [0.8952 3.1520];
                Lcorr = [2.2062 2.2062];
                f1 = 0.0755;
                f2 = 2.1426;
                pBmin = 0.1;
                pBmax = 0.9;
            else
                error('Elevation not supported!')
            end
        case 'Suburban'
            if(elev_deg == 20)
                mu = [2.8194 2.5873];
                sigma = [1.6507 1.3919];
                dur_min = [11.1083 4.4393];
                mu_MA = [-4.8136 -17.0970];
                sigma_MA = [1.9133 2.9350];
                h1 = [-0.4500 0.8991];
                h2 = [-17.9227 -2.4082];
                g1 = [-0.1763 0.0582];
                g2 = [0.8244 4.0347];
                Lcorr = [1.2571 1.2571];
                f1 = 0.0727;
                f2 = 2.8177;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 30)
                mu = [2.9226 2.7375];
                sigma = [1.3840 0.6890];
                dur_min = [6.7899 7.7356];
                mu_MA = [-1.9611 -15.3022];
                sigma_MA = [1.8460 2.9379];
                h1 = [0.2329 0.5146];
                h2 = [-15.0063 -8.9987];
                g1 = [0.0334 0.0880];
                g2 = [1.3323 4.4692];
                Lcorr = [1.6156 1.6156];
                f1 = 0.1281;
                f2 = 2.3949;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 45)
                mu = [4.3019 2.3715];
                sigma = [0.8530 1.3435];
                dur_min = [36.1277 9.5511];
                mu_MA = [-1.2730 -5.6373];
                sigma_MA = [0.9286 2.9302];
                h1 = [0.2050 -0.7188];
                h2 = [-17.5670 -21.0513];
                g1 = [0.0074 -0.2896];
                g2 = [0.7490 -0.3951];
                Lcorr = [1.1191 1.1191];
                f1 = -0.9586;
                f2 = 10.8084;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 60)
                mu = [2.8958 1.9128];
                sigma = [1.7061 0.6869];
                dur_min = [13.9133 2.9398];
                mu_MA = [-1.1987 -13.1811];
                sigma_MA = [1.0492 2.6228];
                h1 = [-1.6501 0.6911];
                h2 = [-18.9375 -6.0721];
                g1 = [-0.1369 0.0598];
                g2 = [0.4477 3.7220];
                Lcorr = [3.0619 3.0619];
                f1 = -0.0419;
                f2 = 5.8920;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 70)
                mu = [4.1684 1.4778];
                sigma = [1.0766 0.7033];
                dur_min = [42.0185 1.8473];
                mu_MA = [0.1600 -10.2225];
                sigma_MA = [0.5082 1.8417];
                h1 = [-3.4369 0.3934];
                h2 = [-18.1632 -9.6284];
                g1 = [-1.1144 -0.1331];
                g2 = [0.9703 0.7223];
                Lcorr = [2.5817 2.5817];
                f1 = -0.1129;
                f2 = 4.0555;
                pBmin = 0.1;
                pBmax = 0.9;
            else
                error('Elevation not supported!')
            end
        case 'Village'
            if(elev_deg == 20)
                mu = [2.0262 1.9451];
                sigma = [1.2355 1.4293];
                dur_min = [2.2401 1.9624];
                mu_MA = [-3.1324 -16.5697];
                sigma_MA = [1.8929 4.0368];
                h1 = [-0.4368 1.0921];
                h2 = [-15.1009 1.6440];
                g1 = [-0.0423 -0.0325];
                g2 = [1.2532 2.4452];
                Lcorr = [0.8380 0.8380];
                f1 = 0.0590;
                f2 = 1.5623;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 30)
                mu = [2.4504 1.7813];
                sigma = [1.1061 1.2802];
                dur_min = [2.3941 2.1484];
                mu_MA = [-1.8384 -15.4143];
                sigma_MA = [1.7960 4.5579];
                h1 = [-0.5582 0.8549];
                h2 = [-14.4416 -2.2415];
                g1 = [-0.4545 -0.0761];
                g2 = [0.8188 1.6768];
                Lcorr = [0.9268 0.9268];
                f1 = -0.0330;
                f2 = 2.7056;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 45)
                mu = [2.2910 1.2738];
                sigma = [1.4229 1.1539];
                dur_min = [2.8605 0.7797];
                mu_MA = [-0.0018 -12.1063];
                sigma_MA = [1.1193 2.9814];
                h1 = [-1.2023 0.6537];
                h2 = [-14.0732 -4.5948];
                g1 = [-0.1033 -0.0815];
                g2 = [0.9299 1.6693];
                Lcorr = [0.9288 0.9288];
                f1 = 0.0002;
                f2 = 1.9694;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 60)
                mu = [3.0956 1.0920];
                sigma = [1.3725 1.2080];
                dur_min = [8.1516 0.7934];
                mu_MA = [-0.5220 -12.1817];
                sigma_MA = [1.0950 3.3604];
                h1 = [0.0831 1.1006];
                h2 = [-16.8546 0.5381];
                g1 = [0.0411 -0.0098];
                g2 = [1.1482 2.4287];
                Lcorr = [1.2251 1.2251];
                f1 = -0.0530;
                f2 = 2.7165;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 70)
                mu = [3.9982 1.4165];
                sigma = [1.3320 0.4685];
                dur_min = [28.3220 2.5168];
                mu_MA = [-1.3403 -11.9560];
                sigma_MA = [0.7793 1.5654];
                h1 = [-0.4861 0.5663];
                h2 = [-19.5316 -6.8615];
                g1 = [-0.2356 -0.2903];
                g2 = [0.7178 -1.2715];
                Lcorr = [1.4378 1.4378];
                f1 = -0.0983;
                f2 = 3.9005;
                pBmin = 0.1;
                pBmax = 0.9;
            else
                error('Elevation not supported!')
            end
        case 'Rural wooded'
            if(elev_deg == 20)
                mu = [2.0294 2.0290];
                sigma = [1.4280 1.5493];
                dur_min = [1.7836 1.5269];
                mu_MA = [-3.2536 -14.3363];
                sigma_MA = [1.6159 2.7753];
                h1 = [-0.5718 0.8186];
                h2 = [-16.1382 -2.9963];
                g1 = [-0.0805 -0.0822];
                g2 = [0.9430 1.7660];
                Lcorr = [1.0863 1.0863];
                f1 = 0.1263;
                f2 = 1.4478;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 30)
                mu = [2.1218 2.2051];
                sigma = [1.4895 1.5741];
                dur_min = [2.4539 2.1289];
                mu_MA = [-1.5431 -12.8884];
                sigma_MA = [1.8811 3.0097];
                h1 = [-0.7288 0.6635];
                h2 = [-14.1626 -4.6034];
                g1 = [-0.1241 -0.0634];
                g2 = [0.9482 2.3898];
                Lcorr = [1.3253 1.3253];
                f1 = 0.0849;
                f2 = 1.6324;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 45)
                mu = [3.1803 2.4017];
                sigma = [1.3427 1.1315];
                dur_min = [6.7673 3.5668];
                mu_MA = [0.0428 -11.3173];
                sigma_MA = [1.6768 2.7467];
                h1 = [-0.9948 0.2929];
                h2 = [-14.4265 -9.7910];
                g1 = [-0.1377 -0.0387];
                g2 = [1.0077 2.6194];
                Lcorr = [2.0419 2.0419];
                f1 = 0.1894;
                f2 = 2.1378;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 60)
                mu = [2.4961 2.2113];
                sigma = [1.4379 1.1254];
                dur_min = [3.7229 1.9001];
                mu_MA = [-1.0828 -12.3044];
                sigma_MA = [1.0022 2.3641];
                h1 = [-1.2973 0.5456];
                h2 = [-16.6791 -6.4660];
                g1 = [-0.1187 -0.0443];
                g2 = [0.6254 2.3029];
                Lcorr = [1.9038 1.9038];
                f1 = 0.1624;
                f2 = 1.8417;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 70)
                mu = [2.8382 2.1470];
                sigma = [1.3804 1.0038];
                dur_min = [6.8051 1.9195];
                mu_MA = [-0.8923 -11.5722];
                sigma_MA = [0.9455 2.3437];
                h1 = [-1.3425 0.3459];
                h2 = [-17.5636 -9.5399];
                g1 = [-0.1210 -0.0275];
                g2 = [0.6444 2.6238];
                Lcorr = [2.1466 2.1466];
                f1 = 0.0593;
                f2 = 2.8854;
                pBmin = 0.1;
                pBmax = 0.9;
            else
                error('Elevation not supported!')
            end
        case 'Residential'
            if(elev_deg == 20)
                mu = [2.9050 2.1969];
                sigma = [1.7236 0.9865];
                dur_min = [10.7373 2.2901];
                mu_MA = [-1.4426 -14.4036];
                sigma_MA = [1.2989 3.0396];
                h1 = [0.4875 0.5813];
                h2 = [-13.5981 -6.9790];
                g1 = [0.1343 -0.0911];
                g2 = [1.8247 2.1475];
                Lcorr = [1.2788 1.2788];
                f1 = 0.2334;
                f2 = 0.7612;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 30)
                mu = [2.7334 1.8403];
                sigma = [1.6971 0.9268];
                dur_min = [10.2996 1.8073];
                mu_MA = [-0.9996 -12.9855];
                sigma_MA = [1.0752 2.8149];
                h1 = [0.3407 0.3553];
                h2 = [-14.8465 -9.9284];
                g1 = [-0.0413 0.0501];
                g2 = [1.2006 3.8667];
                Lcorr = [1.7072 1.7072];
                f1 = 0.0443;
                f2 = 2.2591;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 60)
                mu = [3.4044 2.5534];
                sigma = [1.3980 1.7143];
                dur_min = [10.4862 4.7289];
                mu_MA = [0.4640 -2.3787];
                sigma_MA = [0.7060 0.8123];
                h1 = [0.3710 -2.3834];
                h2 = [-19.6032 -24.6987];
                g1 = [0.0332 0.0172];
                g2 = [0.5053 0.7237];
                Lcorr = [1.8017 1.8017];
                f1 = 3.1149;
                f2 = 3.5721;
                pBmin = 0.1;
                pBmax = 0.9;
            elseif(elev_deg == 70)
                mu = [2.9223 2.5188];
                sigma = [1.0267 1.3166];
                dur_min = [7.3764 7.2801];
                mu_MA = [-0.1628 -2.3703];
                sigma_MA = [0.5104 1.5998];
                h1 = [0.1590 -1.0228];
                h2 = [-20.4767 -22.4769];
                g1 = [0.1137 -0.0986];
                g2 = [0.4579 0.2879];
                Lcorr = [1.3531 1.3531];
                f1 = -0.0538;
                f2 = 5.1204;
                pBmin = 0.1;
                pBmax = 0.9;
            else
                error('Elevation not supported for this environment!')
            end
        otherwise
            error('Environment not supported!')
    end
else
    error('Carrier frequency not supported!')
end

%rng(555)

% Computation of averages
state_mean_duration = exp(mu + sigma.^2/2) .* (1 - erf((log(dur_min) - (mu + sigma.^2)) ./ sigma / sqrt(2))) ./ (1 - erf((log(dur_min) - mu) ./ sigma / sqrt(2)));
MA_min = [mu_MA(1) - 1.645*sigma_MA(1), mu_MA(2) + sqrt(2)*sigma_MA(2)*erfinv(2*pBmin-1)];
MA_max = [mu_MA(1) + 1.645*sigma_MA(1), mu_MA(2) + sqrt(2)*sigma_MA(2)*erfinv(2*pBmax-1)];
trans_mean_duration = f1 * (mu_MA(1) - mu_MA(2) - sigma_MA(2)^2 * (normpdf(MA_min(2),mu_MA(2),sigma_MA(2)) - normpdf(MA_max(2),mu_MA(2),sigma_MA(2))) ...
    / (normcdf(MA_max(2),mu_MA(2),sigma_MA(2)) - normcdf(MA_min(2),mu_MA(2),sigma_MA(2)))) + f2;

state_distr = state_mean_duration + trans_mean_duration;
state_distr = state_distr / sum(state_distr);

K1 = log(10)/10;
b0 = (K1 * g2).^2 / 2;
b1 = K1^2 * g1 .* g2 + K1;
b2 = (K1 * g1).^2 / 2;
v1 = 1 - 2 * sigma_MA.^2 .* b2;
mu1 = (mu_MA + sigma_MA.^2 .* b1) ./ v1;
sigma1 = sigma_MA ./ sqrt(v1);
v2 = (mu_MA.^2 - 2 * sigma_MA.^2 .* b0) ./ v1;
average_power_per_state1 = exp((mu1.^2 - v2) ./ (2 * sigma1.^2)) ./ sqrt(v1) .* (normcdf(MA_max,mu1,sigma1) - normcdf(MA_min,mu1,sigma1)) ...
    ./ (normcdf(MA_max,mu_MA,sigma_MA) - normcdf(MA_min,mu_MA,sigma_MA));

b0 = K1 * h2;
b1 = K1 * h1;
mu1 = mu_MA + sigma_MA.^2 .* b1;
v2 = mu_MA.^2 - 2 * sigma_MA.^2 .* b0;
average_power_per_state2 = exp((mu1.^2 - v2) ./ (2 * sigma_MA.^2)) .* (normcdf(MA_max,mu1,sigma_MA) - normcdf(MA_min,mu1,sigma_MA)) ...
    ./ (normcdf(MA_max,mu_MA,sigma_MA) - normcdf(MA_min,mu_MA,sigma_MA));

average_power_per_state = average_power_per_state1 + average_power_per_state2;

average_power = sum(state_distr .* average_power_per_state);



Ts = 0.02/3;                      % Sampling time [s]
vm = 5000 / 3600;               % Mobile speed (pedestrian) [m/s]
Ls = vm * Ts;                   % Sampling distance [m]
Lmax = Lch;%1e5;                     % Path length [m]
Nsamples = round(Lmax / Ls);    % Total number of samples
c = physconst('LightSpeed');    % Speed of light [m/s]
fD = f_c * vm / c;              % Doppler shift [Hz]
Lacc = 0;
curr_sample = 0;
state_ind = 1;
state = 1;                      % The initial state is 1 (Good) - Bad is 2

Jakes_impulse_response = Jakes_IR(fD, 1/Ts);

y = zeros(1,Nsamples);

state_duration_vec = zeros(1,2500);
MA_vec = zeros(1,2500);
SigmaA_vec = zeros(1,2500);
MP_vec = zeros(1,2500);
L_trans_vec = zeros(1,2500);
while(Lacc < Lmax)

    % Generation of state duration (length)
    curr_state_duration_m = lognrnd(mu(state), sigma(state));
    while(curr_state_duration_m < dur_min(state))
        curr_state_duration_m = lognrnd(mu(state), sigma(state));
    end
    state_duration_vec(state_ind) = curr_state_duration_m;
    Lacc = Lacc + curr_state_duration_m;

    % Generation of parameters of Loo distribution
    app_var = normrnd(mu_MA(state), sigma_MA(state));
    while(app_var < MA_min(state) || app_var > MA_max(state))
        app_var = normrnd(mu_MA(state), sigma_MA(state));
    end
    MA_vec(state_ind) = app_var;
    
    SigmaA_vec(state_ind) = g1(state) * MA_vec(state_ind) + g2(state);
    MP_vec(state_ind) = h1(state) * MA_vec(state_ind) + h2(state);

    % Computation of the transition length
    if(state_ind > 1)
        curr_trans_length_m = f1 * abs(MA_vec(state_ind)-MA_vec(state_ind-1)) + f2;    
        L_trans_vec(state_ind) = curr_trans_length_m;
    else
        curr_trans_length_m = 0;
    end
    Lacc = Lacc + curr_trans_length_m;

    curr_trans_length_samples = round(curr_trans_length_m / Ls);
    curr_state_duration_samples = round(curr_state_duration_m / Ls);

    rho_S = exp(-Ls / Lcorr(state));
    
    % Generation of channel samples in the transition
    if(curr_trans_length_m > 0)
        % Loo parameters in the transition are linearly interpolated
        MA_trans = ((1:curr_trans_length_samples) * MA_vec(state_ind) + (curr_trans_length_samples:-1:1) * MA_vec(state_ind-1)) / (curr_trans_length_samples+1);
        SigmaA_trans = ((1:curr_trans_length_samples) * SigmaA_vec(state_ind) + (curr_trans_length_samples:-1:1) * SigmaA_vec(state_ind-1)) / (curr_trans_length_samples+1);
        MP_trans = ((1:curr_trans_length_samples) * MP_vec(state_ind) + (curr_trans_length_samples:-1:1) * MP_vec(state_ind-1)) / (curr_trans_length_samples+1);

        % Direct component
        A_vec = normrnd(0, SigmaA_trans);
        A_vec = filter(sqrt(1-rho_S^2), [1,-rho_S], A_vec, (1-sqrt(1-rho_S^2))/rho_S*A_vec(1)) + MA_trans;
        alpha_vec = 10.^(A_vec/20);
    
        % Scattered (diffused) component
        diff_vec = normrnd(0, 1, 1, curr_trans_length_samples) + 1i * normrnd(0, 1, 1, curr_trans_length_samples);
        %diff_vec = Jakes(diff_vec, fD, 1/Ts);
        diff_vec = conv(diff_vec, Jakes_impulse_response,'same');
        sigmad = sqrt(0.5 * 10.^(MP_trans/10));
        diff_vec = sigmad .* diff_vec;

        % Superposition of direct and diffused
        alpha_vec = alpha_vec + diff_vec;

        y(curr_sample + (1:curr_trans_length_samples)) = alpha_vec;
        curr_sample = curr_sample + curr_trans_length_samples;
    end

    % Generation of channel samples in the state
    % Direct component
    A_vec = normrnd(0, SigmaA_vec(state_ind), 1, curr_state_duration_samples);
    A_vec = filter(sqrt(1-rho_S^2), [1,-rho_S], A_vec, (1-sqrt(1-rho_S^2))/rho_S*A_vec(1)) + MA_vec(state_ind);
    alpha_vec = 10.^(A_vec/20);

    % Scattered (diffused) component
    diff_vec = normrnd(0, 1, 1, curr_state_duration_samples) + 1i * normrnd(0, 1, 1, curr_state_duration_samples);
    %diff_vec = Jakes(diff_vec, fD, 1/Ts);
    diff_vec = conv(diff_vec, Jakes_impulse_response,'same');
    sigmad = sqrt(0.5 * 10^(MP_vec(state_ind)/10));
    diff_vec = sigmad * diff_vec;

    % Superposition of direct and diffused
    alpha_vec = alpha_vec + diff_vec;

    y(curr_sample + (1:curr_state_duration_samples)) = alpha_vec;
    curr_sample = curr_sample + curr_state_duration_samples;

    % Definition of the new state
    state_ind = state_ind + 1;
    state = 3-state;
end

state_duration_vec = state_duration_vec(1:(state_ind-1));
MA_vec = MA_vec(1:(state_ind-1));
SigmaA_vec = SigmaA_vec(1:(state_ind-1));
MP_vec = MP_vec(1:(state_ind-1));
L_trans_vec = L_trans_vec(1:(state_ind-1));
