import subprocess
import argparse
import logging
from pathlib import Path

# ========== CONFIGURABLE GLOBALS ==========
from config import CHECKSUMS_SUBDIR, SIG_EXTENSION, HASH_ALGORITHM, VERIFYSIGNATURES_LOG_LEVEL

# ========== LOGGING CONFIG ==========
logging.basicConfig(
    level=VERIFYSIGNATURES_LOG_LEVEL,
    format="%(asctime)s [%(levelname)s] %(message)s"
)

# ========== FUNCTION DEFINITIONS ==========

def verify_signature(file_path: Path, sig_path: Path, pubkey_path: Path) -> bool:
    """
    Verifies a SHA256 digital signature using OpenSSL.

    Args:
        file_path (Path): Path to the original file (e.g., .txt or .hash).
        sig_path (Path): Path to the .sig signature file.
        pubkey_path (Path): Path to the public key in PEM format.

    Returns:
        bool: True if verification succeeds, False otherwise.
    """
    try:
        subprocess.run([
            "openssl", "dgst", f"-{HASH_ALGORITHM}",
            "-verify", str(pubkey_path),
            "-signature", str(sig_path),
            str(file_path)
        ], check=True)
        logging.info(f"✅ Verified: {file_path.name}")
        return True
    except subprocess.CalledProcessError:
        logging.error(f"❌ Verification failed: {file_path.name}")
        return False


def get_signature_files(checksums_dir: Path) -> list:
    """
    Retrieves all .sig files from the checksums directory.

    Args:
        checksums_dir (Path): Path to the directory containing signatures and hash files.

    Returns:
        list: List of .sig file paths.
    """
    return list(checksums_dir.glob(f"*{SIG_EXTENSION}"))


def process_signatures(checksums_dir: Path, pubkey_path: Path) -> None:
    """
    Processes and verifies all signature files in the given directory.

    Args:
        checksums_dir (Path): Directory containing the checksum and signature files.
        pubkey_path (Path): Public key for verification.
    """
    if not checksums_dir.exists():
        logging.error(f"❌ Checksum directory not found: {checksums_dir}")
        return

    sig_files = get_signature_files(checksums_dir)
    if not sig_files:
        logging.warning("⚠️ No signature files found.")
        return

    for sig_file in sig_files:
        hash_file = sig_file.with_suffix('')  # Strip .sig
        if not hash_file.exists():
            logging.warning(f"⚠️ Hash file missing for signature: {sig_file.name}")
            continue
        verify_signature(hash_file, sig_file, pubkey_path)


def main():
    """
    Entry point: parses arguments and initiates signature verification.
    """
    parser = argparse.ArgumentParser(description="Verify SHA256 signature files in a forensic case folder")
    parser.add_argument("case_folder", help="Path to the forensic case folder")
    parser.add_argument("pubkey_path", help="Path to the public key (PEM format)")
    args = parser.parse_args()

    case_folder = Path(args.case_folder).resolve()
    pubkey_path = Path(args.pubkey_path).resolve()
    checksums_dir = case_folder / CHECKSUMS_SUBDIR

    process_signatures(checksums_dir, pubkey_path)


# ========== SCRIPT ENTRY POINT ==========
if __name__ == "__main__":
    main()
