import os
import sys
import subprocess
import logging
from concurrent.futures import ThreadPoolExecutor, as_completed
from tqdm import tqdm
import argparse

# === CONFIGURATION (GLOBAL VARIABLES) ===

sys.path.append(os.path.abspath(os.path.join(os.path.dirname(__file__), "../../")))
from config import (
    BASE_PATHS,
    INPUT_RELPATHS,
    OUTPUT_RELPATH,
    VOL_SCRIPT,
    MAX_WORKERS,
    PLUGINS_BY_OS,
    LOG_FILE_SCAN_MEMORY,
    SCANMEMORY_LOG_LEVEL
)

LOG_FILE = LOG_FILE_SCAN_MEMORY

# === ARGUMENT PARSING ===

def parse_args():
    parser = argparse.ArgumentParser(description="Run Volatility plugins in parallel on memory dumps.")
    parser.add_argument('--case', type=str, help="Specific case folder to process (e.g., 'case01'). If omitted, processes all.")
    parser.add_argument('--case-folder', type=str, help="Path to the case folder (required if --memory-dump is used).")
    parser.add_argument('--os', type=str, choices=BASE_PATHS.keys(), help="Operating system type (e.g., 'windows', 'linux'). If omitted, processes all supported OS types.")
    parser.add_argument('--auto', action='store_true', help="Automatically detect and process all cases in the base paths.")
    parser.add_argument('--memory-dump', type=str, help="Path to a specific memory dump file to process (overrides case and os).")
    return parser.parse_args()

# === LOGGING SETUP ===

def setup_logging():
    logging.basicConfig(
        filename=LOG_FILE_SCAN_MEMORY,
        level=SCANMEMORY_LOG_LEVEL,
        format='%(asctime)s - %(levelname)s - %(message)s'
    )
    logging.info("==== Starting Volatility Parallel Scan with Progress Bar ====")

# === TASK BUILDING ===

def build_tasks_auto(case_filter=None):
    """
    Build a list of tasks to run Volatility plugins on memory dumps for different OS cases.

    Args:
        case_filter (str or None): Optional. If specified, only tasks for this case folder will be created.
                                   If None, tasks for all available case folders will be created.

    Returns:
        list of tuples: Each tuple contains (command, os_type, case, plugin), where:
            - command (str): The shell command to run a Volatility plugin on a memory dump.
            - os_type (str): Operating system type (e.g., 'windows', 'linux').
            - case (str): Case folder name.
            - plugin (str): Volatility plugin name to run.

    Logs warnings for missing base paths, missing case folders, or missing memdump files.
    """
    tasks = []
    for os_type, base_path in BASE_PATHS.items():
        if not os.path.isdir(base_path):
            logging.warning(f"{os_type.upper()} base path missing: {base_path}")
            continue

        all_cases = [
            d for d in os.listdir(base_path)
            if os.path.isdir(os.path.join(base_path, d))
        ]

        selected_cases = [case_filter] if case_filter and case_filter in all_cases else all_cases
        if case_filter and case_filter not in all_cases:
            logging.warning(f"{os_type.upper()} has no case folder named '{case_filter}'")
            continue

        for case in selected_cases:
            input_path = os.path.join(base_path, case, "01_acquisition", INPUT_RELPATHS[os_type])
            if not os.path.isfile(input_path):
                logging.warning(f"[{os_type.upper()}] Missing memdump: {case}")
                continue

            output_dir = os.path.join(base_path, case, OUTPUT_RELPATH)
            os.makedirs(output_dir, exist_ok=True)

            for plugin in PLUGINS_BY_OS[os_type]:
                plugin_name = plugin.split('.')[-1]
                output_file = os.path.join(output_dir, f"{plugin_name}.txt")
                command = (
                    f"(echo 'plugin name: {plugin_name}' && "
                    f"{VOL_SCRIPT} -f '{input_path}' {plugin}) > '{output_file}'"
                )
                tasks.append((command, os_type, case, plugin))
    return tasks


# === TASK BUILDING FOR SPECIFIC OS/CASE ===

def build_tasks(os_type, case):
    """
    Build a list of tasks to run Volatility plugins for a specific OS type and case folder.

    Args:
        os_type (str): Operating system type (e.g., 'windows', 'linux').
        case (str): Case folder name.

    Returns:
        list of tuples: Each tuple contains (command, os_type, case, plugin).

    Logs warnings for missing base paths or memdump files.
    """
    tasks = []

    # Controllo che l'OS sia supportato
    if os_type not in BASE_PATHS:
        logging.warning(f"Unsupported OS type: {os_type}")
        return tasks

    base_path = BASE_PATHS[os_type]
    if not os.path.isdir(base_path):
        logging.warning(f"{os_type.upper()} base path missing: {base_path}")
        return tasks

    case_path = os.path.join(base_path, case)
    if not os.path.isdir(case_path):
        logging.warning(f"{os_type.upper()} has no case folder named '{case}'")
        return tasks

    input_path = os.path.join(case_path, "01_acquisition", INPUT_RELPATHS[os_type])
    if not os.path.isfile(input_path):
        logging.warning(f"[{os_type.upper()}] Missing memdump: {case}")
        return tasks

    output_dir = os.path.join(case_path, OUTPUT_RELPATH)
    os.makedirs(output_dir, exist_ok=True)

    for plugin in PLUGINS_BY_OS[os_type]:
        plugin_name = plugin.split('.')[-1]
        output_file = os.path.join(output_dir, f"{plugin_name}.txt")
        command = (
            f"(echo 'plugin name: {plugin_name}' && "
            f"{VOL_SCRIPT} -f '{input_path}' {plugin}) > '{output_file}'"
        )
        tasks.append((command, os_type, case, plugin))

    return tasks

def build_tasks_from_folder(artefact_folder_path, case_folder_path, os_type=None):
    """
    Build a list of tasks to run Volatility plugins directly from a given folder.

    Args:
        artefact_folder_path (str): Path to the folder containing the memory dump.
        case_folder_path (str): Path to the case folder.
        os_type (str, optional): Operating system type. If None, tries to infer from folder_path.

    Returns:
        list of tuples: Each tuple contains (command, os_type, case, plugin).

    Logs warnings for missing memdump files or unsupported OS.
    """
    tasks = []

    if not os.path.isdir(artefact_folder_path):
        logging.warning(f"Invalid artefacts folder: {artefact_folder_path}")
        return tasks

    # Detect case name
    case = os.path.basename(os.path.normpath(case_folder_path))

    # If OS not given, try to infer (based on parent folder)
    if os_type is None:
        parent = os.path.basename(os.path.dirname(artefact_folder_path)).lower()
        if parent in BASE_PATHS:
            os_type = parent
        else:
            logging.warning(f"Could not infer OS type from folder path: {artefact_folder_path}")
            return tasks

    if os_type not in PLUGINS_BY_OS:
        logging.warning(f"Unsupported OS type: {os_type}")
        return tasks

    # Input dump path
    input_path = os.path.join(artefact_folder_path, INPUT_RELPATHS[os_type])
    if not os.path.isfile(input_path):
        logging.warning(f"[{os_type.upper()}] Missing memdump in {artefact_folder_path}")
        return tasks

    # Output dir
    output_dir = os.path.join(case_folder_path, OUTPUT_RELPATH)
    os.makedirs(output_dir, exist_ok=True)

    # Build commands
    for plugin in PLUGINS_BY_OS[os_type]:
        plugin_name = plugin.split('.')[-1]
        output_file = os.path.join(output_dir, f"{plugin_name}.txt")
        command = (
            f"(echo 'plugin name: {plugin_name}' && "
            f"{VOL_SCRIPT} -f '{input_path}' {plugin}) > '{output_file}'"
        )
        tasks.append((command, os_type, case, plugin))

    return tasks

# === EXECUTION FUNCTION ===

def run_volatility_plugin(task):
    """
    Execute a single Volatility plugin command as a subprocess.

    Args:
        task (tuple): A tuple containing (command, os_type, case, plugin).
            - command (str): The shell command to execute.
            - os_type (str): Operating system type.
            - case (str): Case folder name.
            - plugin (str): Volatility plugin name.

    Returns:
        str: A log-friendly string indicating success or failure of the command execution.

    Logs info when starting execution and returns success or failure messages.
    """
    command, os_type, case, plugin = task
    logging.info(f"🔄 START [{os_type.upper()}] {case} → {plugin}")
    try:
        subprocess.run(command, shell=True, check=True)
        return f"✅ [{os_type.upper()}] {case} → {plugin}"
    except subprocess.CalledProcessError as e:
        return f"❌ [{os_type.upper()}] {case} → {plugin} failed: {e}"

# === PARALLEL EXECUTION ===

def execute_tasks_in_parallel(tasks):
    """
    Run a list of Volatility plugin tasks in parallel using a thread pool executor.

    Args:
        tasks (list of tuples): List of tasks as returned by build_tasks(), each containing
                                (command, os_type, case, plugin).

    Returns:
        None

    Logs info messages for each task result and shows progress with a tqdm progress bar.
    """
    with ThreadPoolExecutor(max_workers=MAX_WORKERS) as executor:
        futures = {executor.submit(run_volatility_plugin, task): task for task in tasks}
        with tqdm(total=len(tasks), desc="Running Volatility", ncols=100) as pbar:
            for future in as_completed(futures):
                result = future.result()
                logging.info(result)
                pbar.update(1)

# === CLEANUP OUTPUT ===

def remove_short_output_files(tasks):
    """
    Remove output files with <= 6 lines, only for the OS types used in the given tasks.
    """
    used_os_types = {t[1] for t in tasks}  # estraggo gli os_type dai task

    for os_type in used_os_types:
        base_path = BASE_PATHS[os_type]
        for case in os.listdir(base_path):
            output_dir = os.path.join(base_path, case, OUTPUT_RELPATH)
            if not os.path.isdir(output_dir):
                continue

            for filename in os.listdir(output_dir):
                file_path = os.path.join(output_dir, filename)
                try:
                    with open(file_path, 'r') as f:
                        lines = sum(1 for _ in f)
                    if lines <= 6:
                        os.remove(file_path)
                        logging.info(f"🗑️ Removed short output file: {file_path} ({lines} lines)")
                except Exception as e:
                    logging.warning(f"⚠️ Error checking/removing file {file_path}: {e}")


def print_last_log_lines(n=10):
        """
        Print the last n lines of the log file.

        Args:
            n (int): Number of lines to print from the end of the log file.
        """
        log_path = os.path.abspath(LOG_FILE)
        try:
            with open(log_path, "r") as f:
                lines = f.readlines()
                print(f"\n--- Last {n} lines of log file ({log_path}) ---")
                for line in lines[-n:]:
                    print(line.rstrip())
        except Exception as e:
            print(f"Could not read log file at {log_path}: {e}")

# === MAIN FUNCTION ===

def main():
    args = parse_args()
    setup_logging()
    if args.memory_dump:
        tasks = build_tasks_from_folder(args.memory_dump, args.case_folder, os_type=args.os)
    else:
        tasks = build_tasks(args.os, args.case) if args.os else (build_tasks_auto(args.case) if args.auto else build_tasks_auto())
    if not tasks:
        logging.info("No tasks to execute.")
        print_last_log_lines()
        return
    execute_tasks_in_parallel(tasks)
    remove_short_output_files(tasks)
    print_last_log_lines()
    

if __name__ == "__main__":
    main()
