import logging
import subprocess
import time
import os
import re
import json
import requests
from typing import TypedDict
from langgraph.graph import StateGraph, START, END
from google import genai
import shlex
from datetime import datetime
import uuid
import sys

def ls_date_format(dt):
    return dt.strftime("%b %e  %Y")

current_date = datetime.now()
current_date = current_date.strftime("%Y-%b-%d")


# ========== CONFIGURATION ==========
sys.path.append(os.path.abspath(os.path.join(os.path.dirname(__file__), "../../../")))
from config import (
    GEMINI_API_KEY,
    GEMINI_API_KEY_2,
    GEMINI_API_KEY_3,
    GEMINI_API_KEY_4,
    GEMINI_MODEL_NAME,
    SCRIPT_TIMEOUT,
    SCRIPT_PATH,
    SCRIPT2_PATH,
    PYTHON_BIN,
    SHOW_NOT_SUSPICIOUS,
    MAX_TPM,
    DISK_AGENT_LOG_FILE,
    MAX_RETRIES,
    RETRY_WAIT,
    DISK_AGENT_LOG_LEVEL,
    OLLAMA_MODEL,
    OLLAMA_MODEL_FAST,
    OLLAMA_BASE_URL,
    DISK_AGENT_SUMMARIZE_PROMPT,
    DISK_AGENT_ANALYZE_FILE_PROMPT,
    DISK_AGENT_ANALYZE_FOLDER_PROMPT,
    CHUNK_DISK_PROMPT_TEMPLATE,
    DISK_PROMPT_LLM_LOCAL
)

# GLOBAL VARIABLES PLACEHOLDERS
HDD_IMAGE_PATH = ""
STARTING_FOLDER = ""

LAST_ACTION = ""

# ========== LOGGING SETUP ==========
LOG_FILE = DISK_AGENT_LOG_FILE

file_handler = logging.FileHandler(LOG_FILE, mode='a')
file_handler.setFormatter(logging.Formatter("**%(asctime)s** [%(levelname)s] %(message)s\n"))

logging.basicConfig(
    level=DISK_AGENT_LOG_LEVEL,
    format="%(asctime)s [%(levelname)s] %(message)s",
    handlers=[
        logging.StreamHandler(),
        file_handler
    ]
)

# ========== INITIALIZATION ==========
logging.info("Configuring Gemini client.")
GEMINI_KEYS = [GEMINI_API_KEY, GEMINI_API_KEY_2, GEMINI_API_KEY_3, GEMINI_API_KEY_4]
current_key_index = 0
llm_client = genai.Client(api_key=GEMINI_KEYS[current_key_index])

# ========== TYPES ==========
class MyState(TypedDict):
    history: list[str]
    step: int
    final: bool
    last_output: str
    user_input: str
    skip_action: bool


def summarize_history(history: list[str], local=False) -> str:
    """Summarizes the conversation history using the Ollama API or Gemini LLM."""
    logging.info("Summarizing history...")
    prompt = DISK_AGENT_SUMMARIZE_PROMPT + "\n".join(history)
    logging.info(prompt)
    
    if (local):
        headers = {"Content-Type": "application/json"}
        payload = {
            "model": OLLAMA_MODEL_FAST,
            "prompt": prompt,
            "stream": False
        }
        response = requests.post(
            f"{OLLAMA_BASE_URL}/api/generate",
            headers=headers,
            data=json.dumps(payload)
        )
        response.raise_for_status()
        result = response.json()
        summary = result.get("response", "")
        logging.debug(f"Summary generated:\n{summary}")
        return summary
    else:
        for attempt in range(MAX_RETRIES):
            try:
                response = llm_client.models.generate_content(
                    model=GEMINI_MODEL_NAME,
                    contents=prompt
                )
                summary = response.text.strip()
                logging.info(f"Summary generated:\n{summary}")
                return summary

            except Exception as e:
                error_str = str(e)
                logging.error(f"Gemini API error: {error_str}")

                # --- Daily quota exceeded → rotate key
                if "GenerateRequestsPerDayPerProjectPerModel" in error_str:
                    if current_key_index < len(GEMINI_KEYS) - 1:
                        current_key_index += 1
                        new_key = GEMINI_KEYS[current_key_index]
                        logging.warning(f"Quota exceeded. Switching to API key index {current_key_index}.")
                        llm_client = genai.Client(api_key=new_key)
                        continue  # retry immediately with new key
                    else:
                        logging.error("All Gemini API keys exhausted.")
                        return ""

                # --- Transient errors → retry
                elif any(x in error_str for x in ["RESOURCE_EXHAUSTED", "429", "500", "503"]):
                    wait_time = RETRY_WAIT * (attempt + 1)  # optional backoff
                    logging.warning(f"Transient error. Retry {attempt+1}/{MAX_RETRIES} in {wait_time}s...")
                    time.sleep(wait_time)
                    continue

                # --- Non-retryable error
                else:
                    logging.error("Non-retryable error. Aborting.")
                    return ""

        logging.error("Max retries reached. Aborting.")
        return ""


def chunk_by_rows(text: str, max_rows: int = 200) -> list[str]:
    """
    Split the ls-style text into chunks, preserving complete rows.

    Args:
        text (str): Full ls output.
        max_rows (int): Max number of rows per chunk.

    Returns:
        list[str]: List of chunked ls outputs.
    """
    lines = text.strip().splitlines()
    chunks = []
    for i in range(0, len(lines), max_rows):
        chunk = "\n".join(lines[i:i+max_rows])
        chunks.append(chunk)
    return chunks

# ========== LLM THINK FUNCTION ==========
def llm_think(state: MyState, local=False) -> MyState:
    """
    Use the Gemini LLM to analyze the user input and generate the next step.

    Args:
        state (MyState): Current agent state including history, input, step count.

    Returns:
        MyState: Updated state with LLM output added to history, next step, and flags for final state.
    """
    global llm_client, current_key_index
    if state['step'] == 0:
        logging.info(f"Thinking at step {state['step']} with input: {state['user_input']}")
    else:
        logging.info(f"Thinking at step {state['step']}")


    context = "\n".join(state["history"])
    prompt = ""
    prompt = build_prompt(context, state['user_input'], local=local)

    if len(prompt) * 1.3 >= MAX_TPM :
        logging.warning(f"prompt exceeds {MAX_TPM} token limit, truncating...")
        prompt = prompt[:125000]

    if state['step'] == 0:
        logging.info("Initial prompt for LLM:\n" + prompt)

    logging.info(f"Generated prompt for LLM:\n{prompt}")

    try:
        if(local):

            # chunking for very large prompts
            if (len(state['user_input']) > 1000 and state["step"]==0):
                logging.info("Input too large, applying chunking...")
                chunks = chunk_by_rows(state['user_input'], 50)  # chunk by 50 rows
                seen_lines = set()
                combined_content = ""
                for idx, chunk in enumerate(chunks):
                    chunk_prompt = CHUNK_DISK_PROMPT_TEMPLATE.format(
                        context=combined_content,
                        chunk_text=chunk,
                    )
                    logging.info(f"Processing chunk {idx + 1}/{len(chunks)}")
                    headers = {"Content-Type": "application/json"}
                    payload = {
                        "model": OLLAMA_MODEL,
                        "prompt": chunk_prompt,
                        "stream": False,  # Disable streaming for simpler response handling
                    }
                    response = requests.post(
                        f"{OLLAMA_BASE_URL}/api/generate",
                        headers=headers,
                        data=json.dumps(payload)
                    )
                    
                    # Check for HTTP errors
                    response.raise_for_status()
                    result = response.json()
                    content = result.get("response", "")
                    #store combined content
                    if idx == 0:
                        #copy the first line of the real input to the content (starting folder line)
                        content = state['user_input'].splitlines()[0] + "\n" + content
                    else:
                        # Remove the first two lines in subsequent chunks (header + possible repeated context)
                        content = "\n".join(content.splitlines()[2:])
                        # remove "PERMISSIONS UID ..." lines if present
                        content = content.replace("PERMISSIONS UID GID SIZE LAST_MODIFIED NAME\n", "")
                    content = content.replace("```", "").strip()
                    new_lines = []
                    for line in content.splitlines():
                        if line not in seen_lines:
                            seen_lines.add(line)
                            new_lines.append(line)
                    combined_content += "\n".join(new_lines) + "\n"
                prompt = build_prompt(context, combined_content, local=local)
                state['user_input'] = combined_content  # update user_input for next instruction

            headers = {"Content-Type": "application/json"}
            payload = {
                "model": OLLAMA_MODEL,
                "prompt": prompt,
                "stream": False,  # Disable streaming for simpler response handling
            }
            response = requests.post(
                f"{OLLAMA_BASE_URL}/api/generate",
                headers=headers,
                data=json.dumps(payload)
            )
            
            # Check for HTTP errors
            response.raise_for_status()
            
            # Parse and return response
            result = response.json()
            logging.info("Ollama generation successful.")
            content = result.get("response", "")
            logging.debug(f"Ollama response: {content}")
        else:
            response = llm_client.models.generate_content(
                model=GEMINI_MODEL_NAME,
                contents=prompt
            )
            content = response.text.strip()

        # extract LLM: and Observation: blocks
        llm_match = re.search(r"(LLM:.*?)(?=^Observation:|^Final Answer:|\Z)", content, re.DOTALL | re.MULTILINE)
        obs_match = re.search(r"(Observation:.*?)(?=^Final Answer:|\Z)", content, re.DOTALL | re.MULTILINE)
        final_match = re.search(r"(Final Answer:.*)", content, re.DOTALL | re.MULTILINE)

        llm_part = llm_match.group(1).strip() if llm_match else "LLM section not found."
        obs_part = obs_match.group(1).strip() if obs_match else "Observation section not found."
        final_part = final_match.group(1).strip() if final_match else "Final Answer section not found."

        logging.info(f"--- LLM ---\n{llm_part}")
        logging.info(f"--- Observation ---\n{obs_part}")
        logging.info(f"--- Final Answer ---\n{final_part}")

    except Exception as e:
        # Check if it's a rate-limit / quota error
        error_str = str(e)
        print(error_str)
        # Handle daily quota exceeded
        if "GenerateRequestsPerDayPerProjectPerModel" in error_str:
            if current_key_index < len(GEMINI_KEYS) - 1:
                current_key_index += 1
                new_key = GEMINI_KEYS[current_key_index]
                logging.warning(f"Quota exceeded. Switching to API key index {current_key_index}.")
                llm_client = genai.Client(api_key=new_key)
                return llm_think(state, local)  # retry immediately with new key
            else:
                logging.error("All Gemini API keys exhausted.")
                state["history"].append("LLM Error: All API keys exhausted.")
                state["final"] = True
                return state

        # Handle transient errors (RESOURCE_EXHAUSTED, 429, 500)
        elif any(x in error_str for x in ["RESOURCE_EXHAUSTED", "429", "500", "503"]):
            for attempt in range(MAX_RETRIES):
                logging.warning(f"Quota/server error. Retry {attempt+1}/{MAX_RETRIES} in {RETRY_WAIT}s...")
                time.sleep(RETRY_WAIT)
                try:
                    return llm_think(state, local)  # retry
                except Exception as retry_e:
                    logging.error(f"Retry {attempt+1} failed: {retry_e}")
            logging.error("Max retries reached. Aborting.")
            state["history"].append("LLM Error: Max retries reached.")
            state["final"] = True
            return state

    state["history"].append(content)
    total_chars = len("".join(state["history"]))
    logging.info(f"History total length (chars): {total_chars}")
    # if history is too long, summarize it
    if total_chars > 10000:
        logging.info("history too long, summarizing...")
        summary = summarize_history(state["history"], local)
        state["history"] = [f"[SUMMARY UP TO NOW]\n{summary}"]

    # Priority to Action: if present, handle action first, otherwise check for final answer
    global LAST_ACTION
    action_match = re.search(r"^Action:.*", content, re.MULTILINE)
    if action_match:
        state["last_output"] = action_match.group(0)
        if(LAST_ACTION != state["last_output"]):
            LAST_ACTION = state["last_output"]
            state["skip_tool"] = False
            state["step"] = state["step"] - 5 if state["step"] > 6 else state["step"]
        else:
            # mark to skip tool execution
            state["skip_tool"] = True
            state["history"].append("LOOP DETECTED: The same action was requested twice in succession. "
    "To prevent redundant execution and possible infinite loops, the repeated action output has been suppressed. "
    "Please review your reasoning")
            state["step"]=state["step"]+5
        logging.info(f"Action detected: {state['last_output']}")
    elif final_part and final_part.startswith("Final Answer:"):
        state["final"] = True
        state["last_output"] = final_part
        logging.info("Final answer detected.")
    else:
        state["last_output"] = "Unrecognized response format."
        state["final"] = True
        logging.warning("Unrecognized LLM response format.")

    state["step"] += 1
    return state

# ========== TOOL EXECUTION FUNCTION ==========
def do_tool(state: MyState, local: bool) -> MyState:
    """
    Parses the LLM output for an action, validates the script path, executes it, and logs observations.

    Args:
        state (MyState): Current state with 'last_output' containing the action instruction.

    Returns:
        MyState: Updated state including script output (stdout/stderr) and any execution errors.
    """
    if state.get("skip_tool", False):
        logging.info("Skipping tool execution due to duplicate action.")
        return state
    match = re.match(r"Action:\s*(\S+\.py)(.*)", state["last_output"])
    if not match:
        state["final"] = True
        return state

    script_name = match.group(1)
    args = shlex.split(match.group(2).strip())

    if not os.path.exists(script_name):
        error_msg = f"Tool Error: Script '{script_name}' not found."
        logging.error(error_msg)
        state["history"].append(error_msg)
        state["final"] = True
        return state

    try:
        #print(f"Running: {[PYTHON_BIN, script_name, *args]}")
        result = subprocess.run(
            [PYTHON_BIN, script_name, *args],
            capture_output=True,
            text=True,
            timeout=SCRIPT_TIMEOUT
        )
        output = result.stdout.strip()
        error = result.stderr.strip()

        #print(f"Command executed: {PYTHON_BIN} {script_name} {' '.join(args)}")

        script_output = f"Output of {script_name} {' '.join(args)}:\n{output}"

        if result.returncode != 0:
            script_output += f"\nErrors:\n{error}"
            logging.warning(f"Script {script_name} stderr: {error}")
        else:
            logging.info(f"Script {script_name} executed successfully.")

        if local:
            # truncate output if too long
            if len(script_output) > 500:
                script_output = script_output[:500] + "\n...[truncated]..."
        state["history"].append("action output: '''\n" + script_output +"\n'''\n")
    except Exception as e:
        logging.exception(f"Exception during script execution: {e}")
        state["history"].append(f"Tool Exception: {str(e)}")
        state["final"] = True

    return state

# ========== PROMPT GENERATION ==========
def build_prompt(context: str, user_input: str, local=False, chunked_user_input=None) -> str:
    """
    Construct the LLM prompt using historical context and the current user input.

    Args:
        context (str): History of the conversation or actions.
        user_input (str): Current user input to analyze.

    Returns:
        str: Formatted prompt to be passed to the LLM.
    """
    # Add timestamp + UUID prefix to make the prompt unique
    unique_prefix = f"[RequestID: {int(time.time())}-{uuid.uuid4().hex[:8]}]\n"
    if(local):
        prompt = DISK_PROMPT_LLM_LOCAL.format(
            current_date=current_date,
            SCRIPT2_PATH=SCRIPT2_PATH,
            HDD_IMAGE_PATH=HDD_IMAGE_PATH,
            SCRIPT_PATH=SCRIPT_PATH,
            context=context,
            user_input=user_input if not chunked_user_input else chunked_user_input
        )
        return prompt
    prompt = unique_prefix + DISK_AGENT_ANALYZE_FOLDER_PROMPT.format(
        current_date=current_date,
        SCRIPT2_PATH=SCRIPT2_PATH,
        HDD_IMAGE_PATH=HDD_IMAGE_PATH,
        SCRIPT_PATH=SCRIPT_PATH,
        context=context,
        user_input=user_input
    )
    logging.debug(f"Generated prompt: \n {prompt}")
    return prompt

# ========== GRAPH BUILDING ==========
def build_graph() -> StateGraph:
    """
    Build and compile the LangGraph state machine with 'think' and 'act' stages.

    Returns:
        StateGraph: A compiled LangGraph instance ready for execution.
    """
    graph = StateGraph(MyState)
    graph.add_node("think", llm_think)
    graph.add_node("act", do_tool)
    graph.add_edge(START, "think")
    graph.add_edge("think", "act")
    graph.add_edge("act", "think")
    graph.add_edge("think", END)
    graph.add_edge("act", END)
    return graph.compile()

# ========== MAIN EXECUTION LOOP ==========
def run_agent(initial_input: str, local=False):
    """
    Run the main loop for the agent using the initial user input. The loop
    alternates between LLM thinking and tool execution until a final response is reached.

    Args:
        initial_input (str): The initial directory listing or user input to analyze.

    Returns:
        None. Prints the final result to stdout.
    """
    if len(initial_input) * 1.3 >= MAX_TPM :
        logging.warning(f"prompt exceeds {MAX_TPM} token limit, truncating...")
        prompt = prompt[:125000]

    initial_state = MyState(
        history=[],
        step=0,
        final=False,
        last_output="",
        user_input=initial_input
    )

    logging.info("Starting agent run.")
    state = initial_state
    while not state["final"]:
        state = llm_think(state, local)
        if state["step"] >= 15:
            state["final"] = True
        if state["final"]:
            break
        state = do_tool(state, local)

    logging.info("Agent run completed.")

    # Append entire conversation history and final output to Markdown log
    with open(DISK_AGENT_LOG_FILE, "a") as f:
        f.write("\n---\n\n## Conversation History\n\n")
        for item in state["history"]:
            if item.startswith("LLM:"):
                f.write(f"### 🤖 LLM Response\n```\n{item[5:]}\n```\n")
            elif item.startswith("Observation:"):
                f.write(f"### 🔍 Observation\n```\n{item[12:]}\n```\n")
            elif item.startswith("Tool Error") or item.startswith("Tool Exception"):
                f.write(f"### ❌ Error\n```\n{item}\n```\n")
            else:
                f.write(f"### ℹ️ Note\n```\n{item}\n```\n")
        final_response = f"## Final Response\n\n```\n{state['last_output']}\n```\n"
        f.write(final_response)

    # Save only the last LLM, Observation, and Final Answer if -o <path> is provided
    if args.output:
        last_llm = None
        last_obs = None
        last_final = None

        # Walk history backwards to get the most recent ones
        for entry in reversed(state["history"]):
            if entry.startswith("LLM:") and last_llm is None:
                last_llm = entry[5:].strip()
            elif entry.startswith("Observation:") and last_obs is None:
                last_obs = entry[12:].strip()
            elif entry.startswith("Final Answer:") and last_final is None:
                last_final = entry[len("Final Answer:"):].strip()
            if last_llm and last_obs and last_final:
                break
                
        with open(args.output, "w") as f:
            f.write("# Forensic Agent Final Report\n\n")
            if last_llm:
                f.write("## 🤖 LLM Response\n")
                f.write(f"```\n{last_llm}\n```\n\n")
            if last_obs:
                f.write("## 🔍 Observation\n")
                f.write(f"```\n{last_obs}\n```\n\n")
            if last_final:
                f.write("## ✅ Final Answer\n")
                f.write(f"```\n{last_final}\n```\n\n")

    final_response = f"## Final Response\n\n```\n{state['last_output']}\n```\n"
    logging.info(final_response)

# ========== FILE ONLY ANALYSIS ===========
def analyze_file_with_llm(
    file_to_examine,
    file_content,
    current_date,
    args,
    MAX_TPM,
    GEMINI_MODEL_NAME,
    SHOW_NOT_SUSPICIOUS=True,
    retry_wait=RETRY_WAIT,
    max_retries=MAX_RETRIES,
    local=False
):
    """
    Analyze a file with Gemini LLM and extract suspicious indicators.

    Parameters:
        llm_client: Initialized LLM client instance.
        file_to_examine (str): The name of the file to analyze.
        file_content (str): Content of the file.
        current_date (str): Date string (used in prompt).
        args: argparse arguments object (must contain .output).
        MAX_TPM (int): Token limit for model input.
        GEMINI_MODEL_NAME (str): LLM model name.
        SHOW_NOT_SUSPICIOUS (bool): Whether to save benign results too.
        retry_wait (int): Wait time (seconds) before retry on failure.
        max_retries (int): Maximum number of retries.

    Returns:
        str | None: The analysis text if successful, else None.
    """
    unique_prefix = f"[RequestID: {int(time.time())}-{uuid.uuid4().hex[:8]}]\n"
    prompt = unique_prefix + DISK_AGENT_ANALYZE_FILE_PROMPT.format(
            current_date=current_date,
            file_to_examine=file_to_examine,
            file_content=file_content
        )
    global llm_client, current_key_index

    # Check if prompt exceeds model limit
    if len(prompt) * 1.3 >= MAX_TPM:
        logging.warning(f"Prompt exceeds {MAX_TPM} token limit, truncating...")
        prompt = prompt[:125000]

    # Retry loop
    for attempt in range(1, max_retries + 1):
        try:
            if(local):
                headers = {"Content-Type": "application/json"}
                payload = {
                    "model": OLLAMA_MODEL,
                    "prompt": prompt,
                    "stream": False,  # Disable streaming for simpler response handling
                }
                response = requests.post(
                    f"{OLLAMA_BASE_URL}/api/generate",
                    headers=headers,
                    data=json.dumps(payload),
                )
                
                # Check for HTTP errors
                response.raise_for_status()
                
                # Parse and return response
                result = response.json()
                print("[INFO] Ollama generation successful.")
                analysis = result.get("response", "")
                print(f"[DEBUG] Ollama response: {analysis}")
                if analysis:
                    if args.output:
                        if SHOW_NOT_SUSPICIOUS or any(
                            s in (analysis or "").lower()
                            for s in ["suspicious: true", "suspicious:true", '"suspicious": true', '"suspicious":true']
                        ):
                            with open(args.output, "w") as f:
                                f.write("# File Analysis Report\n\n")
                                f.write(f"{analysis}")
                    return analysis

            else:
                response = llm_client.models.generate_content(
                    model=GEMINI_MODEL_NAME,
                    contents=prompt
                )
                if response.text:
                    analysis = response.text.strip()
                    logging.info("Gemini analysis completed successfully.")

                    if args.output:
                        if SHOW_NOT_SUSPICIOUS or any(
                            s in (analysis or "").lower()
                            for s in ["suspicious: true", "suspicious:true", '"suspicious": true', '"suspicious":true']
                        ):
                            with open(args.output, "w") as f:
                                f.write("# File Analysis Report\n\n")
                                f.write(f"{analysis}")

                    return analysis
                else:
                    logging.error("Empty response from Gemini LLM.")
                    raise RuntimeError("Empty response from Gemini LLM")

        except Exception as e:
            # Check if it's a rate-limit / quota error
            error_str = str(e)
            print(error_str)
            # Handle daily quota exceeded
            if "GenerateRequestsPerDayPerProjectPerModel" in error_str:
                if current_key_index < len(GEMINI_KEYS) - 1:
                    current_key_index += 1
                    new_key = GEMINI_KEYS[current_key_index]
                    logging.warning(f"Quota exceeded. Switching to API key index {current_key_index}.")
                    llm_client = genai.Client(api_key=new_key)
                else:
                    logging.error("All Gemini API keys exhausted.")
                    return None

            # Handle transient errors (RESOURCE_EXHAUSTED, 429, 500)
            elif any(x in error_str for x in ["RESOURCE_EXHAUSTED", "429", "500", "503"]):
                logging.error(f"Attempt {attempt}/{max_retries} - Error during Gemini LLM analysis: {e}")
                if attempt < max_retries:
                    logging.info(f"Retrying in {retry_wait} seconds...")
                    unique_prefix = f"[RequestID: {int(time.time())}-{uuid.uuid4().hex[:8]}]\n"
                    prompt = unique_prefix + DISK_AGENT_ANALYZE_FILE_PROMPT.format(
                            current_date=current_date,
                            file_to_examine=file_to_examine,
                            file_content=file_content
                        )
                    time.sleep(retry_wait)
                else:
                    logging.error(f"Failed to analyze file '{file_to_examine}' after {max_retries} attempts.")
                    return None

# ========== ENTRY POINT ==========
import argparse

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Forensic folder scanning agent.")
    parser.add_argument(
        "-f", "--folder",
        type=str,
        default="/home",
        help="Starting folder path to analyze (default: /home)"
    )
    parser.add_argument(
        "-i", "--image",
        type=str,
        required=True,
        help="Path to HDD image (required)"
    )
    parser.add_argument(
        "-o", "--output",
        type=str,
        help="Optional path to save the final LLM analysis (LLM, Observation, Final Answer)"
    )
    parser.add_argument(
        "-e", "--examine",
        type=str,
        help="Optional path to a file whose content will be examined by the agent"
    )
    parser.add_argument(
        "-l", "--local",
        action="store_true",
        help="Run the agent in local mode (default: False)"
    )
    args = parser.parse_args()

    if args.examine:
        file_to_examine = args.examine
        HDD_IMAGE_PATH = args.image

        try:
            # Use cat_file.py to get the file contents for the agent to analyze
            result = subprocess.run(
                [PYTHON_BIN, SCRIPT2_PATH, HDD_IMAGE_PATH, file_to_examine],
                capture_output=True,
                text=True,
                timeout=SCRIPT_TIMEOUT
            )
            file_content = result.stdout.strip()

            if result.returncode != 0:
                logging.warning(f"Error while reading file '{file_to_examine}': {result.stderr.strip()}")
                file_content = f"Error reading file: {result.stderr.strip()}"
            else:
                analysis = analyze_file_with_llm(
                    file_to_examine=file_to_examine,
                    file_content=file_content,
                    current_date=datetime.now().strftime("%Y-%m-%d"),
                    args=args,
                    MAX_TPM=MAX_TPM,
                    GEMINI_MODEL_NAME=GEMINI_MODEL_NAME,
                    SHOW_NOT_SUSPICIOUS=SHOW_NOT_SUSPICIOUS,
                    retry_wait=RETRY_WAIT,
                    max_retries=MAX_RETRIES,
                    local=args.local
                )
        except Exception as e:
            logging.error(f"Failed to examine file '{file_to_examine}': {str(e)}")
    else:
        STARTING_FOLDER = args.folder
        HDD_IMAGE_PATH = args.image

        try:
            args_list = [HDD_IMAGE_PATH, STARTING_FOLDER]

            # Scanning the content of the starting folder with the folder_scan.py script
            result = subprocess.run(
                [PYTHON_BIN, SCRIPT_PATH, *args_list],
                capture_output=True,
                text=True,
                timeout=SCRIPT_TIMEOUT
            )
            directory_listing = f"starting folder: '{STARTING_FOLDER}'\n" + result.stdout.strip()

            if result.returncode != 0:
                logging.warning(f"Error while listing directory: {result.stderr.strip()}")
            else:
                run_agent(directory_listing, args.local)

        except Exception as e:
            logging.error(f"Failed to list '{STARTING_FOLDER}' directory: {str(e)}")
            directory_listing = f"Error: Could not read '{STARTING_FOLDER}' directory."
