#!/usr/bin/env python3
"""
Barebone DFVFS file system navigator
"""

import sys
import logging

from dfvfs.lib import definitions
from dfvfs.helpers import volume_scanner, source_scanner
from plaso.cli.storage_media_tool import StorageMediaToolVolumeScanner, StorageMediaToolVolumeScannerOptions

from dfvfs.resolver import resolver
from dfvfs.path import factory as path_spec_factory
import stat
import time
from datetime import datetime

def format_date_custom(mtime):
    """Format time to YYYY-MON-DD with month as short name (localized)."""
    try:
        dt_str = mtime.CopyToDateTimeString()  # "2025-09-28 10:32:56.000000"
        dt = datetime.strptime(dt_str.split('.')[0], "%Y-%m-%d %H:%M:%S")
        return dt.strftime("%Y-%b-%d")  # e.g. "2025-Sep-28" or "2025-Gen-28"
    except Exception:
        return "???"


def get_windows_permissions(stat_attr):
    # stat_attr may be None
    if not stat_attr:
        return '----------'
    # Windows: just use directory/file indicators
    if stat_attr.type == 'directory':
        return 'd---------'  # d for directory
    elif stat_attr.type == 'file':
        return '---------'
    elif stat_attr.type == 'link':
        return 'l--------'
    else:
        return '?--------'

def ls_like_windows(directory_entry):
    col_perm, col_uid, col_gid = 12, 6, 6
    col_size, col_time = 12, 20
    header = (
        f"{'PERMISSIONS':<{col_perm}} {'UID':>{col_uid}} "
        f"{'GID':>{col_gid}} {'SIZE':>{col_size}} "
        f"{'LAST_MODIFIED':<{col_time}} NAME"
    )
    print(header)

    entries = list(directory_entry.sub_file_entries)

    if not entries:
        print("_The Directory is empty_")
        return

    for entry in entries:
        stat_attr = entry.GetStatAttribute()

        perms = get_windows_permissions(stat_attr)
        uid = stat_attr.owner_identifier if stat_attr and stat_attr.owner_identifier is not None else '?'
        gid = stat_attr.group_identifier if stat_attr and stat_attr.group_identifier is not None else '?'
        size = stat_attr.size if stat_attr and stat_attr.size is not None else 0

        # Get modification time
        mtime = getattr(entry, "modification_time", None)
        
        if mtime:
            full_str = mtime.CopyToDateTimeString()
            mtime_str = full_str.split('.')[0]
        else:
            mtime_str = "???"

        print(f"{perms:<{col_perm}} {uid:>{col_uid}} {gid:>{col_gid}} {size:>{col_size}} {mtime_str:<{col_time}} {entry.name}")
        
def ls_path_windows(fs, path_spec, path="/"):
    from dfvfs.path import factory as path_spec_factory
    root_ps = path_spec_factory.Factory.NewPathSpec(
        fs.type_indicator, location=path, parent=path_spec
    )
    dir_entry = fs.GetFileEntryByPathSpec(root_ps)

    if dir_entry and dir_entry.IsDirectory():
        ls_like_windows(dir_entry)
    else:
        print(f"Cannot open '{path}' as a directory.")

def find_windows_root(scan_node, depth=0):
    indent = "  " * depth
    if scan_node.type_indicator in ('NTFS', 'FAT'):
        if is_windows_root(scan_node):
            #print(f"{indent}[DEBUG] Windows root filesystem found!")
            return scan_node

    for child in scan_node.sub_nodes:
        found = find_windows_root(child, depth + 1)
        if found:
            return found

    return None

def is_windows_root(scan_node):
    try:
        fs = resolver.Resolver.OpenFileSystem(scan_node.path_spec)
        if fs.type_indicator in ('NTFS', 'FAT'):
            # Check for presence of typical system directories
            root_dir = fs.GetFileEntryByPathSpec(scan_node.path_spec)
            if root_dir:
                entries = [entry.name.lower() for entry in root_dir.sub_file_entries]
                if 'windows' in entries or 'program files' in entries or 'users' in entries:
                    return True
        return False
    except Exception as e:
        print(f"[ERROR] Exception while checking scan_node: {e}")
        return False


def ls_like(directory_entry):
    """Emulates 'ls -l' for a dfVFS FileEntry directory, with aligned columns."""
    # Column widths
    col_perm = 12
    col_uid = 6
    col_gid = 6
    col_size = 12
    col_time = 20

    # Header
    print(f"{'PERMISSIONS':<{col_perm}} {'UID':>{col_uid}} {'GID':>{col_gid}} {'SIZE':>{col_size}} {'LAST_MODIFIED':<{col_time}} NAME")

    for entry in directory_entry.sub_file_entries:
        # Retrieve stat attribute
        stat_attr = entry.GetStatAttribute()
        #print("== StatAttribute fields ==")

        # if stat_attr:
        #     for attr_name in dir(stat_attr):
        #         try:
        #             value = getattr(stat_attr, attr_name)
        #         except Exception as e:
        #             value = f"<error: {e}>"
        #         print(f"{attr_name}: {value!r}")
        # else:
        #     print("No stat attribute available (None)")

        if stat_attr is None:
            print(f"{'?':<{col_perm}} {'?':>{col_uid}} {'?':>{col_gid}} {'?':>{col_size}} {'?':<{col_time}} {entry.name}")
            continue

        # Permissions
        mode = getattr(stat_attr, 'mode', 0)
        perms = stat.filemode(mode)

        # UID & GID
        owner = getattr(stat_attr, 'group_identifier', '?')
        group = getattr(stat_attr, 'owner_identifier', '?')

        # File size
        size = getattr(stat_attr, 'size', 0)

        # Modification time
        mtime = None
        if hasattr(entry, 'modification_time'):
            mtime = entry.modification_time

        if mtime:
            mtime_str = format_date_custom(mtime)
        else:
            mtime_str = "???"

        # Print row with alignment
        print(f"{perms:<{col_perm}} {owner:>{col_uid}} {group:>{col_gid}} {size:>{col_size}} {mtime_str:<{col_time}} {entry.name}")

def ls_path(fs, path_spec, path="/"):
    """Prints ls -l for the specified path inside the fs."""
    from dfvfs.path import factory as path_spec_factory

    root_path_spec = path_spec_factory.Factory.NewPathSpec(
        fs.type_indicator, location=path, parent=path_spec
    )
    dir_entry = fs.GetFileEntryByPathSpec(root_path_spec)

    if dir_entry and dir_entry.IsDirectory():
        ls_like(dir_entry)
    else:
        print(f"'{path}' is not a directory or cannot be opened.")

# -------------------------------
# CONFIGURATION
# -------------------------------
# Verbosity level for logging
LOG_LEVEL = logging.INFO

# -------------------------------
# MAIN CLASSES
# -------------------------------

class MyMediator(volume_scanner.VolumeScannerMediator):
    def GetPartitionIdentifiers(self, volume_system, volume_identifiers):
        # Simple example: always pick the first
        return [volume_identifiers[0]]

    def GetAPFSVolumeIdentifiers(self, volume_system, volume_identifiers):
        # Choose first or customize logic
        return [volume_identifiers[0]]

    def GetLVMVolumeIdentifiers(self, volume_system, volume_identifiers):
        return [volume_identifiers[0]]

    def GetVSSStoreIdentifiers(self, volume_system, volume_identifiers):
        return [volume_identifiers[0]]

    def UnlockEncryptedVolume(self, source_scanner_object, scan_context, locked_scan_node, credentials):
        # Insert logic to unlock encryption
        return True



def is_linux_root(scan_node):
    # print(f"[DEBUG] Checking scan_node: {scan_node}")
    # print(f"[DEBUG] PathSpec: {scan_node.path_spec}")
    # print(f"[DEBUG] Type indicator: {scan_node.type_indicator}")

    try:
        #print("[DEBUG] Opening file system...")
        fs = resolver.Resolver.OpenFileSystem(scan_node.path_spec)
        #print(f"[DEBUG] Opened file system object: {fs}")

        if hasattr(fs, "GetFileEntryByPath"):
            # Works for EXT and some other types
            #print("[DEBUG] Getting root directory by path ('/').")
            root_dir = fs.GetFileEntryByPath("/")
        else:
            # For TSK-based file systems
            #print("[DEBUG] Building PathSpec for root directory in TSK filesystem.")
            root_path_spec = path_spec_factory.Factory.NewPathSpec(
                fs.type_indicator, location="/", parent=scan_node.path_spec
            )
            root_dir = fs.GetFileEntryByPathSpec(root_path_spec)

        #print(f"[DEBUG] Root directory FileEntry: {root_dir}")

        if root_dir is None:
            #print("[DEBUG] No root directory found!")
            return False

        #print("[DEBUG] Listing entries in root directory:")
        entries = []
        for entry in root_dir.sub_file_entries:
            #print(f"  - Found entry: {entry.name}")
            entries.append(entry.name)

        linux_root_dirs = {'usr', 'var', 'etc', 'bin'}
        has_all = linux_root_dirs.issubset(set(entries))
        #print(f"[DEBUG] Has all Linux root dirs? {has_all}")

        return has_all

    except Exception as e:
        print(f"[ERROR] Exception while checking scan_node: {e}")
        return False


def find_linux_root(scan_node, depth=0):
    indent = "  " * depth
    #print(f"{indent}[DEBUG] Visiting scan_node: {scan_node}")
    #print(f"{indent}[DEBUG] Type indicator: {scan_node.type_indicator}")

    if scan_node.type_indicator in ('TSK', 'EXT'):
        #print(f"{indent}[DEBUG] Possible filesystem node, checking if Linux root...")
        if is_linux_root(scan_node):
            #print(f"{indent}[DEBUG] Linux root filesystem found!")
            return scan_node

    for child in scan_node.sub_nodes:
        #print(f"{indent}[DEBUG] Going into child scan_node: {child}")
        found = find_linux_root(child, depth + 1)
        if found:
            return found

    #print(f"{indent}[DEBUG] No Linux root found in this branch.")
    return None

# -------------------------------
# MAIN EXECUTION
# -------------------------------

def main():
    if len(sys.argv) != 3:
        print(f"Usage: {sys.argv[0]} <source_path> <output_path>")
        sys.exit(1)

    logging.basicConfig(level=LOG_LEVEL)
    source_path = sys.argv[1]
    output_path = sys.argv[2]

    windows=False
    linux=False
    
    mediator = MyMediator()
    options = StorageMediaToolVolumeScannerOptions()
    scanner = StorageMediaToolVolumeScanner(mediator=mediator)
    scan_context = scanner.ScanSource(source_path, options, base_path_specs=[])

    if not scan_context:
        print("[ERROR] Failed to scan source.")
        sys.exit(1)

    root_node = scan_context.GetRootScanNode()

    # Try to find Linux root first
    main_partition_node = find_linux_root(root_node)
    if main_partition_node:
        #print(f"[INFO] Main Linux root partition found: {main_partition_node.path_spec}")
        linux=True
    else:
        #print("[INFO] No Linux root partition found, trying Windows...")
        # Try Windows root
        main_partition_node = find_windows_root(root_node)
        if main_partition_node:
            #print(f"[INFO] Windows root partition found: {main_partition_node.path_spec}")
            windows=True
        else:
            #print("[ERROR] No suitable root partition found.")
            sys.exit(1)

    # Open the filesystem
    fs = resolver.Resolver.OpenFileSystem(main_partition_node.path_spec)

    # Get root directory
    if hasattr(fs, "GetFileEntryByPath"):
        # Works for EXT and some other types
        root_dir = fs.GetFileEntryByPath("/") if main_partition_node.type_indicator in ('EXT', 'TSK') else None
    else:
        # For TSK-based file systems
        root_path_spec = path_spec_factory.Factory.NewPathSpec(
            fs.type_indicator, location="/", parent=main_partition_node.path_spec
        )
        root_dir = fs.GetFileEntryByPathSpec(root_path_spec)

    if not root_dir:
        print("[ERROR] Cannot get root directory of the filesystem.")
        sys.exit(1)

    # Now list the root contents using your function
    if(windows):
        ls_path_windows(fs, main_partition_node.path_spec, output_path)
    else:
        ls_path(fs, main_partition_node.path_spec, output_path)

if __name__ == "__main__":
    main()
