import subprocess
import sys
import logging
from pathlib import Path

# ========== GLOBAL CONFIGURABLE VARIABLES ==========
from config import (
    DISK_SCRIPT_PATH,
    MEMORY_SCRIPT_PATH,
    NETWORK_SCRIPT_PATH,
    VENV_PATH,
    ACQUISITION_FOLDER,
    DISK_IMAGE_NAME,
    PCAP_FILE_NAME,
    ASK_MEMORY_SCRIPT,
    PROCESSACQUISITION_LOG_LEVEL,
    FINAL_REPORT_SCRIPT_PATH
)

# ========== LOGGING SETUP ==========
logging.basicConfig(
    level=PROCESSACQUISITION_LOG_LEVEL,
    format="%(asctime)s [%(levelname)s] %(message)s",
    datefmt="%Y-%m-%d %H:%M:%S"
)
logger = logging.getLogger(__name__)

# ========== FUNCTION DEFINITIONS ==========

def run_script(script_path, args=None, use_venv=False, venv_path=None):
    """
    Executes an external Python script with optional arguments and optional virtual environment.
    """
    if args is None:
        args = []

    if use_venv:
        if venv_path is None:
            raise ValueError("venv_path must be specified if use_venv=True")
        python_executable = Path(venv_path) / "bin" / "python3"
        if not python_executable.exists():
            logger.error("Virtualenv python not found at %s", python_executable)
            return False
        cmd = [str(python_executable), script_path] + args
    else:
        cmd = [sys.executable, script_path] + args

    logger.info("Running script: %s", " ".join(cmd))
    result = subprocess.run(cmd)
    if result.returncode != 0:
        logger.error("Script %s failed with exit code %d", script_path, result.returncode)
        return False

    logger.info("Script %s finished successfully", script_path)
    return True


def run_disk_scan(case_folder: Path, artefacts=None, local=False) -> bool:
    """
    Executes the disk forensic scan using the predefined disk scan script.
    """
    if artefacts:
         disk_image_path = artefacts / DISK_IMAGE_NAME
    else:
         disk_image_path = case_folder / ACQUISITION_FOLDER / DISK_IMAGE_NAME
    logger.debug("Disk image path: %s", disk_image_path)
    if local:
        return run_script(DISK_SCRIPT_PATH, args=[str(disk_image_path), str(case_folder), "--local"], use_venv=True, venv_path=VENV_PATH)
    return run_script(DISK_SCRIPT_PATH, args=[str(disk_image_path), str(case_folder)], use_venv=True, venv_path=VENV_PATH)


def run_network_scan(case_folder: Path, artefacts=None) -> bool:
    """
    Executes the network forensic scan using the predefined network scan script.
    """
    if artefacts:
        pcap_path = artefacts / PCAP_FILE_NAME
    else:
        pcap_path = case_folder / ACQUISITION_FOLDER / PCAP_FILE_NAME
    logger.debug("PCAP file path: %s", pcap_path)
    return run_script(NETWORK_SCRIPT_PATH, args=["--case-folder", str(case_folder), str(pcap_path)], use_venv=True, venv_path=VENV_PATH)


def run_memory_scan(case_name: str, case_folder: Path, os_type: str, artefacts=None, local=False) -> bool:
    """
    Executes memory forensic scan + ask_gemini processing.
    """
    if artefacts:
        memory_args = ["--case", case_name, "--os", os_type, "--memory-dump", str(artefacts), "--case-folder", str(case_folder)]
    else:
        memory_args = ["--case", case_name, "--os", os_type]
    logger.info("Starting memory scan for case: %s", case_name)
    success = run_script(MEMORY_SCRIPT_PATH, args=memory_args, use_venv=True, venv_path=VENV_PATH)
    if not success:
        logger.warning("Memory scan failed for case: %s", case_name)
        return False

    memory_output_dir = case_folder / "03_processing" / "memory"
    if not memory_output_dir.exists():
        logger.warning("Memory output directory not found: %s", memory_output_dir)
        return success

    logger.info("Processing memory plugin outputs in: %s", memory_output_dir)
    for plugin_file in memory_output_dir.glob("*.txt"):
        clean_example = memory_output_dir / "clean_examples" / f"clean_{plugin_file.name}"
        if local:
            args = ["--input", str(plugin_file), "--case_folder", str(case_folder), "--local"]
        else:
            args = ["--input", str(plugin_file), "--case_folder", str(case_folder)]
        if clean_example.exists():
            args.extend(["--clean", str(clean_example)])
            logger.debug("Using clean example: %s", clean_example)

        run_script(ASK_MEMORY_SCRIPT, args=args, use_venv=False)

    # clean rate limiter json and lock
    for json_file in [Path("rate_limiter.json"), Path("rate_limiter.lock")]:
        if json_file.exists():
            json_file.unlink()
            logger.info("Deleted temporary file: %s", json_file)

    return success


def main():
    """
    Entry point of the script.
    """
    import argparse
    parser = argparse.ArgumentParser(description="Run forensic scan scripts")
    parser.add_argument("case_folder", help="Path to case folder for forensic analysis")
    parser.add_argument("--os", type=str, choices=["windows", "linux"], required=True, help="Operating system type of the memory dump")
    parser.add_argument("--artefacts", type=str, help="Path to artefacts directory for processing (optional)")
    parser.add_argument("--local", action="store_true", help="Run AI locally without internet connection")
    args = parser.parse_args()

    artefacts = Path(args.artefacts) if args.artefacts else None
    case_folder = Path(args.case_folder).resolve()
    case_name = case_folder.name
    os_type = args.os
    logger.info("Starting forensic acquisition for case: %s", case_name)

    if not run_disk_scan(case_folder, artefacts, local=args.local):
        cont = input("Disk scan failed. Continue anyway? (y/N): ").strip().lower()
        if cont != "y":
            logger.info("Exiting due to disk scan failure")
            return

    if not run_memory_scan(case_name, case_folder, os_type, artefacts, local=args.local):
        cont = input("Memory scan failed. Continue anyway? (y/N): ").strip().lower()
        if cont != "y":
            logger.info("Exiting due to memory scan failure")
            return

    if not run_network_scan(case_folder, artefacts):
        cont = input("Network scan failed. Continue anyway? (y/N): ").strip().lower()
        if cont != "y":
            logger.info("Exiting due to network scan failure")
            return

    logger.info("Forensic acquisition completed successfully for case: %s", case_name)

    logger.info("Creating final report...")
    if args.local:
        run_script(FINAL_REPORT_SCRIPT_PATH, args=[str(case_folder), "--local"], use_venv=True, venv_path=VENV_PATH)
    else:
        run_script(FINAL_REPORT_SCRIPT_PATH, args=[str(case_folder)], use_venv=True, venv_path=VENV_PATH)


if __name__ == "__main__":
    main()