import streamlit as st
import pandas as pd
import io
import base64
import numpy as np
import numpy_financial as npf
import altair as alt
from fpdf import FPDF # NOTE: You need to install the fpdf2 library (`pip install fpdf2`)
import math

st.set_page_config(layout="wide")

# Function to convert image to Base64
def get_image_as_base64(file_path):
    try:
        with open(file_path, "rb") as f:
            return base64.b64encode(f.read()).decode()
    except FileNotFoundError:
        return None # Return None if file not found

# Function to create the PDF report (placeholder, can be expanded later)
def create_pdf(general_info, selected_feeders):
    pdf = FPDF()
    pdf.add_page()
    
    # Title
    pdf.set_font("Helvetica", "B", 16)
    pdf.cell(0, 10, "Value Creation Tool - Report", 0, 1, "C")
    pdf.ln(10)

    # General Information Section
    pdf.set_font("Helvetica", "B", 12)
    pdf.cell(0, 10, "General Information", 0, 1)
    pdf.set_font("Helvetica", "", 10)
    for key, value in general_info.items():
        pdf.cell(0, 8, f"{key}: {value}", 0, 1)
    pdf.ln(10)

    # SCG Deployment Plan Section
    pdf.set_font("Helvetica", "B", 12)
    pdf.cell(0, 10, "SCG Deployment Plan", 0, 1)
    pdf.set_font("Helvetica", "", 10)
    if selected_feeders:
        pdf.multi_cell(0, 8, "Smart Cable Guard is deployed on the following feeders:")
        for feeder in selected_feeders:
            pdf.cell(10) # Indent
            pdf.cell(0, 8, f"- {feeder}", 0, 1)
    else:
        pdf.cell(0, 8, "No Smart Cable Guard systems are deployed in the current selection.", 0, 1)
    
    return pdf.output(dest='S').encode('latin-1')


# Convert your images to Base64
logo_base66 = get_image_as_base64("logo.jpg")
dnv_logo_base66 = get_image_as_base64("DNV_GL_logo.PNG")

# Define a common height for your images
COMMON_IMAGE_HEIGHT = 80

# --- Custom CSS for right-aligning titles and bigger font for expander titles ---
st.markdown(f"""
<style>
/* Style for the main app title "Nexans Value Creation Tool" */
h1.css-1dp5vir {{ /* This targets the h1 element specifically for the main title */
    text-align: right;
    font-size: 2.8em; /* Significantly larger font for the main title */
    font-weight: bold;
}}

/* Style for expander titles (the text you click to expand) */
/* This targets the <p> tag inside the expander header, which holds the title text */
div[data-testid="stExpander"] .stExpanderToggle p {{
    text-align: right;
    font-size: 2.0em; /* Larger font for expander titles, like a prominent header */
    font-weight: bold; /* Make it bold to stand out as a main header */
    margin-bottom: 0.5rem; /* Add some space below the expander title if needed */
}}

/* Ensure subheaders (like "Grid Topology", "Weak Spots") are still smaller and left-aligned */
h3 {{
    font-size: 1.5em; /* A good size for subheaders, larger than body text but smaller than main headers */
    text-align: left; /* Keep subheaders left-aligned */
    margin-top: 1.5rem; /* Space above subheaders */
    margin-bottom: 1rem; /* Space below subheaders */
}}

/* CSS for images to control height */
.logo-image {{
    height: {COMMON_IMAGE_HEIGHT}px; /* Set desired height */
    width: auto; /* Maintain aspect ratio */
    display: block; /* Ensures it behaves like a block element for centering/alignment */
    margin-left: auto; /* For right alignment in column */
    margin-right: 0; /* For right alignment in column */
}}

/* Custom CSS for right-aligning the checkbox */
.stCheckbox > label > div[data-testid="stMarkdownContainer"] {{
    display: flex;
    justify-content: flex-end; /* Aligns content (checkbox label) to the right */
    width: 100%;
}}
</style>
""", unsafe_allow_html=True)


# Initialize session state for page navigation if not already set
if 'show_smart_cable_guard_ui' not in st.session_state:
    st.session_state.show_smart_cable_guard_ui = False


# Initialize session state for country-specific data retrieval checkboxes
if 'retrieve_country_data_sections' not in st.session_state:
    st.session_state.retrieve_country_data_sections = {
        'downtime_faults_breakdown': False,
        'material_excavation_costs': False,
        'weak_spots_section': False,
        'faults_section': False,
        'financial_data_section': False,
        'regulatory_penalties_section': False
    }

# --- Define Country-Specific Data ---
default_country_data = {
    # Downtime & Faults Breakdown
    "weak_spots_in_cable_system": 93,
    "faults_in_cable_system": 89,
    "intrinsic_cable_faults": 92,

    # Material and Excavation Costs
    "single_joint_cost": 300.0,
    "mv_cable_cost": 13,
    "excavation_rate": 85.0,

    # Weak Spots Restoration Costs
    "urban_weak_spot_restoration_cost": 7000.0,
    "rural_weak_spot_restoration_cost": 8000.0,
    "joints_per_weak_spot_repair": 2,
    "weak_spot_cable_section_to_replace": 6.0,
    "urban_weak_spot_administrative_cost": 1500.0,
    "urban_weak_spot_repair_cost": 2000.0,
    "rural_weak_spot_administrative_cost": 900.0,
    "rural_weak_spot_repair_cost": 1600.0,
    "urban_weak_spot_localization_cost": 600.0,
    "rural_weak_spot_localization_cost": 500.0,
    "weak_spot_localization_time": 0.5,
    "weak_spot_administrative_time": 0.2,
    "weak_spot_excavation_time": 0.45,
    "weak_spot_repair_time": 0.3,

    # Faults Restoration Costs
    "urban_fault_restoration_cost": 10000.0,
    "rural_fault_restoration_cost": 12000.0,
    "joints_per_fault_repair": 2,
    "fault_cable_section_to_replace": 15.0,
    "urban_fault_administrative_cost": 2000.0,
    "urban_fault_repair_cost": 3000.0,
    "rural_fault_administrative_cost": 1200.0,
    "rural_fault_repair_cost": 2400.0,
    "urban_fault_localization_cost": 1200.0,
    "rural_fault_localization_cost": 900.0,
    "fault_localization_time": 0.15,
    "fault_administrative_time": 0.45,
    "fault_excavation_time": 0.25,
    "fault_repair_time": 0.15,

    # Financial Data
    "client_rate_of_return": 0.05,
    "client_ens_cost": 0.0,

    # Regulatory Penalties
    "saidi_reduction_compensation": 0.0,
    "saifi_reduction_compensation": 0.0,
    "compensation_for_residential": 0.005,
    "compensation_for_industrial": 0.0,
}

# The actual dataset that will hold country-specific values
country_specific_data_repo = {country: default_country_data for country in [
    "United States", "Canada", "Mexico", "Brazil", "Argentina",
    "United Kingdom", "France", "Germany", "Italy", "Spain",
    "Netherlands", "Belgium", "Sweden", "Norway", "Denmark",
    "Finland", "Switzerland", "Austria", "Poland", "Czech Republic",
    "Australia", "New Zealand", "Japan", "China", "India",
    "South Korea", "Singapore", "Malaysia", "Indonesia", "Thailand",
    "Vietnam", "South Africa", "Egypt", "Nigeria", "United Arab Emirates",
    "Saudi Arabia", "Qatar", "Turkey", "Russia", "Ukraine",
    "Colombia", "Chile", "Peru", "Other"
]}

# Initialize all session state variables from the default data dict
if 'current_country_defaults' not in st.session_state:
    st.session_state.current_country_defaults = default_country_data

for key, value in default_country_data.items():
    if key not in st.session_state:
        st.session_state[key] = value

# Initialize keys for radio buttons
if 'weak_spot_total_cost_known' not in st.session_state:
    st.session_state.weak_spot_total_cost_known = "yes"
if 'fault_total_cost_known' not in st.session_state:
    st.session_state.fault_total_cost_known = "yes"

# Initialize for results tables and other states
if 'electrical_system_data' not in st.session_state:
    st.session_state.electrical_system_data = []
if 'electrical_system_data_df' not in st.session_state: # FIX: Initialize the df version
    st.session_state.electrical_system_data_df = pd.DataFrame()
if 'current_cost_df' not in st.session_state:
    st.session_state.current_cost_df = pd.DataFrame()
if 'lowest_cost_df' not in st.session_state:
    st.session_state.lowest_cost_df = pd.DataFrame()
if 'client_choice_df' not in st.session_state:
    st.session_state.client_choice_df = pd.DataFrame()
if 'results_table_data_df' not in st.session_state:
    st.session_state.results_table_data_df = pd.DataFrame()
if 'financial_results_df' not in st.session_state:
    st.session_state.financial_results_df = pd.DataFrame()

# --- GLOBAL VARIABLES ---
scg_unit_system_cost = 10000
scg_annual_suscription = 1500
scg_operational_lifetime = 10

# --- CALCULATION FUNCTION ---
def perform_all_calculations():
    # Guard clause: Do nothing if there's no input data
    if not st.session_state.electrical_system_data:
        return

    # Create a DataFrame from the uploaded data
    raw_df = pd.DataFrame(st.session_state.electrical_system_data)

    # Get parameters from session state
    faults_in_cable_system_perc = st.session_state.faults_in_cable_system / 100
    weak_spots_in_cable_system_perc = st.session_state.weak_spots_in_cable_system / 100
    intrinsic_cable_faults_perc = st.session_state.intrinsic_cable_faults / 100

    # --- 5.1. current_cost CALCULATION ---
    current_cost = raw_df.copy()
    current_cost['SCG Installed (YES = 1 or NO = 0)'] = 0

    # Python translation of formulas for current_cost
    current_cost['SCG Systems Needed'] = 0 # As SCG Installed is 0
    current_cost['Annual Faults not related to Cable System (faults/year)'] = current_cost['Annual Fault Frequency (faults/year)'] * (1 - faults_in_cable_system_perc)
    current_cost['Annual Weak-Spot Restorations not related to Cable System (restorations/year)'] = current_cost['Annual Weak-Spot Restorations (restorations/year)'] * (1 - weak_spots_in_cable_system_perc)
    current_cost['New Annual Faults related to Cable System (faults/year)'] = current_cost['Annual Fault Frequency (faults/year)'] * faults_in_cable_system_perc
    current_cost['New Annual Weak-Spot Restorations related to Cable System (restorations/year)'] = current_cost['Annual Weak-Spot Restorations (restorations/year)'] * weak_spots_in_cable_system_perc

    # Weak-spot Restoration Cost
    urban_ws_breakdown_cost = ((st.session_state.single_joint_cost * st.session_state.joints_per_weak_spot_repair) + (st.session_state.mv_cable_cost * st.session_state.weak_spot_cable_section_to_replace)) + st.session_state.urban_weak_spot_localization_cost + (current_cost['Cable-System Weak-Spot Restoration Time (min)'] * st.session_state.weak_spot_excavation_time * st.session_state.excavation_rate / 60) + st.session_state.urban_weak_spot_administrative_cost + st.session_state.urban_weak_spot_repair_cost
    rural_ws_breakdown_cost = ((st.session_state.single_joint_cost * st.session_state.joints_per_weak_spot_repair) + (st.session_state.mv_cable_cost * st.session_state.weak_spot_cable_section_to_replace)) + st.session_state.rural_weak_spot_localization_cost + (current_cost['Cable-System Weak-Spot Restoration Time (min)'] * st.session_state.weak_spot_excavation_time * st.session_state.excavation_rate / 60) + st.session_state.rural_weak_spot_administrative_cost + st.session_state.rural_weak_spot_repair_cost
    ws_cost_if_known = np.where(current_cost['Area Type (Urban = U, Rural = R)'] == 'U', st.session_state.urban_weak_spot_restoration_cost, st.session_state.rural_weak_spot_restoration_cost)
    ws_cost_if_not_known = np.where(current_cost['Area Type (Urban = U, Rural = R)'] == 'U', urban_ws_breakdown_cost, rural_ws_breakdown_cost)
    current_cost['Weak-spot Restoration Cost (€/restoration)'] = np.where(st.session_state.weak_spot_total_cost_known == 'yes', ws_cost_if_known, ws_cost_if_not_known)

    # Fault Restoration Cost
    urban_f_breakdown_cost = ((st.session_state.single_joint_cost * st.session_state.joints_per_fault_repair) + (st.session_state.mv_cable_cost * st.session_state.fault_cable_section_to_replace)) + st.session_state.urban_fault_localization_cost + (current_cost['Cable-System Fault Restoration Time (min)'] * st.session_state.fault_excavation_time * st.session_state.excavation_rate / 60) + st.session_state.urban_fault_administrative_cost + st.session_state.urban_fault_repair_cost
    rural_f_breakdown_cost = ((st.session_state.single_joint_cost * st.session_state.joints_per_fault_repair) + (st.session_state.mv_cable_cost * st.session_state.fault_cable_section_to_replace)) + st.session_state.rural_fault_localization_cost + (current_cost['Cable-System Fault Restoration Time (min)'] * st.session_state.fault_excavation_time * st.session_state.excavation_rate / 60) + st.session_state.rural_fault_administrative_cost + st.session_state.rural_fault_repair_cost
    f_cost_if_known = np.where(current_cost['Area Type (Urban = U, Rural = R)'] == 'U', st.session_state.urban_fault_restoration_cost, st.session_state.rural_fault_restoration_cost)
    f_cost_if_not_known = np.where(current_cost['Area Type (Urban = U, Rural = R)'] == 'U', urban_f_breakdown_cost, rural_f_breakdown_cost)
    current_cost['Fault Restoration Cost (€/restoration)'] = np.where(st.session_state.fault_total_cost_known == 'yes', f_cost_if_known, f_cost_if_not_known)

    current_cost['Annual Cable-System Weak-spot Restoration Cost (€/year)'] = current_cost['Weak-spot Restoration Cost (€/restoration)'] * current_cost['New Annual Weak-Spot Restorations related to Cable System (restorations/year)']
    current_cost['Annual Cable-System Fault Restoration Cost (€/year)'] = current_cost['Fault Restoration Cost (€/restoration)'] * current_cost['New Annual Faults related to Cable System (faults/year)']
    
    downtime_no_redundancy = (current_cost['Cable-System Fault Restoration Time (min)'] * current_cost['New Annual Faults related to Cable System (faults/year)']) + (current_cost['Cable-System Weak-Spot Restoration Time (min)'] * current_cost['New Annual Weak-Spot Restorations related to Cable System (restorations/year)'] )
    downtime_with_redundancy = current_cost['New Annual Faults related to Cable System (faults/year)'] * current_cost['Power Rerouting Time (min)']
    current_cost['Downtime due to Cable System Issues per Year (min/year)'] = np.where(current_cost['Redundancy (YES = 1 or NO = 0)'] == 1, downtime_with_redundancy, downtime_no_redundancy)

    current_cost['Experienced Downtime for Residential Users due to Cable System Issues per Year (min-users/year)'] = current_cost['Downtime due to Cable System Issues per Year (min/year)'] * current_cost['Number of Residential Users']
    current_cost['Experienced Downtime for Industrial Users due to Cable System Issues per Year (min-users/year)'] = current_cost['Downtime due to Cable System Issues per Year (min/year)'] * current_cost['Number of Industrial Users']
    current_cost['Total Experienced Downtime due to Cable System Issues per Year (min-users/year)'] = current_cost['Downtime due to Cable System Issues per Year (min/year)'] * (current_cost['Number of Residential Users'] + current_cost['Number of Industrial Users'])

    interruptions_no_redundancy = current_cost['New Annual Faults related to Cable System (faults/year)'] + current_cost['New Annual Weak-Spot Restorations related to Cable System (restorations/year)']
    interruptions_with_redundancy = current_cost['New Annual Faults related to Cable System (faults/year)']
    current_cost['Interruptions due to Cable System Issues per Year (faults/year)'] = np.where(current_cost['Redundancy (YES = 1 or NO = 0)'] == 1, interruptions_with_redundancy, interruptions_no_redundancy)
    
    current_cost['Experienced Interruptions for Residential Users due to Cable System Issues per Year (int-users/year)'] = current_cost['Interruptions due to Cable System Issues per Year (faults/year)'] * current_cost['Number of Residential Users']
    current_cost['Experienced Interruptions for Industrial Users due to Cable System Issues per Year (int-users/year)'] = current_cost['Interruptions due to Cable System Issues per Year (faults/year)'] * current_cost['Number of Industrial Users']
    current_cost['Total Experienced Interruptions due to Cable System Issues per Year (int-users/year)'] = current_cost['Interruptions due to Cable System Issues per Year (faults/year)'] * (current_cost['Number of Residential Users'] + current_cost['Number of Industrial Users'])
    
    st.session_state.current_cost_df = current_cost

    # --- 5.2. lowest_cost CALCULATION ---
    lowest_cost = raw_df.copy()
    lowest_cost['SCG Installed (YES = 1 or NO = 0)'] = 1
    
    # Formulas for lowest_cost
    systems_per_type = (lowest_cost['MV Cable Length (m)'] * (lowest_cost['EPR (%)']/100/5000)) + (lowest_cost['MV Cable Length (m)'] * (lowest_cost['XLPE (%)']/100/5000)) + (lowest_cost['MV Cable Length (m)'] * (lowest_cost['PILC (%)']/100/10000))
    lowest_cost['SCG Systems Needed'] = np.ceil(systems_per_type)
    lowest_cost['Annual Faults not related to Cable System (faults/year)'] = lowest_cost['Annual Fault Frequency (faults/year)'] * (1 - faults_in_cable_system_perc)
    lowest_cost['Annual Weak-Spot Restorations not related to Cable System (restorations/year)'] = lowest_cost['Annual Weak-Spot Restorations (restorations/year)'] * (1 - weak_spots_in_cable_system_perc)
    lowest_cost['New Annual Faults related to Cable System (faults/year)'] = lowest_cost['Annual Fault Frequency (faults/year)'] * faults_in_cable_system_perc * (1 - intrinsic_cable_faults_perc)
    lowest_cost['New Annual Weak-Spot Restorations related to Cable System (restorations/year)'] = (lowest_cost['Annual Weak-Spot Restorations (restorations/year)'] * weak_spots_in_cable_system_perc) + (lowest_cost['Annual Fault Frequency (faults/year)'] * faults_in_cable_system_perc * intrinsic_cable_faults_perc)
    
    # Restoration costs with SCG (localization cost subtracted)
    ws_cost_if_known_scg = np.where(lowest_cost['Area Type (Urban = U, Rural = R)'] == 'U', st.session_state.urban_weak_spot_restoration_cost - st.session_state.urban_weak_spot_localization_cost, st.session_state.rural_weak_spot_restoration_cost - st.session_state.rural_weak_spot_localization_cost)
    ws_cost_if_not_known_scg_urban = ((st.session_state.single_joint_cost * st.session_state.joints_per_weak_spot_repair) + (st.session_state.mv_cable_cost * st.session_state.weak_spot_cable_section_to_replace)) + (lowest_cost['Cable-System Weak-Spot Restoration Time (min)'] * st.session_state.weak_spot_excavation_time * st.session_state.excavation_rate / 60) + st.session_state.urban_weak_spot_administrative_cost + st.session_state.urban_weak_spot_repair_cost
    ws_cost_if_not_known_scg_rural = ((st.session_state.single_joint_cost * st.session_state.joints_per_weak_spot_repair) + (st.session_state.mv_cable_cost * st.session_state.weak_spot_cable_section_to_replace)) + (lowest_cost['Cable-System Weak-Spot Restoration Time (min)'] * st.session_state.weak_spot_excavation_time * st.session_state.excavation_rate / 60) + st.session_state.rural_weak_spot_administrative_cost + st.session_state.rural_weak_spot_repair_cost
    ws_cost_if_not_known_scg = np.where(lowest_cost['Area Type (Urban = U, Rural = R)'] == 'U', ws_cost_if_not_known_scg_urban, ws_cost_if_not_known_scg_rural)
    lowest_cost['Weak-spot Restoration Cost (€/restoration)'] = np.where(st.session_state.weak_spot_total_cost_known == 'yes', ws_cost_if_known_scg, ws_cost_if_not_known_scg)

    f_cost_if_known_scg = np.where(lowest_cost['Area Type (Urban = U, Rural = R)'] == 'U', st.session_state.urban_fault_restoration_cost - st.session_state.urban_fault_localization_cost, st.session_state.rural_fault_restoration_cost - st.session_state.rural_fault_localization_cost)
    f_cost_if_not_known_scg_urban = ((st.session_state.single_joint_cost * st.session_state.joints_per_fault_repair) + (st.session_state.mv_cable_cost * st.session_state.fault_cable_section_to_replace)) + (lowest_cost['Cable-System Fault Restoration Time (min)'] * st.session_state.fault_excavation_time * st.session_state.excavation_rate / 60) + st.session_state.urban_fault_administrative_cost + st.session_state.urban_fault_repair_cost
    f_cost_if_not_known_scg_rural = ((st.session_state.single_joint_cost * st.session_state.joints_per_fault_repair) + (st.session_state.mv_cable_cost * st.session_state.fault_cable_section_to_replace)) + (lowest_cost['Cable-System Fault Restoration Time (min)'] * st.session_state.fault_excavation_time * st.session_state.excavation_rate / 60) + st.session_state.rural_fault_administrative_cost + st.session_state.rural_fault_repair_cost
    f_cost_if_not_known_scg = np.where(lowest_cost['Area Type (Urban = U, Rural = R)'] == 'U', f_cost_if_not_known_scg_urban, f_cost_if_not_known_scg_rural)
    lowest_cost['Fault Restoration Cost (€/restoration)'] = np.where(st.session_state.fault_total_cost_known == 'yes', f_cost_if_known_scg, f_cost_if_not_known_scg)
    
    lowest_cost['Annual Cable-System Weak-spot Restoration Cost (€/year)'] = lowest_cost['Weak-spot Restoration Cost (€/restoration)'] * lowest_cost['New Annual Weak-Spot Restorations related to Cable System (restorations/year)']
    lowest_cost['Annual Cable-System Fault Restoration Cost (€/year)'] = lowest_cost['Fault Restoration Cost (€/restoration)'] * lowest_cost['New Annual Faults related to Cable System (faults/year)']
    
    downtime_factor_fault = st.session_state.fault_administrative_time + st.session_state.fault_excavation_time + st.session_state.fault_repair_time
    downtime_factor_ws = st.session_state.weak_spot_administrative_time + st.session_state.weak_spot_excavation_time + st.session_state.weak_spot_repair_time
    downtime_no_redundancy_scg = (lowest_cost['Cable-System Fault Restoration Time (min)'] * downtime_factor_fault * lowest_cost['New Annual Faults related to Cable System (faults/year)']) + (lowest_cost['Cable-System Weak-Spot Restoration Time (min)'] * downtime_factor_ws * lowest_cost['New Annual Weak-Spot Restorations related to Cable System (restorations/year)'])
    downtime_with_redundancy_scg = lowest_cost['New Annual Faults related to Cable System (faults/year)'] * lowest_cost['Power Rerouting Time (min)']
    lowest_cost['Downtime due to Cable System Issues per Year (min/year)'] = np.where(lowest_cost['Redundancy (YES = 1 or NO = 0)'] == 1, downtime_with_redundancy_scg, downtime_no_redundancy_scg)

    lowest_cost['Experienced Downtime for Residential Users due to Cable System Issues per Year (min-users/year)'] = lowest_cost['Downtime due to Cable System Issues per Year (min/year)'] * lowest_cost['Number of Residential Users']
    lowest_cost['Experienced Downtime for Industrial Users due to Cable System Issues per Year (min-users/year)'] = lowest_cost['Downtime due to Cable System Issues per Year (min/year)'] * lowest_cost['Number of Industrial Users']
    lowest_cost['Total Experienced Downtime due to Cable System Issues per Year (min-users/year)'] = lowest_cost['Downtime due to Cable System Issues per Year (min/year)'] * (lowest_cost['Number of Residential Users'] + lowest_cost['Number of Industrial Users'])
    
    interruptions_no_redundancy_scg = lowest_cost['New Annual Faults related to Cable System (faults/year)'] + lowest_cost['New Annual Weak-Spot Restorations related to Cable System (restorations/year)']
    interruptions_with_redundancy_scg = lowest_cost['New Annual Faults related to Cable System (faults/year)']
    lowest_cost['Interruptions due to Cable System Issues per Year (faults/year)'] = np.where(lowest_cost['Redundancy (YES = 1 or NO = 0)'] == 1, interruptions_with_redundancy_scg, interruptions_no_redundancy_scg)

    lowest_cost['Experienced Interruptions for Residential Users due to Cable System Issues per Year (int-users/year)'] = lowest_cost['Interruptions due to Cable System Issues per Year (faults/year)'] * lowest_cost['Number of Residential Users']
    lowest_cost['Experienced Interruptions for Industrial Users due to Cable System Issues per Year (int-users/year)'] = lowest_cost['Interruptions due to Cable System Issues per Year (faults/year)'] * lowest_cost['Number of Industrial Users']
    lowest_cost['Total Experienced Interruptions due to Cable System Issues per Year (int-users/year)'] = lowest_cost['Interruptions due to Cable System Issues per Year (faults/year)'] * (lowest_cost['Number of Residential Users'] + lowest_cost['Number of Industrial Users'])

    total_users = lowest_cost['Number of Residential Users'].sum() + lowest_cost['Number of Industrial Users'].sum()
    lowest_cost['Contribution to SAIDI Reduction (min/year)'] = (current_cost['Total Experienced Downtime due to Cable System Issues per Year (min-users/year)'] - lowest_cost['Total Experienced Downtime due to Cable System Issues per Year (min-users/year)']) / total_users
    lowest_cost['Contribution to SAIFI Reduction (int/year)'] = (current_cost['Total Experienced Interruptions due to Cable System Issues per Year (int-users/year)'] - lowest_cost['Total Experienced Interruptions due to Cable System Issues per Year (int-users/year)']) / total_users
    lowest_cost['Contribution to SAIDI Reduction Compensation (€/year)'] = lowest_cost['Contribution to SAIDI Reduction (min/year)'] * total_users * st.session_state.saidi_reduction_compensation / scg_operational_lifetime
    
    lowest_cost['Contribution to SAIFI Reduction Compensation (€/year)'] = lowest_cost['Contribution to SAIFI Reduction (int/year)'] * total_users * st.session_state.saifi_reduction_compensation / scg_operational_lifetime
    
    lowest_cost['Downtime Reduction Experienced By Residential Users (min-users/year)'] = current_cost['Experienced Downtime for Residential Users due to Cable System Issues per Year (min-users/year)'] - lowest_cost['Experienced Downtime for Residential Users due to Cable System Issues per Year (min-users/year)']
    lowest_cost['Downtime Reduction Experienced By Industrial Users (min-users/year)'] = current_cost['Experienced Downtime for Industrial Users due to Cable System Issues per Year (min-users/year)'] - lowest_cost['Experienced Downtime for Industrial Users due to Cable System Issues per Year (min-users/year)']
    lowest_cost['Contribution to Downtime Compensation Reduction to Pay to Residential Users (€/year)'] = lowest_cost['Downtime Reduction Experienced By Residential Users (min-users/year)'] * st.session_state.compensation_for_residential
    lowest_cost['Contribution to Downtime Compensation Reduction to Pay to Industrial Users (€/year)'] = lowest_cost['Downtime Reduction Experienced By Industrial Users (min-users/year)'] * st.session_state.compensation_for_industrial
    lowest_cost['Energy Not Supplied Reduction (MWh/year)'] = ((current_cost['Downtime due to Cable System Issues per Year (min/year)'] / 60) - (lowest_cost['Downtime due to Cable System Issues per Year (min/year)'] / 60)) * (lowest_cost['Average Load (kW)'] / 1000)
    lowest_cost['Contribution to Energy Not Supplied Cost Reduction (€/year)'] = lowest_cost['Energy Not Supplied Reduction (MWh/year)'] * st.session_state.client_ens_cost
    lowest_cost['Contribution to Annual Cable-System Weak-spot Restoration Cost Reduction (€/year)'] = current_cost['Annual Cable-System Weak-spot Restoration Cost (€/year)'] - lowest_cost['Annual Cable-System Weak-spot Restoration Cost (€/year)']
    lowest_cost['Contribution to Annual Cable-System Fault Restoration Cost Reduction (€/year)'] = current_cost['Annual Cable-System Fault Restoration Cost (€/year)'] - lowest_cost['Annual Cable-System Fault Restoration Cost (€/year)']
    lowest_cost['New Income (€/year)'] = lowest_cost['Contribution to SAIDI Reduction Compensation (€/year)'] + lowest_cost['Contribution to SAIFI Reduction Compensation (€/year)'] + lowest_cost['Contribution to Downtime Compensation Reduction to Pay to Residential Users (€/year)'] + lowest_cost['Contribution to Downtime Compensation Reduction to Pay to Industrial Users (€/year)'] + lowest_cost['Contribution to Energy Not Supplied Cost Reduction (€/year)'] + lowest_cost['Contribution to Annual Cable-System Weak-spot Restoration Cost Reduction (€/year)'] + lowest_cost['Contribution to Annual Cable-System Fault Restoration Cost Reduction (€/year)']
    
    st.session_state.lowest_cost_df = lowest_cost

    # --- 5.3. client_choice CALCULATION ---
    client_choice = raw_df.copy()
    
    scg_selection = st.session_state.results_table_data_df['SCG Installed (YES = 1 or NO = 0)'].tolist()
    client_choice['SCG Installed (YES = 1 or NO = 0)'] = scg_selection

    systems_per_type_cc = (client_choice['MV Cable Length (m)'] * (client_choice['EPR (%)']/100/5000)) + (client_choice['MV Cable Length (m)'] * (client_choice['XLPE (%)']/100/5000)) + (client_choice['MV Cable Length (m)'] * (client_choice['PILC (%)']/100/10000))
    client_choice['SCG Systems Needed'] = np.where(client_choice['SCG Installed (YES = 1 or NO = 0)'] == 1, np.ceil(systems_per_type_cc), 0)
    
    client_choice['Annual Faults not related to Cable System (faults/year)'] = client_choice['Annual Fault Frequency (faults/year)'] * (1 - faults_in_cable_system_perc)
    client_choice['Annual Weak-Spot Restorations not related to Cable System (restorations/year)'] = client_choice['Annual Weak-Spot Restorations (restorations/year)'] * (1 - weak_spots_in_cable_system_perc)
    
    faults_with_scg = client_choice['Annual Fault Frequency (faults/year)'] * faults_in_cable_system_perc * (1 - intrinsic_cable_faults_perc)
    faults_no_scg = client_choice['Annual Fault Frequency (faults/year)'] * faults_in_cable_system_perc
    client_choice['New Annual Faults related to Cable System (faults/year)'] = np.where(client_choice['SCG Installed (YES = 1 or NO = 0)'] == 1, faults_with_scg, faults_no_scg)
    
    ws_with_scg = (client_choice['Annual Weak-Spot Restorations (restorations/year)'] * weak_spots_in_cable_system_perc) + (client_choice['Annual Fault Frequency (faults/year)'] * faults_in_cable_system_perc * intrinsic_cable_faults_perc)
    ws_no_scg = client_choice['Annual Weak-Spot Restorations (restorations/year)'] * weak_spots_in_cable_system_perc
    client_choice['New Annual Weak-Spot Restorations related to Cable System (restorations/year)'] = np.where(client_choice['SCG Installed (YES = 1 or NO = 0)'] == 1, ws_with_scg, ws_no_scg)

    client_choice['Weak-spot Restoration Cost (€/restoration)'] = np.where(client_choice['SCG Installed (YES = 1 or NO = 0)'] == 1, lowest_cost['Weak-spot Restoration Cost (€/restoration)'], current_cost['Weak-spot Restoration Cost (€/restoration)'])
    client_choice['Fault Restoration Cost (€/restoration)'] = np.where(client_choice['SCG Installed (YES = 1 or NO = 0)'] == 1, lowest_cost['Fault Restoration Cost (€/restoration)'], current_cost['Fault Restoration Cost (€/restoration)'])

    client_choice['Annual Cable-System Weak-spot Restoration Cost (€/year)'] = client_choice['Weak-spot Restoration Cost (€/restoration)'] * client_choice['New Annual Weak-Spot Restorations related to Cable System (restorations/year)']
    client_choice['Annual Cable-System Fault Restoration Cost (€/year)'] = client_choice['Fault Restoration Cost (€/restoration)'] * client_choice['New Annual Faults related to Cable System (faults/year)']

    downtime_no_redundancy_cc_scg = (client_choice['Cable-System Fault Restoration Time (min)'] * downtime_factor_fault * client_choice['New Annual Faults related to Cable System (faults/year)']) + (client_choice['Cable-System Weak-Spot Restoration Time (min)'] * downtime_factor_ws * client_choice['New Annual Weak-Spot Restorations related to Cable System (restorations/year)'])
    downtime_with_redundancy_cc_scg = client_choice['New Annual Faults related to Cable System (faults/year)'] * client_choice['Power Rerouting Time (min)']
    downtime_cc_scg = np.where(client_choice['Redundancy (YES = 1 or NO = 0)'] == 1, downtime_with_redundancy_cc_scg, downtime_no_redundancy_cc_scg)
    downtime_cc_no_scg = np.where(client_choice['Redundancy (YES = 1 or NO = 0)'] == 1, downtime_with_redundancy, downtime_no_redundancy)
    client_choice['Downtime due to Cable System Issues per Year (min/year)'] = np.where(client_choice['SCG Installed (YES = 1 or NO = 0)'] == 1, downtime_cc_scg, downtime_cc_no_scg)

    client_choice['Experienced Downtime for Residential Users due to Cable System Issues per Year (min-users/year)'] = client_choice['Downtime due to Cable System Issues per Year (min/year)'] * client_choice['Number of Residential Users']
    client_choice['Experienced Downtime for Industrial Users due to Cable System Issues per Year (min-users/year)'] = client_choice['Downtime due to Cable System Issues per Year (min/year)'] * client_choice['Number of Industrial Users']
    client_choice['Total Experienced Downtime due to Cable System Issues per Year (min-users/year)'] = client_choice['Downtime due to Cable System Issues per Year (min/year)'] * (client_choice['Number of Residential Users'] + client_choice['Number of Industrial Users'])

    interruptions_no_redundancy_cc_scg = client_choice['New Annual Faults related to Cable System (faults/year)'] + client_choice['New Annual Weak-Spot Restorations related to Cable System (restorations/year)']
    interruptions_with_redundancy_cc_scg = client_choice['New Annual Faults related to Cable System (faults/year)']
    interruptions_cc_scg = np.where(client_choice['Redundancy (YES = 1 or NO = 0)'] == 1, interruptions_with_redundancy_cc_scg, interruptions_no_redundancy_cc_scg)
    interruptions_cc_no_scg = np.where(client_choice['Redundancy (YES = 1 or NO = 0)'] == 1, interruptions_with_redundancy, interruptions_no_redundancy)
    client_choice['Interruptions due to Cable System Issues per Year (faults/year)'] = np.where(client_choice['SCG Installed (YES = 1 or NO = 0)'] == 1, interruptions_cc_scg, interruptions_cc_no_scg)

    client_choice['Experienced Interruptions for Residential Users due to Cable System Issues per Year (int-users/year)'] = client_choice['Interruptions due to Cable System Issues per Year (faults/year)'] * client_choice['Number of Residential Users']
    client_choice['Experienced Interruptions for Industrial Users due to Cable System Issues per Year (int-users/year)'] = client_choice['Interruptions due to Cable System Issues per Year (faults/year)'] * client_choice['Number of Industrial Users']
    client_choice['Total Experienced Interruptions due to Cable System Issues per Year (int-users/year)'] = client_choice['Interruptions due to Cable System Issues per Year (faults/year)'] * (client_choice['Number of Residential Users'] + client_choice['Number of Industrial Users'])

    client_choice['Contribution to SAIDI Reduction (min/year)'] = (current_cost['Total Experienced Downtime due to Cable System Issues per Year (min-users/year)'] - client_choice['Total Experienced Downtime due to Cable System Issues per Year (min-users/year)']) / total_users if total_users > 0 else 0
    client_choice['Contribution to SAIFI Reduction (int/year)'] = (current_cost['Total Experienced Interruptions due to Cable System Issues per Year (int-users/year)'] - client_choice['Total Experienced Interruptions due to Cable System Issues per Year (int-users/year)']) / total_users if total_users > 0 else 0
    client_choice['Contribution to SAIDI Reduction Compensation (€/year)'] = client_choice['Contribution to SAIDI Reduction (min/year)'] * total_users * st.session_state.saidi_reduction_compensation / scg_operational_lifetime
    
    client_choice['Contribution to SAIFI Reduction Compensation (€/year)'] = client_choice['Contribution to SAIFI Reduction (int/year)'] * total_users * st.session_state.saifi_reduction_compensation / scg_operational_lifetime
    
    client_choice['Downtime Reduction Experienced By Residential Users (min-users/year)'] = current_cost['Experienced Downtime for Residential Users due to Cable System Issues per Year (min-users/year)'] - client_choice['Experienced Downtime for Residential Users due to Cable System Issues per Year (min-users/year)']
    client_choice['Downtime Reduction Experienced By Industrial Users (min-users/year)'] = current_cost['Experienced Downtime for Industrial Users due to Cable System Issues per Year (min-users/year)'] - client_choice['Experienced Downtime for Industrial Users due to Cable System Issues per Year (min-users/year)']
    client_choice['Contribution to Downtime Compensation Reduction to Pay to Residential Users (€/year)'] = client_choice['Downtime Reduction Experienced By Residential Users (min-users/year)'] * st.session_state.compensation_for_residential
    client_choice['Contribution to Downtime Compensation Reduction to Pay to Industrial Users (€/year)'] = client_choice['Downtime Reduction Experienced By Industrial Users (min-users/year)'] * st.session_state.compensation_for_industrial
    client_choice['Energy Not Supplied Reduction (MWh/year)'] = ((current_cost['Downtime due to Cable System Issues per Year (min/year)'] / 60) - (client_choice['Downtime due to Cable System Issues per Year (min/year)'] / 60)) * (client_choice['Average Load (kW)'] / 1000)
    client_choice['Contribution to Energy Not Supplied Cost Reduction (€/year)'] = client_choice['Energy Not Supplied Reduction (MWh/year)'] * st.session_state.client_ens_cost
    client_choice['Contribution to Annual Cable-System Weak-spot Restoration Cost Reduction (€/year)'] = current_cost['Annual Cable-System Weak-spot Restoration Cost (€/year)'] - client_choice['Annual Cable-System Weak-spot Restoration Cost (€/year)']
    client_choice['Contribution to Annual Cable-System Fault Restoration Cost Reduction (€/year)'] = current_cost['Annual Cable-System Fault Restoration Cost (€/year)'] - client_choice['Annual Cable-System Fault Restoration Cost (€/year)']
    client_choice['New Income (€/year)'] = client_choice['Contribution to SAIDI Reduction Compensation (€/year)'] + client_choice['Contribution to SAIFI Reduction Compensation (€/year)'] + client_choice['Contribution to Downtime Compensation Reduction to Pay to Residential Users (€/year)'] + client_choice['Contribution to Downtime Compensation Reduction to Pay to Industrial Users (€/year)'] + client_choice['Contribution to Energy Not Supplied Cost Reduction (€/year)'] + client_choice['Contribution to Annual Cable-System Weak-spot Restoration Cost Reduction (€/year)'] + client_choice['Contribution to Annual Cable-System Fault Restoration Cost Reduction (€/year)']
    
    st.session_state.client_choice_df = client_choice

    # --- 5.4. results_table_data CALCULATION ---
    results = pd.DataFrame()
    results['SCG Installed (YES = 1 or NO = 0)'] = client_choice['SCG Installed (YES = 1 or NO = 0)']
    results['SCG Systems Needed'] = client_choice['SCG Systems Needed']
    results['Feeder'] = client_choice['Feeder Name']
    results['Downtime Reduction (min/year)'] = current_cost['Downtime due to Cable System Issues per Year (min/year)'] - client_choice['Downtime due to Cable System Issues per Year (min/year)']
    results['Faults Reduction (faults/year)'] = current_cost['New Annual Faults related to Cable System (faults/year)'] - client_choice['New Annual Faults related to Cable System (faults/year)']
    results['Undelivered Energy Reduction (MWh/year)'] = client_choice['Energy Not Supplied Reduction (MWh/year)']
    results['Savings (€/year)'] = client_choice['New Income (€/year)']
    results['LTE-M Coverage at Both Ends'] = 'YES'
    results['220 V Socket at Both Ends'] = 'YES'
    results['Continuity of the Earth Screen'] = 'YES'
    st.session_state.results_table_data_df = results

    # --- 5.5 financial_results CALCULATION ---
    total_new_income = client_choice['New Income (€/year)'].sum()
    total_scg_systems_needed = client_choice['SCG Systems Needed'].sum()

    cash_flows = []
    cf_0 = total_new_income - (total_scg_systems_needed * scg_unit_system_cost) - (total_scg_systems_needed * scg_annual_suscription)
    cash_flows.append(cf_0)
    cf_subsequent = total_new_income - (total_scg_systems_needed * scg_annual_suscription)
    for _ in range(1, scg_operational_lifetime):
        cash_flows.append(cf_subsequent)

    financial_results_data = {'Year': range(scg_operational_lifetime)}
    financial_results = pd.DataFrame(financial_results_data)
    financial_results['Cash Flow (€)'] = cash_flows
    
    discount_rate = st.session_state.client_rate_of_return
    financial_results['Discounted Cash Flow (€)'] = financial_results['Cash Flow (€)'] / ((1 + discount_rate) ** financial_results['Year'])
    financial_results['Cumulative Discounted Cash Flow (€)'] = financial_results['Discounted Cash Flow (€)'].cumsum()
    
    st.session_state.financial_results_df = financial_results


# --- Top Right Images ---
# ... (Unchanged)
if st.session_state.show_smart_cable_guard_ui:
    col_main_content, col_logo1, col_logo2 = st.columns([0.7, 0.15, 0.15])
    with col_logo1:
        if logo_base66:
            st.markdown(f'<img src="data:image/jpeg;base64,{logo_base66}" class="logo-image">', unsafe_allow_html=True)
    with col_logo2:
        if dnv_logo_base66:
            st.markdown(f'<img src="data:image/png;base64,{dnv_logo_base66}" class="logo-image">', unsafe_allow_html=True)
else:
    col_main_content, col_logo = st.columns([0.8, 0.2])
    with col_logo:
        if logo_base66:
            st.markdown(f'<img src="data:image/jpeg;base64,{logo_base66}" class="logo-image">', unsafe_allow_html=True)


# --- Conditional UI Rendering ---
if not st.session_state.show_smart_cable_guard_ui:
    # ... (Unchanged)
    with col_main_content:
        st.title("Nexans Grid Reliability Portfolio")
        st.write("Please select a product from the Nexans Grid Reliability Portfolio to proceed:")
        col1, col2, col3, col4 = st.columns(4)
        with col1:
            if st.button("SynchroSense", use_container_width=True):
                st.info("SynchroSense selected. (No further action defined for this product yet.)")
        with col2:
            if st.button("Smart Cable Guard", use_container_width=True):
                st.session_state.show_smart_cable_guard_ui = True
                st.rerun()
        with col3:
            if st.button("Infracheck", use_container_width=True):
                st.info("Infracheck selected. (No further action defined for this product yet.)")
        with col4:
            if st.button("Ultracker", use_container_width=True):
                st.info("Ultracker selected. (No further action defined for this product yet.)")

elif st.session_state.show_smart_cable_guard_ui:
    with col_main_content:
        if st.button("← Back to Product Selection"):
            st.session_state.show_smart_cable_guard_ui = False
            st.session_state.clear()
            st.rerun()

        st.title("Nexans Value Creation Tool")
        st.write("This tool helps you evaluate the financial and operational impact of deploying Smart Cable Guard across your grid.")

        with st.expander("General Information"):
            # ... (Unchanged)
            st.write("")
            col1, col2 = st.columns(2)
            with col1:
                st.text_input("Company Name:", key="company_name")
                st.text_input("Name of the user:", key="user_name")
                st.text_input("Email:", key="user_email")
                st.text_input("Phone Number:", key="user_phone")
            with col2:
                countries = list(country_specific_data_repo.keys())
                default_country_index = countries.index("France") if "France" in countries else 0
                country = st.selectbox("Country:", options=countries, index=default_country_index, key="selected_country")
                st.session_state.current_country_defaults = country_specific_data_repo.get(country, default_country_data)
                st.text_input("Role at the Company:", key="user_role")
                industry_types = ["Distribution System Operator", "Industrial Site", "Renewable Energy Provider", "Data Center", "Other"]
                st.selectbox("Industry Type:", options=industry_types, key="industry_type")


        with st.expander("Electrical System Data"):
            # ... (Unchanged)
            st.write("This section is for providing the technical and operational data of your grid. Accurate information is essential for a precise simulation of Smart Cable Guard benefits.")
            st.markdown("---")
            
            st.subheader("Grid Topology")
            column_headers = [
                "Feeder Name", "MV Cable Length (m)", "EPR (%)", "XLPE (%)", "PILC (%)",
                "Number of Joints", "Number of Secondary Substations", "Area Type (Urban = U, Rural = R)",
                "Average Load (kW)", "Number of Residential Users", "Number of Industrial Users",
                "Cable-System Fault Restoration Time (min)", "Annual Fault Frequency (faults/year)",
                "Cable-System Weak-Spot Restoration Time (min)", "Annual Weak-Spot Restorations (restorations/year)",
                "Redundancy (YES = 1 or NO = 0)", "Power Rerouting Time (min)"
            ]
            default_fill_values = {
                "MV Cable Length (m)": 8092.0, "EPR (%)": 40.0, "XLPE (%)": 30.0, "PILC (%)": 30.0,
                "Number of Joints": 55, "Number of Secondary Substations": 5, "Area Type (Urban = U, Rural = R)": 'U',
                "Average Load (kW)": 1250.2, "Number of Residential Users": 640, "Number of Industrial Users": 14,
                "Cable-System Fault Restoration Time (min)": 400, "Annual Fault Frequency (faults/year)": 3.0,
                "Cable-System Weak-Spot Restoration Time (min)": 45, "Annual Weak-Spot Restorations (restorations/year)": 7.0,
                "Redundancy (YES = 1 or NO = 0)": 0, "Power Rerouting Time (min)": 46
            }
            st.write("Please provide your grid topology data. Each row must contain the following 17 values in this order:")
            st.markdown("""
- Feeder Name
- MV Cable Length (m)
- EPR (%)
- XLPE (%)
- PILC (%)
- Number of Joints
- Number of Secondary Substations
- Area Type (Urban = U, Rural = R)
- Average Load (kW)
- Number of Residential Users
- Number of Industrial Users
- Cable-System Fault Restoration Time (min)
- Annual Fault Frequency (faults/year)
- Cable-System Weak-Spot Restoration Time (min)
- Annual Weak-Spot Restorations (restorations/year)
- Redundancy (YES = 1 or NO = 0)
- Power Rerouting Time (min)
""")
            st.write("")
            
            col_step1, col_step2 = st.columns([0.3, 0.7])
            with col_step1:
                st.write("**Step 1:** Enter the number of feeders:")
                num_feeders = st.number_input("Number of Feeders:", min_value=1, value=5, step=1, label_visibility="collapsed")
            with col_step2:
                st.write("**Step 2:** Download the CSV template, fill it with your data, and save it.")
                template_data = {header: [''] * num_feeders for header in column_headers}
                for i in range(num_feeders):
                    template_data["Feeder Name"][i] = f"Feeder {i+1}"
                template_df = pd.DataFrame(template_data)
                csv_string = template_df.to_csv(index=False, sep=';')
                st.download_button(label="Download Grid Topology CSV Template", data=csv_string, file_name="grid_topology_template.csv", mime="text/csv")

            st.write("")
            st.write("**Step 3:** Upload your filled CSV file here.")
            st.info("If you don't have data for certain fields, leave them blank in your CSV, and they will be automatically populated with default or country-specific values.")
            uploaded_file = st.file_uploader("Upload your filled CSV file", type=["csv"])

            if uploaded_file is not None:
                try:
                    df = pd.read_csv(uploaded_file, sep=';', header=0)
                    for col in column_headers:
                        if col not in df.columns:
                            df[col] = pd.NA
                    df = df[column_headers]
                    df = df.replace('', pd.NA)

                    for col, default_val in default_fill_values.items():
                        if df[col].isnull().any():
                            if pd.api.types.is_numeric_dtype(type(default_val)):
                                df[col] = pd.to_numeric(df[col], errors='coerce')
                            df[col] = df[col].fillna(default_val)
                    for col, default_val in default_fill_values.items():
                         df[col] = df[col].astype(type(default_val))
                    
                    st.session_state.electrical_system_data = df.to_dict(orient='records')
                    st.session_state.electrical_system_data_df = df.copy() 
                    st.success("CSV file uploaded, missing values filled, and parsed successfully!")
                    st.dataframe(df)
                    
                    if 'results_table_data_df' not in st.session_state or st.session_state.results_table_data_df.empty or len(st.session_state.results_table_data_df) != len(df):
                        initial_results = pd.DataFrame({
                            'SCG Installed (YES = 1 or NO = 0)': [1] * len(df),
                            'Feeder': df['Feeder Name'].tolist()
                        })
                        st.session_state.results_table_data_df = initial_results
                except Exception as e:
                    st.error(f"Error reading or processing CSV file: {e}. Please check format.")
                    st.session_state.electrical_system_data = []

            st.markdown("---")
            col_downtime_title, col_downtime_checkbox = st.columns([0.8, 0.2])
            with col_downtime_title:
                st.subheader("Cable-Related Downtime and Faults Breakdown")
            with col_downtime_checkbox:
                st.checkbox("Retrieve Data for Country", key="downtime_faults_breakdown_checkbox", on_change=lambda: st.session_state.update(retrieve_country_data_sections={'downtime_faults_breakdown': st.session_state.downtime_faults_breakdown_checkbox}))

            defaults = st.session_state.current_country_defaults
            if st.session_state.retrieve_country_data_sections.get('downtime_faults_breakdown'):
                st.session_state.weak_spots_in_cable_system = defaults["weak_spots_in_cable_system"]
                st.session_state.faults_in_cable_system = defaults["faults_in_cable_system"]
                st.session_state.intrinsic_cable_faults = defaults["intrinsic_cable_faults"]

            st.markdown("<strong>Annual Fault Frequency and Annual Weak-Spot Repairs</strong>", unsafe_allow_html=True)
            col_downtime1, col_downtime2 = st.columns(2)
            with col_downtime1:
                st.number_input("Percentage of Weak-Spot Repairs due to cable system (0-100):", min_value=0, max_value=100, key="weak_spots_in_cable_system")
            with col_downtime2:
                st.number_input("Percentage of Faults due to cable system (0-100):", min_value=0, max_value=100, key="faults_in_cable_system")
            
            st.write("")
            st.markdown("<strong>Cable-related Faults in your system by origin</strong>", unsafe_allow_html=True)
            st.number_input("Percentage of Intrinsic Cable Faults (0-100):", min_value=0, max_value=100, key="intrinsic_cable_faults")
            
        with st.expander("Restoration Costs"):
            # ... (Unchanged)
            st.write("Use this section to enter the cost of a weak-spot restoration and a fault restoration...")
            st.markdown("---")
            col_mat_title, col_mat_checkbox = st.columns([0.8, 0.2])
            with col_mat_title:
                st.subheader("Material and Excavation Costs")
            with col_mat_checkbox:
                 st.checkbox("Retrieve Data for Country", key="material_excavation_costs_checkbox", on_change=lambda: st.session_state.update(retrieve_country_data_sections={'material_excavation_costs': st.session_state.material_excavation_costs_checkbox}))
            
            if st.session_state.retrieve_country_data_sections.get('material_excavation_costs'):
                st.session_state.single_joint_cost = defaults["single_joint_cost"]
                st.session_state.mv_cable_cost = defaults["mv_cable_cost"]
                st.session_state.excavation_rate = defaults["excavation_rate"]

            col_mat1, col_mat2, col_mat3 = st.columns(3)
            with col_mat1:
                st.number_input("Enter the cost of MV cable joint (€/unit)", key="single_joint_cost", format="%.2f")
            with col_mat2:
                st.number_input("Enter the cost of MV underground cable (€/m):", key="mv_cable_cost", format="%.2f")
            with col_mat3:
                st.number_input("Enter the cost of excavation (€/hour)", key="excavation_rate", format="%.2f")

            st.markdown("---")
            col_ws_title, col_ws_checkbox = st.columns([0.8, 0.2])
            with col_ws_title:
                st.subheader("Weak Spots")
            with col_ws_checkbox:
                st.checkbox("Retrieve Data for Country", key="weak_spots_section_checkbox", on_change=lambda: st.session_state.update(retrieve_country_data_sections={'weak_spots_section': st.session_state.weak_spots_section_checkbox}))

            if st.session_state.retrieve_country_data_sections.get('weak_spots_section'):
                st.session_state.urban_weak_spot_restoration_cost = defaults["urban_weak_spot_restoration_cost"]
                st.session_state.rural_weak_spot_restoration_cost = defaults["rural_weak_spot_restoration_cost"]
                st.session_state.joints_per_weak_spot_repair = defaults["joints_per_weak_spot_repair"]
                st.session_state.weak_spot_cable_section_to_replace = defaults["weak_spot_cable_section_to_replace"]
                st.session_state.urban_weak_spot_administrative_cost = defaults["urban_weak_spot_administrative_cost"]
                st.session_state.urban_weak_spot_repair_cost = defaults["urban_weak_spot_repair_cost"]
                st.session_state.rural_weak_spot_administrative_cost = defaults["rural_weak_spot_administrative_cost"]
                st.session_state.rural_weak_spot_repair_cost = defaults["rural_weak_spot_repair_cost"]
                st.session_state.urban_weak_spot_localization_cost = defaults["urban_weak_spot_localization_cost"]
                st.session_state.rural_weak_spot_localization_cost = defaults["rural_weak_spot_localization_cost"]
                st.session_state.weak_spot_localization_time = defaults["weak_spot_localization_time"]
                st.session_state.weak_spot_administrative_time = defaults["weak_spot_administrative_time"]
                st.session_state.weak_spot_excavation_time = defaults["weak_spot_excavation_time"]
                st.session_state.weak_spot_repair_time = defaults["weak_spot_repair_time"]

            st.radio("Do you know the total costs? (yes/no):", ("yes", "no"), key="weak_spot_total_cost_known")
            if st.session_state.weak_spot_total_cost_known == 'yes':
                col_ws_total1, col_ws_total2 = st.columns(2)
                with col_ws_total1:
                    st.number_input("Enter the Weak-spot Restoration Cost in Urban Areas", key="urban_weak_spot_restoration_cost", format="%.2f")
                with col_ws_total2:
                    st.number_input("Enter the Weak-spot Restoration Cost in Rural Areas", key="rural_weak_spot_restoration_cost", format="%.2f")
            else:
                st.number_input("Enter number of joints replaced per weak-spot repair (units/repair):", key="joints_per_weak_spot_repair")
                st.number_input("Enter length of cable replaced per weak-spot repair (m/repair):", key="weak_spot_cable_section_to_replace", format="%.2f")
                st.number_input("Enter the administrative cost of a weak-spot restoration in urban areas (€/restoration):", key="urban_weak_spot_administrative_cost", format="%.2f")
                st.number_input("Enter the repair cost of a weak-spot restoration in urban areas (€/restoration):", key="urban_weak_spot_repair_cost", format="%.2f")
                st.number_input("Enter the administrative cost of a weak-spot restoration in rural areas (€/restoration):", key="rural_weak_spot_administrative_cost", format="%.2f")
                st.number_input("Enter the repair cost of a weak-spot restoration in rural areas (€/restoration):", key="rural_weak_spot_repair_cost", format="%.2f")

            st.write(""); st.write(""); st.write(""); st.write("")
            col_ws_loc1, col_ws_loc2 = st.columns(2)
            with col_ws_loc1:
                st.number_input("Enter the localization cost of a weak-spot restoration in urban areas (€/restoration):", key="urban_weak_spot_localization_cost", format="%.2f")
            with col_ws_loc2:
                st.number_input("Enter the localization cost of a weak-spot restoration in rural areas (€/restoration):", key="rural_weak_spot_localization_cost", format="%.2f")

            st.write(""); st.write(""); st.write(""); st.write("")
            st.number_input("Enter the share of the weak-spot restoration time used for localization (%):", key="weak_spot_localization_time", format="%.2f")
            st.number_input("Enter the share of the weak-spot restoration time spent in administrative matters (%):", key="weak_spot_administrative_time", format="%.2f")
            st.number_input("Enter the share of the weak-spot restoration time used for excavation (%):", key="weak_spot_excavation_time", format="%.2f")
            st.number_input("Enter the share of the weak-spot restoration time used for repair (%):", key="weak_spot_repair_time", format="%.2f")

            st.markdown("---")
            col_f_title, col_f_checkbox = st.columns([0.8, 0.2])
            with col_f_title:
                st.subheader("Faults")
            with col_f_checkbox:
                st.checkbox("Retrieve Data for Country", key="faults_section_checkbox", on_change=lambda: st.session_state.update(retrieve_country_data_sections={'faults_section': st.session_state.faults_section_checkbox}))

            if st.session_state.retrieve_country_data_sections.get('faults_section'):
                st.session_state.urban_fault_restoration_cost = defaults["urban_fault_restoration_cost"]
                st.session_state.rural_fault_restoration_cost = defaults["rural_fault_restoration_cost"]
                st.session_state.joints_per_fault_repair = defaults["joints_per_fault_repair"]
                st.session_state.fault_cable_section_to_replace = defaults["fault_cable_section_to_replace"]
                st.session_state.urban_fault_administrative_cost = defaults["urban_fault_administrative_cost"]
                st.session_state.urban_fault_repair_cost = defaults["urban_fault_repair_cost"]
                st.session_state.rural_fault_administrative_cost = defaults["rural_fault_administrative_cost"]
                st.session_state.rural_fault_repair_cost = defaults["rural_fault_repair_cost"]
                st.session_state.urban_fault_localization_cost = defaults["urban_fault_localization_cost"]
                st.session_state.rural_fault_localization_cost = defaults["rural_fault_localization_cost"]
                st.session_state.fault_localization_time = defaults["fault_localization_time"]
                st.session_state.fault_administrative_time = defaults["fault_administrative_time"]
                st.session_state.fault_excavation_time = defaults["fault_excavation_time"]
                st.session_state.fault_repair_time = defaults["fault_repair_time"]

            st.radio("Do you know the total costs? (yes/no):", ("yes", "no"), key="fault_total_cost_known")
            if st.session_state.fault_total_cost_known == 'yes':
                col_f_total1, col_f_total2 = st.columns(2)
                with col_f_total1:
                    st.number_input("Enter the Fault Restoration Cost in Urban Areas", key="urban_fault_restoration_cost", format="%.2f")
                with col_f_total2:
                    st.number_input("Enter the Fault Restoration Cost in Rural Areas", key="rural_fault_restoration_cost", format="%.2f")
            else:
                st.number_input("Enter number of joints replaced per fault repair (units/repair):", key="joints_per_fault_repair")
                st.number_input("Enter length of cable replaced per fault repair (m/repair):", key="fault_cable_section_to_replace", format="%.2f")
                st.number_input("Enter the administrative cost of a fault restoration in urban areas (€/restoration):", key="urban_fault_administrative_cost", format="%.2f")
                st.number_input("Enter the repair cost of a fault restoration in urban areas (€/restoration):", key="urban_fault_repair_cost", format="%.2f")
                st.number_input("Enter the administrative cost of a fault restoration in rural areas (€/restoration):", key="rural_fault_administrative_cost", format="%.2f")
                st.number_input("Enter the repair cost of a fault restoration in rural areas (€/restoration):", key="rural_fault_repair_cost", format="%.2f")
            
            st.write(""); st.write(""); st.write(""); st.write("")
            col_f_loc1, col_f_loc2 = st.columns(2)
            with col_f_loc1:
                st.number_input("Enter the localization cost of a fault restoration in urban areas (€/restoration):", key="urban_fault_localization_cost", format="%.2f")
            with col_f_loc2:
                st.number_input("Enter the localization cost of a fault restoration in rural areas (€/restoration):", key="rural_fault_localization_cost", format="%.2f")

            st.write(""); st.write(""); st.write(""); st.write("")
            st.number_input("Enter the share of the fault restoration time used for localization (%):", key="fault_localization_time", format="%.2f")
            st.number_input("Enter the share of the fault restoration time spent in administrative matters (%):", key="fault_administrative_time", format="%.2f")
            st.number_input("Enter the share of the fault restoration time used for excavation (%):", key="fault_excavation_time", format="%.2f")
            st.number_input("Enter the share of the fault restoration time used for repair (%):", key="fault_repair_time", format="%.2f")

        with st.expander("Financial Data"):
            # ... (Unchanged)
            col_fin_title, col_fin_checkbox = st.columns([0.8, 0.2])
            with col_fin_title:
                st.subheader("Client's Financial Parameters")
            with col_fin_checkbox:
                st.checkbox("Retrieve Data for Country", key="financial_data_section_checkbox", on_change=lambda: st.session_state.update(retrieve_country_data_sections={'financial_data_section': st.session_state.financial_data_section_checkbox}))
            
            if st.session_state.retrieve_country_data_sections.get('financial_data_section'):
                st.session_state.client_rate_of_return = defaults["client_rate_of_return"]
                st.session_state.client_ens_cost = defaults["client_ens_cost"]
            
            fin_col1, fin_col2 = st.columns(2)
            with fin_col1:
                st.number_input("Enter the client's rate of return (%):", key="client_rate_of_return", format="%.4f")
            with fin_col2:
                st.number_input("Enter the client's ENS cost (€/MWh):", key="client_ens_cost", format="%.2f")
        
        with st.expander("Regulatory Penalties"):
            # ... (Unchanged)
            col_reg_title, col_reg_checkbox = st.columns([0.8, 0.2])
            with col_reg_title:
                st.subheader("Reliability Indices and Regulatory Penalties")
            with col_reg_checkbox:
                st.checkbox("Retrieve Data for Country", key="regulatory_penalties_section_checkbox", on_change=lambda: st.session_state.update(retrieve_country_data_sections={'regulatory_penalties_section': st.session_state.regulatory_penalties_section_checkbox}))

            if st.session_state.retrieve_country_data_sections.get('regulatory_penalties_section'):
                st.session_state.saidi_reduction_compensation = defaults["saidi_reduction_compensation"]
                st.session_state.saifi_reduction_compensation = defaults["saifi_reduction_compensation"]
                st.session_state.compensation_for_residential = defaults["compensation_for_residential"]
                st.session_state.compensation_for_industrial = defaults["compensation_for_industrial"]

            reg_col1, reg_col2 = st.columns(2)
            with reg_col1:
                st.number_input("Enter the compensation for reducing SAIDI (€/minute/user)", key="saidi_reduction_compensation", format="%.4f")
            with reg_col2:
                st.number_input("Enter the compensation for reducing SAIFI (€/interruption/user)", key="saifi_reduction_compensation", format="%.4f")

            reg_col3, reg_col4 = st.columns(2)
            with reg_col3:
                st.number_input("Enter any compensation to pay to residential users due to downtime (€/min/user)", key="compensation_for_residential", format="%.13f")
            with reg_col4:
                st.number_input("Enter any compensation to pay to industrial users due to downtime (€/min/user)", key="compensation_for_industrial", format="%.13f")

        # --- Trigger calculations ---
        if st.session_state.electrical_system_data:
            perform_all_calculations()

        # --- Results Expander ---
        with st.expander("Results", expanded=True):
            if not st.session_state.results_table_data_df.empty:
                st.subheader("Client's Choice (select SCG installation below)")
                
                edited_df = st.data_editor(
                    st.session_state.results_table_data_df,
                    column_config={
                        "SCG Installed (YES = 1 or NO = 0)": st.column_config.SelectboxColumn("SCG Installed",options=[0, 1],required=True),
                        "SCG Systems Needed": st.column_config.NumberColumn("SCG Systems", format="%d"),
                        "Feeder": "Feeder Name",
                        "Downtime Reduction (min/year)": st.column_config.NumberColumn("Downtime Reduction (min/yr)", format="%.2f"),
                        "Faults Reduction (faults/year)": st.column_config.NumberColumn("Faults Reduction (faults/yr)", format="%.2f"),
                        "Undelivered Energy Reduction (MWh/year)": st.column_config.NumberColumn("ENS Reduction (MWh/yr)", format="%.2f"),
                        "Savings (€/year)": st.column_config.NumberColumn("Savings (€/yr)", format="€ %.2f"),
                        "LTE-M Coverage at Both Ends": "LTE-M Coverage",
                        "220 V Socket at Both Ends": "220V Socket",
                        "Continuity of the Earth Screen": "Earth Screen Continuity"
                    },
                    disabled=["SCG Systems Needed", "Feeder", "Downtime Reduction (min/year)", "Faults Reduction (faults/year)",
                              "Undelivered Energy Reduction (MWh/year)", "Savings (€/year)", "LTE-M Coverage at Both Ends",
                              "220 V Socket at Both Ends", "Continuity of the Earth Screen"],
                    hide_index=True, use_container_width=True, key="editable_results_table"
                )

                if not edited_df.equals(st.session_state.results_table_data_df):
                    st.session_state.results_table_data_df = edited_df
                    st.rerun()

                st.markdown("---")
                st.subheader("Financial KPIs")
                cc_df = st.session_state.client_choice_df
                fin_df = st.session_state.financial_results_df
                res_df = st.session_state.results_table_data_df

                hardware_cost = cc_df['SCG Systems Needed'].sum() * scg_unit_system_cost
                operational_cost = cc_df['SCG Systems Needed'].sum() * scg_annual_suscription
                initial_investment = hardware_cost + operational_cost
                total_savings = cc_df['New Income (€/year)'].sum()
                npv = fin_df['Discounted Cash Flow (€)'].sum()
                
                try:
                    irr = npf.irr(fin_df['Cash Flow (€)']) * 100
                    irr_text = f"{irr:.2f}%"
                except:
                    irr_text = "N/A"

                payback_period = "N/A"
                positive_cum_cf = fin_df[fin_df["Cumulative Discounted Cash Flow (€)"] >= 0]
                if not positive_cum_cf.empty:
                    first_positive_year = positive_cum_cf["Year"].iloc[0]
                    payback_period = str(first_positive_year + 1)
                
                roi = (fin_df['Cash Flow (€)'].sum() / hardware_cost * 100) if hardware_cost > 0 else "N/A"
                roi_text = f"{roi:.2f}%" if isinstance(roi, (int, float)) else roi

                kpi_col1, kpi_col2 = st.columns(2)
                with kpi_col1:
                    st.metric("Initial Investment (€)", f"{initial_investment:,.2f}")
                    st.metric("Hardware Cost (€)", f"{hardware_cost:,.2f}")
                    st.metric("Operational Cost (€/year)", f"{operational_cost:,.2f}")
                    st.metric("Total Savings (€/year)", f"{total_savings:,.2f}")
                with kpi_col2:
                    st.metric("NPV (€)", f"{npv:,.2f}")
                    st.metric("IRR (%)", irr_text)
                    st.metric("Payback Period (years)", payback_period)
                    st.metric("ROI (%)", roi_text)

                st.write("")
                st.subheader("Operational KPIs")
                saidi_reduction = cc_df['Contribution to SAIDI Reduction (min/year)'].sum()
                saifi_reduction = cc_df['Contribution to SAIFI Reduction (int/year)'].sum()
                total_downtime_reduction = res_df['Downtime Reduction (min/year)'].sum()
                total_undelivered_energy_reduction = res_df['Undelivered Energy Reduction (MWh/year)'].sum()

                op_kpi_col1, op_kpi_col2 = st.columns(2)
                with op_kpi_col1:
                    st.metric("SAIDI reduction (min/customer/year)", f"{saidi_reduction:.2f}")
                    st.metric("SAIFI reduction (faults/customer/year)", f"{saifi_reduction:.2f}")
                with op_kpi_col2:
                    st.metric("Total Downtime Reduction (min/year)", f"{total_downtime_reduction:,.2f}")
                    st.metric("Total Undelivered Energy Reduction (MWh/year)", f"{total_undelivered_energy_reduction:,.2f}")
                
                st.markdown("---")
                st.subheader("Cash Flow Analysis")
                if not st.session_state.financial_results_df.empty:
                    chart_data = st.session_state.financial_results_df.copy()
                    chart_data['Year'] = chart_data['Year'] + 1
                    melted_data = pd.melt(chart_data, id_vars=['Year'], value_vars=['Cash Flow (€)', 'Cumulative Discounted Cash Flow (€)'], var_name='Category', value_name='Value (€)')
                    chart = alt.Chart(melted_data).mark_bar().encode(
                        x=alt.X('Year:O', title='Year'),
                        y=alt.Y('Value (€):Q', title='€'),
                        color=alt.Color('Category:N', title='Metric'),
                        xOffset='Category:N'
                    ).properties(width=alt.Step(40))
                    st.altair_chart(chart, use_container_width=True)

                st.markdown("---")
                st.subheader("Optimization Solvers")
                solver_col1, solver_col2 = st.columns(2)

                with solver_col1:
                    st.write("#### Achieve a Target SAIFI Reduction")
                    max_saifi_reduction = st.session_state.lowest_cost_df['Contribution to SAIFI Reduction (int/year)'].sum()
                    st.info(f"The maximum SAIFI reduction is: {max_saifi_reduction:.2f}")
                    
                    target_saifi = st.number_input("Enter your target SAIFI reduction:", min_value=0.0, max_value=max_saifi_reduction if max_saifi_reduction > 0 else 1.0, value=max_saifi_reduction, format="%.2f")

                    if st.button("Calculate Optimal Investment for Target SAIFI Reduction"):
                        scg_install_plan = pd.Series(0, index=st.session_state.current_cost_df.index)
                        ranked_feeders = st.session_state.lowest_cost_df.sort_values(by='New Income (€/year)', ascending=False)
                        
                        total_users = st.session_state.current_cost_df['Number of Residential Users'].sum() + st.session_state.current_cost_df['Number of Industrial Users'].sum()
                        total_interruptions_base_scenario = st.session_state.current_cost_df['Total Experienced Interruptions due to Cable System Issues per Year (int-users/year)'].sum()

                        for index in ranked_feeders.index:
                            scg_install_plan.loc[index] = 1
                            
                            total_interruptions_current_scenario = (scg_install_plan * st.session_state.lowest_cost_df['Total Experienced Interruptions due to Cable System Issues per Year (int-users/year)'] + (1 - scg_install_plan) * st.session_state.current_cost_df['Total Experienced Interruptions due to Cable System Issues per Year (int-users/year)']).sum()
                            current_saifi_reduction = (total_interruptions_base_scenario - total_interruptions_current_scenario) / total_users if total_users > 0 else 0

                            if current_saifi_reduction >= target_saifi:
                                break
                        
                        st.session_state.results_table_data_df['SCG Installed (YES = 1 or NO = 0)'] = scg_install_plan.values
                        st.rerun()

                with solver_col2:
                    st.write("#### Achieve a Target SAIDI Reduction")
                    max_saidi_reduction = st.session_state.lowest_cost_df['Contribution to SAIDI Reduction (min/year)'].sum()
                    st.info(f"The maximum SAIDI reduction is: {max_saidi_reduction:.2f}")

                    target_saidi = st.number_input("Enter your target SAIDI reduction:", min_value=0.0, max_value=max_saidi_reduction if max_saidi_reduction > 0 else 1.0, value=max_saidi_reduction, format="%.2f")

                    if st.button("Calculate Optimal Investment for Target SAIDI Reduction"):
                        scg_install_plan = pd.Series(0, index=st.session_state.current_cost_df.index)
                        ranked_feeders = st.session_state.lowest_cost_df.sort_values(by='New Income (€/year)', ascending=False)

                        total_users = st.session_state.current_cost_df['Number of Residential Users'].sum() + st.session_state.current_cost_df['Number of Industrial Users'].sum()
                        total_downtime_base_scenario = st.session_state.current_cost_df['Total Experienced Downtime due to Cable System Issues per Year (min-users/year)'].sum()

                        for index in ranked_feeders.index:
                            scg_install_plan.loc[index] = 1
                            
                            total_downtime_current_scenario = (scg_install_plan * st.session_state.lowest_cost_df['Total Experienced Downtime due to Cable System Issues per Year (min-users/year)'] + (1 - scg_install_plan) * st.session_state.current_cost_df['Total Experienced Downtime due to Cable System Issues per Year (min-users/year)']).sum()
                            current_saidi_reduction = (total_downtime_base_scenario - total_downtime_current_scenario) / total_users if total_users > 0 else 0
                            
                            if current_saidi_reduction >= target_saidi:
                                break
                        
                        st.session_state.results_table_data_df['SCG Installed (YES = 1 or NO = 0)'] = scg_install_plan.values
                        st.rerun()

                st.markdown("---")
                st.subheader("Client's Choice Summary")
                st.dataframe(st.session_state.client_choice_df)
                
            else:
                st.warning("Please upload Electrical System Data to view results.")