import subprocess
import sqlite3
import json
import yaml
import argparse
import sys

def get_pod_uuids(deployment_name):
    result = subprocess.run(['kubectl', 'get', 'pods', '-l', f'app={deployment_name}', '-o', 'json'], capture_output=True, text=True)
    pods = json.loads(result.stdout)
    pod_uuids = [pod['metadata']['uid'] for pod in pods.get('items', [])]
    return pod_uuids

def insert_verifier_pod_mapping(verifier_id, pod_uuids):
    conn = sqlite3.connect('verifier_pod_mapping.db')
    cursor = conn.cursor()
    
    for pod_uuid in pod_uuids:
        cursor.execute('INSERT INTO verifier_pod_mapping (verifier_id, pod_uuid) VALUES (?, ?)', (verifier_id, pod_uuid))
    
    conn.commit()
    conn.close()

def main():

    if len(sys.argv) < 5 or sys.argv[1] != '-f' or sys.argv[3] != '-verifier':
        print("Usage: python3 kubectl_apply.py -f <deployment.yaml> -verifier <verifierID>")
        sys.exit(1)
    
    parser = argparse.ArgumentParser(description='Apply a Kubernetes deployment and store pod-verifier mapping.')
    parser.add_argument('-f', '--file', required=True, help='Path to the deployment YAML file')
    parser.add_argument('-verifier', '--verifier_id', required=True, help='Verifier ID')
    
    args = parser.parse_args()
    
    deployment_file = args.file
    verifier_id = args.verifier_id
    
    subprocess.run(['kubectl', 'apply', '-f', deployment_file], check=True)
    
    with open(deployment_file, 'r') as file:
        deployment = yaml.safe_load(file)
        deployment_name = deployment['spec']['selector']['matchLabels']['app']
    
    pod_uuids = get_pod_uuids(deployment_name)
    insert_verifier_pod_mapping(verifier_id, pod_uuids)

if __name__ == '__main__':
    main()
