clc, clear, close all

%% Configurazione del manipolatore e parametri iniziali
robot = rigidBodyTree('DataFormat', 'row');

% Definizione di un semplice manipolatore a 7 gradi di libertà
for i = 1:7
    body = rigidBody(sprintf('link%d', i));
    joint = rigidBodyJoint(sprintf('joint%d', i), 'revolute');
    setFixedTransform(joint, trvec2tform([0.1, 0, 0]));
    body.Joint = joint;
    if i == 1
        addBody(robot, body, 'base');
    else
        addBody(robot, body, sprintf('link%d', i-1));
    end
end
endEffector = sprintf('link7');

% Traiettoria da A a B
A = [0, 0, 0];
B = [1, 1, 1];
v_max = 0.5; % Velocità massima
a_max = 0.2; % Accelerazione massima

% Calcolo dei parametri del profilo trapezoidale
d = norm(B - A);
t_acc = v_max / a_max;
if d <= v_max^2 / a_max
    t_tot = 2 * sqrt(d / a_max);
else
    t_tot = t_acc + d / v_max;
end

t = linspace(0, t_tot, 100); % Tempo discreto

% Calcolo della traiettoria
pos = zeros(length(t), 3);
vel = zeros(length(t), 3);
acc = zeros(length(t), 3);
for i = 1:3
    [pos(:,i), vel(:,i), acc(:,i)] = trapezoidal_trajectory(A(i), B(i), v_max, a_max, t);
end

% Inizializzazione per cinematica e dinamica
ik = inverseKinematics('RigidBodyTree', robot);
weights = [1, 1, 1, 1, 1, 1];
initial_guess = zeros(1, 7);

q_traj = zeros(length(t), 7);
q_dot_traj = zeros(length(t), 7);
tau_traj = zeros(length(t), 7);

for i = 1:length(t)
    % Cinematica inversa
    q_sol = ik(endEffector, trvec2tform(pos(i,:)), weights, initial_guess);
    q_traj(i, :) = q_sol;

    % Cinematica differenziale
    J = geometricJacobian(robot, q_sol, endEffector);
    q_dot = pinv(J) * [vel(i,:), 0, 0, 0]'; % Adatta il vettore velocità alle dimensioni del Jacobiano
    q_dot_traj(i, :) = q_dot';

    % Dinamica inversa
    tau = inverseDynamics(robot, q_sol, q_dot, acc(i,:));

    % Minimizzazione dei momenti alla base
    N = null(J);
    tau_base = tau(1:3); % Consideriamo i momenti alla base
    null_torque = N * (-N' * tau_base); % Proiezione nel null-space
    tau_opt = tau + null_torque;

    tau_traj(i, :) = tau_opt;
end

% Grafici
figure;
subplot(3,1,1); plot(t, pos); title('Traiettoria - Posizione'); xlabel('Tempo [s]'); ylabel('Posizione [m]');
subplot(3,1,2); plot(t, vel); title('Traiettoria - Velocità'); xlabel('Tempo [s]'); ylabel('Velocità [m/s]');
subplot(3,1,3); plot(t, acc); title('Traiettoria - Accelerazione'); xlabel('Tempo [s]'); ylabel('Accelerazione [m/s^2]');

figure;
plot(t, tau_traj);
title('Coppie ottimizzate');
xlabel('Tempo [s]'); ylabel('Coppie [Nm]');
legend(arrayfun(@(x) sprintf('Joint %d', x), 1:7, 'UniformOutput', false));

%% Funzione per generare il profilo trapezoidale
function [pos, vel, acc] = trapezoidal_trajectory(p0, pf, v_max, a_max, t)
    d = pf - p0;
    t_acc = v_max / a_max;
    if abs(d) <= v_max^2 / a_max
        t_tot = 2 * sqrt(abs(d) / a_max);
        t_acc = t_tot / 2;
    else
        t_tot = t_acc + abs(d) / v_max;
    end

    pos = zeros(size(t));
    vel = zeros(size(t));
    acc = zeros(size(t));

    for i = 1:length(t)
        if t(i) <= t_acc
            acc(i) = sign(d) * a_max;
            vel(i) = acc(i) * t(i);
            pos(i) = p0 + 0.5 * acc(i) * t(i)^2;
        elseif t(i) <= t_tot - t_acc
            acc(i) = 0;
            vel(i) = sign(d) * v_max;
            pos(i) = p0 + 0.5 * sign(d) * a_max * t_acc^2 + vel(i) * (t(i) - t_acc);
        else
            acc(i) = -sign(d) * a_max;
            vel(i) = sign(d) * v_max + acc(i) * (t(i) - (t_tot - t_acc));
            pos(i) = pf - 0.5 * acc(i) * (t_tot - t(i))^2;
        end
    end
end
