#!/bin/bash

source logging.sh
source variables.sh

# DESCRIPTION #
log $DEBUG_LEVEL $VERBOSE_INFO "Finishing the setup of the host ...."

# Get the length of NODES
node_count=${#NODES[@]}
provisioned_count=$(grep -oP 'MACHINE PROVISIONNED : \K\d+' "$PATH_TO_BIN" | tail -n 1)

# If provisioned_count is not set or is less than node_count, skip final setup
if [[ -z "$provisioned_count" || "$provisioned_count" -lt "$node_count" ]]; then
  log $DEBUG_LEVEL $VERBOSE_DEBUG "Provisioning incomplete - $provisioned_count out of $node_count machines provisioned."
  log $DEBUG_LEVEL $VERBOSE_INFO "Skipping final setup as not all machines are provisioned."

# If provisioned_count equals node_count, proceed with final setup
elif [[ "$provisioned_count" -eq "$node_count" ]]; then
  log $DEBUG_LEVEL $VERBOSE_INFO "All machines have been successfully provisioned. Running final setup..."

  # HERE Final setup #
  # Check if the keys folder exists
  if [ -d "$PATH_TO_SSH_KEYS_FOLDER" ]; then
    log $DEBUG_LEVEL $VERBOSE_INFO "$PATH_TO_SSH_KEYS_FOLDER folder exists"

    # Ensure the SSH_ROOT_FOLDER directory exists
    if [ ! -d "$SSH_ROOT_FOLDER" ]; then
      log $DEBUG_LEVEL $VERBOSE_DEBUG "SSH_ROOT_FOLDER directory does not exist. Creating it."
      sudo mkdir -p "$SSH_ROOT_FOLDER"
      sudo chmod 700 "$SSH_ROOT_FOLDER"
    fi

    # Ensure the authorized_keys file exists
    if [ ! -f "$SSH_ROOT_FOLDER/authorized_keys" ]; then
      log $DEBUG_LEVEL $VERBOSE_DEBUG "SSH_ROOT_FOLDER/authorized_keys file does not exist. Creating it."
      sudo touch "$SSH_ROOT_FOLDER/authorized_keys"
      sudo chmod 600 "$SSH_ROOT_FOLDER/authorized_keys"
    fi

    if [ "$DEBUG_LEVEL" == "$VERBOSE_DEBUG" ]; then
      # Print the keys folder contents
      log $DEBUG_LEVEL $VERBOSE_DEBUG "Public keys in $PATH_TO_SSH_KEYS_FOLDER :"
      ls "$PATH_TO_SSH_KEYS_FOLDER"/*.pub

      # Print the authorized_keys file before adding keys
      log $DEBUG_LEVEL $VERBOSE_DEBUG "Authorized keys before adding new keys:"
      sudo cat "$SSH_ROOT_FOLDER/authorized_keys"
    fi

    # Append all keys from PATH_TO_SSH_KEYS_FOLDER to authorized_keys
    cat "$PATH_TO_SSH_KEYS_FOLDER"/*.pub | sudo tee -a "$SSH_ROOT_FOLDER/authorized_keys" > /dev/null
    log $DEBUG_LEVEL $VERBOSE_INFO "All public keys from $PATH_TO_SSH_KEYS_FOLDER have been appended to authorized_keys"

    # Log the action
    log $DEBUG_LEVEL $VERBOSE_INFO "Added all keys from $PATH_TO_SSH_KEYS_FOLDER to $SSH_ROOT_FOLDER/authorized_keys."

    # Print the authorized_keys file after adding keys
    log $DEBUG_LEVEL $VERBOSE_DEBUG "Authorized keys after adding new keys:"
    if [ "$DEBUG_LEVEL" == "$VERBOSE_DEBUG" ]; then
      sudo cat "$SSH_ROOT_FOLDER/authorized_keys"
    fi

    # Generate the ssh config file
    ./host/configs/generate_ssh_config.sh

    # HERE Running the final tests & scenarios for first configurations & ensuring everything work fine #
    sudo -su $(hostname) $PATH_TO_TESTS_FILE
    sudo -su $(hostname) $PATH_TO_SCENARIOS_FILE

  else
    log $DEBUG_LEVEL $VERBOSE_ERROR "The $PATH_TO_SSH_KEYS_FOLDER directory does not exist. You are destroying machines."
  fi

# If provisioned_count is greater than node_count, log that setup is complete and exit
else
  log $DEBUG_LEVEL $VERBOSE_INFO "Machines already set up: skipping the final setup process..."
fi
