#!/bin/bash
# Typical Bash Script running in provisioning machine

# Get the script filename (without the path) for dynamic naming
if [[ -z "$1" ]]; then
  # If no argument is provided, use $0 to get the script name
  SCRIPT_NAME="${0##*/}"
else
  # If an argument is provided, use it as the script name
  SCRIPT_NAME="$1"
fi

BASENAME="${SCRIPT_NAME%.*}"  # Remove the file extension

# Go to root
cd /

# -------------------------------------------------
# HERE: Source necessary variables
# -------------------------------------------------

# This loads the DEBUG_LEVEL and other variables from sourcer.sh and variables.sh
source /vagrant/sourcer.sh variables.sh utils.sh

# -------------------------------------------------
# HERE: Description
# -------------------------------------------------

log $DEBUG_LEVEL $VERBOSE_INFO "Starting script execution: $SCRIPT_NAME"

# HERE: Define paths for logs and backups
LOG_FILE="$PATH_TO_LOG_DIR/${BASENAME}.log"
initialize_log_file "$LOG_FILE"
TIMESTAMP=$(date +%Y%m%d_%H%M%S)  # Timestamp for uniqueness

# Initialize error count
ERROR_COUNT=0

# -------------------------------------------------
# HERE: Requirements & existence
# -------------------------------------------------
# Create logs directory if not already exists
log $DEBUG_LEVEL $VERBOSE_DEBUG "Creating logs directory..."
mkdir -p $PATH_TO_TELER_DIR
mkdir -p /tmp
check_file_existence $PATH_TO_TELER_LOGS $PATH_TO_APACHE_LOGS

log $DEBUG_LEVEL $VERBOSE_DEBUG "Cleaning Teler logs..."
# Clean the logs
> $PATH_TO_TELER_LOGS
> $PATH_TO_APACHE_LOGS

log $DEBUG_LEVEL $VERBOSE_DEBUG "Killing previous Teler processes..."
if pgrep -f teler ; then
  pgrep -f teler | xargs kill -9
  log $DEBUG_LEVEL $VERBOSE_DEBUG "Teler process was running : killed."

else
    log $DEBUG_LEVEL $VERBOSE_DEBUG "Teler process wasn't running."
fi


# -------------------------------------------------
# HERE: Beginning of script
# -------------------------------------------------

# Call the functions
install_docker
run_dvwa_container
install_teler
configure_teler

# Configuring ossec conf for log parsing.
add_local_file "syslog" $PATH_TO_TELER_LOGS

log $DEBUG_LEVEL $VERBOSE_DEBUG "Restarting victim agent..."
sudo systemctl restart wazuh-agent

# Avoid broken pipe by allocating memory
systemctl stop suricata

# Rotate the log file
logrotate -f $PATH_TO_APACHE_LOGS

log $DEBUG_LEVEL $VERBOSE_DEBUG "Activating teler to read apache logs..."

# ensure to be in the teler folder
cd /teler
stdbuf -oL tail -n 0 -F $PATH_TO_APACHE_LOGS | ./teler -c teler.yml > /dev/null 2>&1 &

# Capture the PIDs of the teler processes
TELER_PIDS=$(pgrep -f teler)

# Verify if the teler process is running
if pgrep -f teler > /dev/null; then
    log $DEBUG_LEVEL $VERBOSE_DEBUG "Teler process is running with PID $TELER_PIDS."
else
    log $DEBUG_LEVEL $VERBOSE_ERROR "Teler process is not running."
    ERROR_COUNT=$((ERROR_COUNT + 1))
fi


# -------------------------------------------------
# End of Script
# -------------------------------------------------

if [ $ERROR_COUNT -eq 0 ]; then
    log $DEBUG_LEVEL $VERBOSE_INFO "$SCRIPT_NAME successfully realized."
else
    log $DEBUG_LEVEL $VERBOSE_INFO "Errors encountered during script execution: $SCRIPT_NAME. Total errors: $ERROR_COUNT"
fi
