#!/bin/bash
# Typical Bash Script running in provisioning machine

# Get the script filename (without the path) for dynamic naming
if [[ -z "$1" ]]; then
  # If no argument is provided, use $0 to get the script name
  SCRIPT_NAME="${0##*/}"
else
  # If an argument is provided, use it as the script name
  SCRIPT_NAME="$1"
fi

BASENAME="${SCRIPT_NAME%.*}"  # Remove the file extension

# Go to root
cd /

# -------------------------------------------------
# HERE: Source necessary variables
# -------------------------------------------------

# This loads the DEBUG_LEVEL and other variables from sourcer.sh and variables.sh
source /vagrant/sourcer.sh variables.sh utils.sh

# -------------------------------------------------
# HERE: Description
# -------------------------------------------------

log $DEBUG_LEVEL $VERBOSE_INFO "Starting script execution: $SCRIPT_NAME"

# HERE: Define paths for logs and backups
LOG_FILE="$PATH_TO_LOG_DIR/${BASENAME}.log"
initialize_log_file "$LOG_FILE"
TIMESTAMP=$(date +%Y%m%d_%H%M%S)  # Timestamp for uniqueness
check_folder_existence "/tmp" $PATH_TO_OSSEC_LISTS_DIR
# Initialize error count
ERROR_COUNT=0

# -------------------------------------------------
# HERE: Beginning of script
# -------------------------------------------------


log $DEBUG_LEVEL $VERBOSE_DEBUG "Get AlienVault IP list..."
sudo wget https://raw.githubusercontent.com/firehol/blocklist-ipsets/master/alienvault_reputation.ipset -O $PATH_TO_ALIENVAULT_IP_LIST


# Add also the hacker IP to the IP list
log $DEBUG_LEVEL $VERBOSE_DEBUG "Adding Hacker IP to blacklist..."
sudo echo "$ATTACKER_IP" >> $PATH_TO_ALIENVAULT_IP_LIST

# Check for existence
log $DEBUG_LEVEL $VERBOSE_DEBUG "Converting IP list to CDB format..."
if ! [ -f "$PATH_TO_IPLIST_TO_CBDLIST_PYTHON" ]; then
  sudo wget https://wazuh.com/resources/iplist-to-cdblist.py -O $PATH_TO_IPLIST_TO_CBDLIST_PYTHON
fi

sudo python3 $PATH_TO_IPLIST_TO_CBDLIST_PYTHON $PATH_TO_ALIENVAULT_IP_LIST $PATH_TO_ALIENVAULT_BLACKLIST
sudo chown wazuh:wazuh $PATH_TO_ALIENVAULT_BLACKLIST
sudo rm -rf $PATH_TO_ALIENVAULT_IP_LIST

# Test logs parsing and rules
log $DEBUG_LEVEL $VERBOSE_DEBUG "Testing logs parsing and rules..."
echo '192.168.12.182 - - [08/Feb/2024:20:11:22 +0000] "GET / HTTP/1.1" 200 10926 "-" "curl/7.81.0"' | sudo /var/ossec/bin/wazuh-logtest

# Change level rules for web_rules
log $DEBUG_LEVEL $VERBOSE_DEBUG "Changing level rules for web_rules..."
change_level_rules "31108" "3" "$PATH_TO_WEB_RULES_XML"

# Change to store logs temporarily in archive with /var/ossec/etc/ossec.conf
log $DEBUG_LEVEL $VERBOSE_DEBUG "Changing to store logs temporarily in archive..."
enable_log_archive

# Add local rules for AlienVault IP list
log $DEBUG_LEVEL $VERBOSE_DEBUG "Adding local rules for AlienVault IP list..."
add_local_rules "$PATH_TO_LOCAL_RULES_ALIENVAULT"

# Add list to ossec.conf
log $DEBUG_LEVEL $VERBOSE_DEBUG "Adding list to ossec.conf..."
add_list "$PATH_TO_RELATIVE_ALIENVAULT_BLACKLIST"

# Add active response for AlienVault IP list
log $DEBUG_LEVEL $VERBOSE_DEBUG "Adding active response for AlienVault IP list..."
add_active_response "firewall-drop" "local" "100100" "60"

# Restart Wazuh manager
log $DEBUG_LEVEL $VERBOSE_DEBUG "Restarting Wazuh manager..."
sudo systemctl restart wazuh-manager

# -------------------------------------------------
# End of Script
# -------------------------------------------------

if [ $ERROR_COUNT -eq 0 ]; then
    log $DEBUG_LEVEL $VERBOSE_INFO "$SCRIPT_NAME successfully realized."
else
    log $DEBUG_LEVEL $VERBOSE_INFO "Errors encountered during script execution: $SCRIPT_NAME. Total errors: $ERROR_COUNT"
fi
