#!/bin/bash

# -------------------------------------------------
# NOTE: Typical Bash Script running on host machine
# -------------------------------------------------

# -------------------------------------------------
# HERE: References Section
# -------------------------------------------------
# Add your references below:
# 1. [Scenario tutorial](https://example.com/scenario-tutorial)

# HACK For debug
cd /home/koraty/Documenti/master_thesis/vagrant/

# HERE Sourcing
source logging.sh
source variables.sh
source utils.sh

# Get the script filename (without the path) for dynamic naming
if [[ -z "$1" ]]; then
  # If no argument is provided, use $0 to get the script name
  SCRIPT_NAME="${0##*/}"
else
  # If an argument is provided, use it as the script name
  SCRIPT_NAME="$1"
fi

BASENAME="${SCRIPT_NAME%.*}"  # Remove the file extension

# HERE: Description
log $DEBUG_LEVEL $VERBOSE_INFO "Starting script execution: $SCRIPT_NAME"

# HERE: Define paths for logs and backups
LOG_FILE="$PATH_TO_LOG_DIR/${BASENAME}.log"
initialize_log_file "$LOG_FILE"

# -------------------------------------------------
# -------------------------------------------------
# HERE: Beginning of script
# -------------------------------------------------
# -------------------------------------------------

# Define the path to the CSV file and script
CSV_FILE=$PATH_TO_CSV_NODES_INFO

# Defining scripts for each VM for the scenario
ATTACKER_SCRIPT_PATH=$PATH_TO_SCENARIO_ATTACKER
DEFENDER_SCRIPT_PATH=$PATH_TO_SCENARIO_DEFENDER
SERVER_SCRIPT_PATH=$PATH_TO_SCENARIO_MANAGER

# Initialize error count
ERROR_COUNT=0

# -------------------------------------------------
# HERE: Checking for existence
# -------------------------------------------------

# Check if the CSV file exists
if [ ! -f "$CSV_FILE" ]; then
    log $DEBUG_LEVEL $VERBOSE_ERROR "CSV file not found: $CSV_FILE"
    ERROR_COUNT=$((ERROR_COUNT + 1))
else
    log $DEBUG_LEVEL $VERBOSE_DEBUG "CSV file found: $CSV_FILE"
fi

# Check if the script paths exist
for SCRIPT_PATH in "$ATTACKER_SCRIPT_PATH" "$DEFENDER_SCRIPT_PATH" "$SERVER_SCRIPT_PATH"; do
    if [ ! -f "$SCRIPT_PATH" ]; then
        log $DEBUG_LEVEL $VERBOSE_ERROR "Script not found: $SCRIPT_PATH"
        ERROR_COUNT=$((ERROR_COUNT + 1))
    else
        log $DEBUG_LEVEL $VERBOSE_DEBUG "Script found: $SCRIPT_PATH"
    fi
done

# Print the content of the CSV file for debugging
log $DEBUG_LEVEL $VERBOSE_DEBUG "CSV file content:"
cat "$CSV_FILE"

# Loop through each node in $NODES and check its type from the CSV file
while IFS=',' read -r node type ip; do
    # Skip the header row
    if [ "$node" == "Node" ]; then
        continue
    fi

    # Trim whitespace, single quotes, and carriage returns from the type variable
    type=$(echo "$type" | sed "s/^[[:space:]]*['\"]*\([^'\"]*\)['\"]*[[:space:]]*$/\1/" | tr -d '\r')

    # Log the trimmed type
    log $DEBUG_LEVEL $VERBOSE_DEBUG "Trimmed type: $type"

    # Log which node is being processed
    log $DEBUG_LEVEL $VERBOSE_DEBUG "Processing node $node with type $type..."

    # Log the comparison being made
    log $DEBUG_LEVEL $VERBOSE_DEBUG "Comparing type: $type"

    # Use case for different node types
    case "$type" in
        "server")
            log $DEBUG_LEVEL $VERBOSE_INFO "Node $node is a server. Running server-specific script."
            # Run server-specific tasks here
            if ! ssh root@"$node" 'bash -s' < $SERVER_SCRIPT_PATH "${SERVER_SCRIPT_PATH##*/}"; then
                log $DEBUG_LEVEL $VERBOSE_ERROR "SSH connection failed for server node $node. Continuing to next node."
                ERROR_COUNT=$((ERROR_COUNT + 1))
            else
                log $DEBUG_LEVEL $VERBOSE_DEBUG "Server script executed successfully on node $node."
            fi
            ;;
        "defender")
            log $DEBUG_LEVEL $VERBOSE_INFO "Node $node is a defender. Running defender-specific script."

            # Run defender-specific tasks here
            # DRAFT : If need a timeout (having backend running script) :                         
            # timeout 15 ssh root@"$node" "bash -s -- '$DEFENDER_SCRIPT_NAME'" < "$DEFENDER_SCRIPT_PATH"
)
            if ! ssh root@"$node" 'bash -s' < $DEFENDER_SCRIPT_PATH "${DEFENDER_SCRIPT_PATH##*/}"; then
                log $DEBUG_LEVEL $VERBOSE_ERROR "SSH connection failed for defender node $node. Continuing to next node."
                ERROR_COUNT=$((ERROR_COUNT + 1))
            else
                log $DEBUG_LEVEL $VERBOSE_DEBUG "Defender script executed successfully on node $node."
            fi
            ;;
        "attacker")
            log $DEBUG_LEVEL $VERBOSE_INFO "Node $node is an attacker. Running attacker-specific script."
            # Run attacker-specific tasks here
            if ! ssh root@"$node" 'bash -s' < $ATTACKER_SCRIPT_PATH "${ATTACKER_SCRIPT_PATH##*/}"; then
                log $DEBUG_LEVEL $VERBOSE_ERROR "SSH connection failed for attacker node $node. Continuing to next node."
                ERROR_COUNT=$((ERROR_COUNT + 1))
            else
                log $DEBUG_LEVEL $VERBOSE_DEBUG "Attacker script executed successfully on node $node."
            fi
            ;;
        *)
            log $DEBUG_LEVEL $VERBOSE_WARNING "Unknown node type for $node. Skipping node."
            ;;
    esac
done < "$CSV_FILE"

# -------------------------------------------------
# HERE: Checking for logs
# -------------------------------------------------

rule_id="YOUR_RULE_ID"
log $DEBUG_LEVEL $VERBOSE_INFO "Retrieving manager logs: $PATH_TO_ALERT_LOGS and Rule ID: $rule_id"

# Call the parse_rule_id function and increment ERROR_COUNT if it returns 1
# Run the parse_rule_id function on the remote wazidx1 agent via SSH
ssh root@$MANAGER_NAME "bash -s" << 'EOF'
cd /
source /vagrant/sourcer.sh utils.sh variables.sh
parse_rule_id "$PATH_TO_ALERT_LOGS" "$rule_id"
if [ $? -ne 0 ]; then
  ERROR_COUNT=$((ERROR_COUNT + 1))
fi
EOF

# End of Script
if [ $ERROR_COUNT -eq 0 ]; then
    log $DEBUG_LEVEL $VERBOSE_INFO "Script executed successfully: $SCRIPT_NAME"
else
    log $DEBUG_LEVEL $VERBOSE_INFO "Errors encountered during script execution: $SCRIPT_NAME. Total errors: $ERROR_COUNT"
fi
