use crate::structures::algebraics::FiniteField;

use std::fmt::Debug;

/// Inversion table on the Finite Field F_3329
/// INV_3329[i] = i^-1 mod 3329
const INV_3329: [usize; 3329] = [
    0, 1, 1665, 1110, 2497, 666, 555, 2378, 2913, 370, 333, 908, 1942, 3073, 1189, 222, 3121, 1175,
    185, 2453, 1831, 3012, 454, 579, 971, 799, 3201, 1233, 2259, 574, 111, 1933, 3225, 2522, 2252,
    2473, 1757, 90, 2891, 2134, 2580, 406, 3170, 2400, 227, 74, 1954, 425, 2150, 2242, 2064, 2611,
    3265, 1005, 2281, 2179, 2794, 3037, 287, 1862, 1720, 2947, 2631, 1004, 3277, 2612, 2925, 795,
    1126, 193, 2901, 422, 2543, 3101, 45, 1376, 3110, 2032, 1067, 1812, 1290, 411, 203, 1845, 3249,
    235, 1200, 1301, 1778, 1571, 37, 439, 977, 1754, 1877, 2488, 1075, 858, 1121, 3060, 1032, 2505,
    2970, 2521, 3297, 1934, 2167, 3049, 2805, 733, 2754, 30, 1397, 1856, 3183, 2779, 1808, 1821,
    931, 1119, 860, 2201, 3138, 1245, 2980, 2823, 502, 1232, 3303, 800, 2970, 2897, 2295, 826,
    2062, 2244, 563, 243, 1761, 479, 3115, 2361, 211, 582, 2936, 2778, 3215, 1857, 1687, 2480,
    1520, 948, 1555, 1980, 1016, 2384, 2198, 2184, 906, 335, 645, 2401, 1870, 2798, 1766, 1836,
    2585, 1515, 2041, 2285, 1782, 2122, 600, 1347, 2315, 2492, 889, 2840, 2450, 1711, 1683, 2391,
    1884, 2092, 2153, 18, 877, 1620, 2603, 2554, 1244, 3207, 2202, 69, 2093, 3090, 2225, 1166,
    3194, 2990, 516, 3177, 2917, 82, 3149, 747, 2925, 1174, 3313, 223, 967, 142, 2748, 2360, 3189,
    480, 3067, 2654, 2447, 3253, 1377, 2139, 15, 209, 2363, 2678, 928, 44, 3256, 2544, 3054, 2619,
    904, 2186, 2679, 85, 2130, 1436, 2224, 3134, 430, 1036, 2765, 137, 1569, 1780, 2287, 1213,
    1490, 615, 3076, 1916, 251, 3079, 616, 1188, 3316, 1943, 400, 2628, 653, 2653, 3113, 2977,
    2812, 201, 413, 2743, 1031, 3230, 1122, 2248, 1946, 1256, 2613, 3099, 2545, 1995, 1904, 2804,
    3222, 2168, 763, 2882, 1770, 1939, 291, 58, 1468, 2419, 3053, 286, 3272, 2795, 2593, 1704,
    2508, 1020, 1240, 1581, 2840, 1294, 2970, 835, 2442, 1921, 3278, 2011, 508, 1950, 1192, 2840,
    2763, 1723, 1092, 2864, 451, 3308, 1832, 1868, 1987, 2958, 2864, 2690, 935, 1854, 1399, 1354,
    883, 1963, 915, 2102, 2955, 10, 2422, 159, 2685, 731, 2807, 2838, 891, 781, 2725, 1747, 300,
    2036, 2338, 307, 3238, 3205, 1246, 607, 2109, 2650, 3083, 1416, 1225, 373, 2520, 3227, 2506,
    1706, 2860, 2953, 2606, 1286, 1046, 1152, 2741, 415, 9, 2997, 2103, 357, 810, 3229, 2966, 2861,
    1277, 1827, 622, 2630, 3268, 1721, 1101, 1738, 1699, 2486, 1879, 2071, 3209, 2873, 2777, 3185,
    583, 1694, 765, 805, 1079, 3049, 258, 2366, 3045, 1173, 3123, 1580, 41, 2864, 2407, 2336, 2454,
    2577, 3127, 1930, 3083, 369, 3321, 2379, 1776, 1303, 2148, 427, 67, 787, 1374, 47, 2426, 421,
    3259, 194, 240, 2294, 3198, 1307, 1639, 2590, 3304, 2133, 3291, 91, 2353, 2283, 2734, 511,
    1672, 980, 1769, 3046, 2846, 519, 3003, 2458, 464, 316, 22, 2751, 3292, 845, 1271, 660, 1527,
    2513, 2558, 870, 448, 3014, 3173, 1276, 2952, 2967, 1707, 728, 1065, 2034, 2798, 2495, 1112,
    2331, 3231, 139, 215, 263, 518, 2881, 1798, 3000, 1733, 2372, 3281, 2735, 890, 2990, 2808,
    3240, 2271, 612, 745, 1487, 1972, 1542, 1538, 505, 2622, 2098, 1790, 2165, 3204, 2981, 308,
    3044, 594, 651, 3322, 1817, 2595, 1170, 200, 3065, 2146, 449, 2823, 1169, 3095, 732, 3221,
    3050, 1905, 897, 1406, 2341, 1747, 3166, 1871, 2592, 2619, 3273, 2180, 1680, 3070, 1717, 561,
    2246, 1124, 797, 973, 1311, 1460, 779, 2971, 1807, 3214, 3184, 2937, 1210, 3285, 6, 952, 1793,
    1402, 1465, 1611, 540, 1084, 136, 3087, 1037, 1441, 1961, 885, 3074, 2634, 1819, 1810, 1069,
    29, 3219, 734, 2875, 3290, 23, 2359, 3117, 143, 394, 2468, 1030, 3062, 414, 2961, 1153, 852,
    3024, 1254, 1948, 2173, 2887, 620, 1829, 2455, 1206, 3083, 1983, 3143, 1198, 237, 1106, 2082,
    351, 2885, 1137, 2625, 1057, 2159, 2585, 1411, 249, 254, 2142, 975, 441, 596, 1501, 2044, 700,
    3046, 2562, 2526, 1184, 2210, 2551, 2471, 1741, 226, 1157, 2538, 1578, 916, 1965, 934, 3006,
    2658, 1745, 2727, 730, 2993, 160, 2488, 602, 2132, 2893, 918, 3104, 2364, 260, 2341, 1911,
    2106, 2194, 3060, 2056, 453, 1803, 2714, 713, 1478, 831, 5, 2775, 1211, 2289, 1744, 2689, 3007,
    2127, 3278, 3112, 3068, 2318, 1419, 2976, 2110, 2234, 3303, 2671, 2195, 2046, 2538, 2512, 150,
    3214, 1018, 2510, 1169, 2815, 1818, 3175, 3075, 1003, 3267, 2115, 623, 1116, 1968, 2720, 2719,
    1968, 2989, 2973, 710, 1186, 2363, 2975, 2277, 663, 1851, 2280, 1254, 3264, 3278, 2065, 1253,
    2738, 853, 1455, 1430, 2553, 3141, 3077, 471, 2265, 3139, 2001, 523, 109, 576, 1703, 3035,
    2796, 1872, 2733, 1669, 1514, 3163, 1837, 2716, 496, 1843, 205, 405, 3289, 2343, 2570, 1483,
    84, 3303, 2395, 2303, 1482, 2578, 2344, 311, 1383, 1315, 2007, 2466, 2892, 2524, 2704, 2215,
    1000, 847, 2867, 2514, 2993, 1243, 3140, 2604, 1431, 2700, 3043, 3269, 341, 2893, 1994, 3053,
    3100, 2214, 423, 1956, 1269, 847, 2643, 2047, 1251, 1858, 67, 2204, 2207, 2357, 25, 129, 2024,
    2004, 2703, 2979, 1437, 3083, 3296, 3226, 890, 1622, 2389, 1685, 1859, 1120, 2557, 2868, 3192,
    1997, 2638, 2683, 3033, 2537, 2969, 3228, 2697, 2629, 2932, 1958, 1024, 2473, 665, 3325, 1111,
    2854, 303, 888, 3154, 2316, 1603, 2738, 3234, 2710, 2942, 1700, 2121, 2058, 790, 2351, 3180,
    1688, 2175, 1836, 722, 1875, 1756, 3294, 2253, 95, 2209, 120, 1129, 2811, 1651, 1599, 2433,
    3150, 1916, 3160, 770, 463, 2878, 1340, 1205, 3251, 1830, 3310, 186, 1710, 3151, 2424, 1540,
    2801, 1992, 1367, 568, 1917, 3025, 836, 176, 2154, 340, 2549, 548, 1523, 1008, 1423, 527, 1924,
    985, 2334, 2409, 1229, 1541, 232, 1144, 1614, 2995, 11, 3052, 3040, 1469, 2061, 773, 2087,
    1749, 2300, 1365, 1161, 2892, 3251, 2335, 2921, 1825, 2943, 2001, 435, 2314, 226, 3286, 1507,
    118, 2211, 1363, 2302, 1156, 3244, 1883, 3148, 1684, 2518, 2039, 364, 2044, 2197, 3174, 1017,
    2640, 3063, 1775, 2912, 3322, 556, 1537, 2830, 1973, 2448, 2842, 1734, 1937, 1772, 1351, 1796,
    2928, 1923, 2677, 3105, 210, 1731, 3116, 251, 24, 2531, 544, 1186, 618, 2889, 92, 743, 1656,
    445, 1561, 1494, 1463, 1404, 899, 3284, 1764, 2800, 3071, 306, 2983, 2037, 2920, 1991, 2429,
    986, 1566, 1186, 1113, 3258, 2461, 1917, 2361, 478, 53, 1049, 1805, 1724, 1907, 1915, 2463,
    2434, 2491, 3155, 1348, 2641, 2610, 690, 820, 297, 2090, 2718, 1370, 3325, 1481, 2573, 2188,
    1318, 1692, 585, 268, 100, 825, 2363, 2898, 241, 565, 1889, 1743, 2660, 1212, 3083, 1781, 3160,
    2042, 3278, 3010, 3275, 1006, 3189, 662, 2617, 1312, 2113, 1360, 3199, 2275, 3251, 1993, 2547,
    342, 1583, 642, 2600, 472, 1296, 78, 1934, 573, 3301, 1234, 1090, 1725, 840, 96, 3304, 3295,
    3147, 398, 1945, 3058, 2787, 2788, 562, 3194, 2479, 3280, 2151, 2094, 2318, 1605, 1978, 466,
    3299, 2648, 2111, 1314, 2568, 312, 3265, 384, 2423, 1165, 3133, 3091, 2269, 1248, 2994, 1663,
    3, 833, 476, 2663, 2561, 2705, 2365, 1362, 3229, 119, 2470, 98, 270, 1082, 2206, 2533, 68,
    3137, 3208, 861, 2183, 3173, 2385, 2045, 2645, 2672, 2107, 609, 705, 2005, 1317, 3134, 2574,
    3096, 905, 3172, 2199, 862, 1679, 2793, 3274, 2282, 367, 589, 2581, 1689, 3309, 1465, 1676,
    1732, 2844, 3048, 3223, 1935, 1736, 2767, 197, 1800, 818, 692, 515, 3130, 2360, 403, 207, 17,
    3134, 2925, 2241, 3281, 426, 2909, 1304, 2674, 627, 2784, 974, 2712, 255, 14, 3108, 2210, 310,
    2568, 2579, 3290, 2892, 2681, 603, 3093, 918, 3277, 2656, 3008, 2820, 873, 1431, 658, 1783,
    2055, 2217, 668, 1041, 247, 1840, 1359, 2275, 2143, 2233, 2649, 2977, 608, 2193, 2673, 3264,
    356, 2957, 2998, 919, 2560, 1789, 3242, 127, 25, 2735, 2240, 2152, 1064, 1885, 2308, 298, 1746,
    2415, 774, 1854, 2825, 1177, 2723, 3187, 1999, 2945, 2041, 1263, 720, 592, 1382, 2769, 2376,
    2940, 1880, 2380, 66, 2535, 1252, 2609, 3279, 3075, 3195, 3323, 3036, 789, 402, 458, 2670,
    2646, 2235, 467, 378, 1503, 3212, 1329, 2914, 2537, 2644, 3028, 3218, 1197, 2284, 3161, 886,
    80, 2919, 2337, 2984, 301, 2856, 1066, 3147, 3111, 2655, 2110, 87, 3216, 2083, 1182, 2629, 130,
    433, 3251, 2371, 2620, 2266, 1053, 2671, 3177, 762, 1323, 2804, 1028, 2260, 1894, 3085, 2566,
    1316, 2190, 1537, 2021, 2094, 2402, 3256, 2080, 1108, 1667, 1903, 3052, 2546, 2270, 1168, 3225,
    450, 872, 2125, 3009, 1869, 3168, 2310, 1878, 173, 2679, 887, 1556, 961, 1534, 1929, 732, 2447,
    2374, 1143, 1488, 797, 1055, 3303, 1893, 933, 2900, 2579, 3000, 884, 3178, 3053, 3201, 2501,
    1269, 2541, 424, 3283, 1739, 3132, 1575, 3020, 509, 2736, 1255, 3057, 2249, 2271, 3278, 3317,
    909, 287, 3252, 2603, 3202, 1731, 446, 3224, 3298, 112, 1468, 2822, 1976, 3199, 118, 2281, 984,
    142, 2192, 1405, 3024, 2443, 123, 696, 2327, 2462, 2319, 1908, 355, 2105, 2868, 2342, 354,
    1914, 2320, 2144, 2803, 3051, 1996, 1668, 2590, 1873, 2388, 2441, 627, 1589, 3296, 2009, 1639,
    2820, 2697, 1742, 2291, 566, 3033, 2307, 2091, 3147, 3224, 3141, 715, 2070, 2941, 3319, 3235,
    1755, 3307, 1548, 2732, 2591, 2797, 3167, 2818, 3010, 2665, 3063, 1926, 2223, 1719, 3270, 288,
    1535, 2516, 1686, 3182, 3216, 1658, 3004, 936, 1447, 2823, 3316, 2499, 1025, 3253, 749, 3246,
    204, 2583, 2993, 3022, 2115, 664, 2715, 2586, 3164, 1767, 1813, 1867, 3011, 3309, 788, 3148,
    621, 2950, 1278, 2406, 2922, 42, 930, 3212, 1809, 2758, 2635, 2816, 3323, 2405, 167, 1289,
    3250, 2836, 2757, 1820, 3213, 278, 894, 2322, 1049, 2668, 460, 817, 2162, 198, 2836, 2926,
    2367, 1352, 3065, 2217, 953, 500, 1159, 2099, 903, 1379, 3055, 1257, 390, 2329, 3159, 2286,
    3084, 2810, 3241, 1302, 2911, 2484, 152, 1350, 2369, 3134, 3253, 2883, 2645, 1835, 1915, 2694,
    2342, 1615, 2974, 3191, 244, 1550, 89, 3293, 3306, 1043, 3236, 978, 1674, 635, 3246, 2088,
    1963, 277, 2726, 1853, 2659, 2290, 1890, 2698, 1849, 1692, 2944, 1102, 2165, 1936, 2371, 2843,
    2170, 1675, 1695, 1729, 865, 1830, 839, 2256, 1091, 1337, 2764, 2946, 3269, 1863, 2224, 2790,
    1616, 2167, 538, 1613, 1682, 237, 2451, 3306, 727, 2427, 2968, 2507, 3034, 2594, 2241, 2120,
    2485, 758, 1739, 3098, 1321, 816, 2934, 2663, 2300, 3033, 1436, 2174, 3311, 1513, 1858, 2517,
    3014, 2628, 2544, 3278, 2792, 2181, 863, 2043, 2171, 673, 1752, 2643, 2885, 3007, 1513, 3213,
    2734, 3271, 2773, 3327, 2, 2220, 1332, 1427, 740, 1816, 2817, 444, 2350, 3241, 2695, 2822,
    1598, 2466, 1148, 537, 1715, 3281, 180, 2603, 812, 1471, 148, 850, 1155, 2725, 2010, 1029,
    2745, 395, 2565, 2838, 1895, 2838, 386, 844, 2873, 2752, 735, 2787, 822, 2857, 470, 2602, 3142,
    878, 179, 1647, 1716, 2791, 1681, 1713, 2619, 2769, 1466, 60, 383, 2229, 313, 2238, 2737, 2490,
    2317, 2464, 1600, 3297, 1652, 2823, 486, 124, 1393, 1580, 2227, 385, 1631, 1896, 631, 3103,
    1039, 3062, 641, 600, 343, 3238, 1241, 915, 2694, 1655, 2351, 93, 1453, 855, 36, 3240, 1779,
    171, 138, 2433, 2332, 987, 2194, 1828, 1494, 3180, 446, 284, 3055, 2623, 1979, 3177, 949, 418,
    2027, 88, 1759, 1909, 1043, 170, 2872, 2939, 2072, 274, 231, 2426, 1230, 504, 2829, 2376, 557,
    1928, 1977, 962, 402, 2157, 3131, 2831, 2512, 2869, 661, 2279, 2671, 2435, 3045, 1780, 1509,
    572, 2261, 1327, 2040, 3162, 2588, 1668, 513, 2358, 571, 1520, 117, 2399, 3287, 3319, 923,
    3285, 379, 2708, 2261, 875, 20, 1147, 1462, 2347, 1562, 1828, 743, 614, 3081, 1214, 1971, 1988,
    746, 3125, 83, 3201, 2572, 743, 830, 2665, 714, 1882, 2393, 1989, 1931, 113, 147, 1433, 813,
    2418, 3041, 59, 2233, 2874, 3067, 2346, 3160, 319, 1343, 162, 532, 1622, 3093, 108, 2518, 3238,
    94, 842, 388, 1258, 2614, 2684, 937, 182, 2902, 2686, 1960, 1097, 1038, 3043, 632, 2173, 1690,
    3296, 1697, 2442, 2699, 2552, 2605, 1456, 739, 1973, 1333, 278, 526, 2433, 2673, 1415, 2975,
    2650, 655, 3304, 2974, 3084, 1010, 867, 1002, 2633, 3176, 886, 305, 2340, 2801, 898, 2345,
    1464, 2771, 1794, 1353, 3003, 1855, 3217, 31, 1769, 1163, 3258, 1790, 2390, 1949, 3039, 2420,
    12, 257, 2930, 1080, 272, 2074, 593, 2820, 309, 2138, 3109, 3254, 46, 2905, 2036, 2060, 828,
    2306, 1887, 567, 2445, 329, 2412, 637, 2396, 2212, 702, 2274, 2114, 1841, 1954, 955, 882, 3002,
    1400, 1795, 2368, 1773, 153, 2314, 656, 2265, 2683, 161, 1460, 320, 787, 2457, 1004, 520, 493,
    1059, 2447, 277, 3089, 1662, 2221, 1249, 2049, 925, 2895, 1308, 2623, 1139, 2013, 763, 1696,
    2267, 1691, 3133, 2189, 2006, 2557, 2232, 2112, 2276, 2618, 3205, 2622, 1325, 2896, 3199, 801,
    2978, 2910, 2609, 3242, 2865, 674, 1882, 906, 3303, 2345, 3028, 3257, 1096, 410, 3249, 1813,
    168, 1877, 2964, 1151, 285, 3181, 3288, 2495, 2000, 2405, 3282, 2951, 2862, 2758, 2346, 659,
    2871, 846, 2540, 2765, 2502, 134, 1086, 50, 2383, 2077, 3082, 2833, 116, 1449, 1221, 1785,
    3264, 1947, 2737, 3023, 89, 3212, 2048, 2994, 2222, 606, 2979, 2353, 3139, 2555, 2578, 1580,
    3031, 1021, 3108, 2575, 1177, 2753, 592, 3302, 3202, 2996, 3183, 2003, 2410, 331, 2036, 2973,
    2241, 656, 1136, 221, 352, 680, 1096, 2016, 3134, 1970, 3309, 3134, 2288, 3285, 2776, 2938,
    2378, 171, 2730, 3078, 1341, 1985, 673, 1300, 3243, 756, 1060, 3141, 437, 39, 2414, 1591, 3019,
    3199, 221, 2377, 3074, 617, 2355, 545, 2702, 2525, 2025, 2084, 2903, 48, 2752, 1236, 3096,
    3312, 3122, 3186, 2630, 199, 2814, 2637, 2511, 3193, 3132, 2638, 3257, 1394, 106, 2350, 485,
    2427, 1653, 1395, 1892, 3304, 851, 2740, 2962, 2711, 1719, 1212, 2482, 2467, 1962, 573, 2424,
    1897, 3251, 3000, 2012, 2988, 2624, 840, 2054, 2321, 1516, 1800, 944, 156, 1146, 2047, 121,
    192, 3261, 1887, 2787, 2247, 3059, 3230, 859, 3210, 932, 2084, 2628, 624, 768, 769, 2853, 2496,
    3326, 1666, 1998, 2081, 2724, 238, 196, 2164, 1737, 2945, 1722, 3017, 761, 2015, 1218, 1513,
    1275, 3279, 3015, 1724, 2257, 1235, 1178, 49, 1266, 135, 2767, 541, 2206, 2766, 1384, 2931,
    806, 34, 857, 3233, 2489, 2433, 2239, 2095, 28, 2756, 1811, 3251, 2033, 2857, 729, 2687, 1746,
    2987, 782, 1336, 494, 2718, 2626, 1969, 1216, 2017, 712, 2667, 1804, 2323, 54, 2815, 2963,
    1287, 169, 1548, 246, 2117, 669, 2834, 1440, 3180, 3088, 431, 132, 2504, 3229, 3061, 2744,
    1637, 2011, 2805, 756, 1848, 2500, 1959, 3107, 2591, 3032, 2509, 2634, 2383, 3175, 1981, 2149,
    838, 1727, 866, 1414, 3086, 2434, 1524, 2280, 3276, 3266, 2632, 3076, 868, 2560, 2216, 477,
    1763, 2343, 900, 921, 2073, 1292, 346, 3023, 1922, 529, 1565, 2333, 2842, 1925, 1858, 1834,
    1768, 2884, 1673, 1753, 3237, 440, 2711, 2143, 2785, 2462, 3305, 580, 213, 141, 3119, 224,
    1484, 3070, 1233, 3197, 1978, 1557, 3047, 1595, 487, 881, 3020, 499, 1792, 2773, 7, 417, 3218,
    3178, 1521, 3144, 155, 1132, 1285, 2965, 2954, 811, 3309, 179, 1446, 1853, 3005, 2691, 1966,
    2782, 3211, 3070, 2539, 3103, 3095, 2894, 1328, 2050, 1504, 408, 994, 2565, 2101, 2999, 1964,
    2693, 1579, 1242, 2660, 2515, 1860, 289, 3241, 3318, 334, 3171, 2185, 3097, 1788, 2100, 920,
    995, 3176, 1405, 2802, 1906, 1488, 1806, 2781, 780, 2989, 2839, 3153, 2493, 304, 3073, 2761,
    1962, 2376, 3019, 1164, 488, 178, 3283, 3143, 19, 3163, 598, 2124, 1989, 425, 2866, 2559, 1001,
    3077, 1011, 1728, 1730, 1678, 2182, 3032, 3209, 1120, 3232, 1076, 35, 1573, 1454, 2607, 2739,
    2818, 1641, 1813, 2642, 2539, 1271, 1622, 1629, 387, 1451, 511, 2255, 2557, 1013, 2670, 2441,
    3025, 2139, 2218, 4, 2664, 1480, 2305, 1371, 3309, 3196, 2296, 101, 360, 1624, 296, 2310, 2563,
    2161, 2217, 461, 2852, 3249, 3134, 1644, 940, 2955, 774, 103, 32, 1078, 2932, 2430, 626, 2146,
    1303, 3200, 3304, 972, 81, 1125, 3262, 1262, 2962, 1282, 1518, 2482, 2059, 1373, 3322, 72, 229,
    276, 2999, 3140, 2988, 892, 2782, 3125, 1898, 725, 189, 2086, 2416, 815, 1398, 2460, 1996,
    1114, 3121, 804, 2933, 1695, 1322, 2014, 1098, 3018, 3260, 751, 1847, 1026, 1142, 234, 3245,
    1846, 2939, 2858, 40, 2924, 3124, 1486, 2833, 613, 2316, 166, 1815, 1660, 1428, 1457, 533, 294,
    3134, 2753, 3220, 2806, 2992, 2686, 1064, 2858, 718, 188, 776, 2835, 1874, 2476, 591, 2076,
    2147, 51, 65, 2069, 1881, 1477, 3082, 1468, 764, 2621, 1310, 2619, 2020, 2420, 2191, 610, 917,
    1361, 2421, 2706, 2877, 62, 2326, 1918, 2637, 3175, 514, 2992, 819, 2311, 947, 3179, 2375,
    1623, 1283, 2798, 2322, 1274, 1510, 2051, 2173, 245, 2675, 261, 215, 1299, 1201, 322, 2772,
    1585, 1040, 2118, 762, 3324, 2498, 1850, 2616, 2278, 1526, 2870, 1273, 683, 1135, 1223, 3079,
    2652, 3069, 965, 225, 2402, 436, 1197, 2727, 1984, 3169, 336, 2599, 2266, 1584, 671, 1571,
    2395, 3028, 2413, 1751, 1675, 2380, 1438, 1900, 3145, 778, 2783, 2145, 2467, 767, 1115, 2629,
    2949, 1828, 2733, 2888, 2354, 1187, 3075, 3080, 3155, 2408, 2834, 2894, 704, 2192, 2108, 3082,
    1247, 2223, 3092, 2131, 2682, 3322, 3157, 2123, 874, 1500, 2709, 442, 2819, 3045, 2075, 408,
    2685, 2176, 2032, 2915, 2347, 2299, 1693, 2935, 3186, 212, 970, 3306, 2119, 1702, 2595, 110,
    3300, 2260, 1519, 3174, 695, 1919, 2444, 1368, 1888, 2292, 1906, 3193, 2245, 2789, 1718, 2279,
    1927, 3200, 2377, 3323, 667, 2119, 392, 142, 115, 1522, 2436, 2550, 2701, 3265, 2356, 2522,
    2205, 1083, 2768, 3276, 1714, 1649, 1149, 56, 293, 734, 1458, 1827, 3228, 988, 1923, 2484,
    1424, 279, 108, 2805, 338, 492, 1338, 2880, 2847, 264, 3129, 2159, 693, 1512, 1671, 2886, 2735,
    1949, 3021, 348, 1789, 1163, 2371, 1231, 3203, 2824, 3039, 954, 3021, 1842, 2584, 2717, 1058,
    1337, 521, 339, 2439, 1832, 880, 957, 3260, 1160, 282, 2111, 2811, 3066, 3114, 3190, 1762,
    1830, 2217, 834, 3027, 1295, 2264, 3017, 1622, 2026, 377, 3301, 2236, 315, 2877, 2243, 771,
    3312, 1802, 2669, 2057, 2484, 1701, 1199, 3307, 3013, 2865, 871, 1990, 2810, 483, 1075, 1560,
    2349, 1657, 1152, 595, 2710, 976, 3290, 38, 2860, 2313, 2403, 2002, 2022, 131, 930, 3089, 3135,
    70, 3324, 2149, 3282, 1955, 2519, 3258, 2902, 1181, 2026, 1553, 2614, 8, 1272, 2742, 3063, 202,
    3248, 1291, 993, 922, 649, 3288, 2994, 206, 2156, 2363, 963, 3071, 2742, 2250, 2524, 2564,
    1635, 2746, 144, 552, 2900, 1784, 1256, 3114, 843, 1630, 1591, 2228, 1170, 61, 1531, 2707,
    3166, 2052, 468, 363, 2569, 2519, 3076, 1226, 748, 3320, 2914, 588, 470, 2283, 3291, 2387, 373,
    469, 3287, 823, 58, 809, 2956, 2104, 2329, 1909, 679, 1220, 1882, 2083, 1788, 507, 3022, 3279,
    1293, 3029, 1157, 3100, 2548, 3270, 2571, 522, 2598, 644, 2318, 907, 3319, 371, 393, 2411,
    3030, 2446, 2767, 1930, 1473, 2394, 639, 672, 1203, 1342, 626, 1497, 11, 2876, 1297, 2237,
    1606, 2230, 150, 2137, 3043, 2821, 2982, 2339, 3072, 887, 3313, 2855, 2035, 2985, 1748, 2089,
    2309, 821, 3077, 736, 2198, 57, 3251, 1940, 910, 1861, 2839, 3038, 139, 1559, 447, 484, 1161,
    107, 2603, 1425, 1334, 784, 1880, 1543, 2073, 1383, 3109, 2207, 99, 2298, 586, 2916, 3128, 517,
    3004, 1880, 676, 2676, 2766, 2929, 2773, 13, 458, 2713, 250, 3078, 1411, 1917, 2712, 1839,
    1385, 1042, 1549, 1760, 1486, 2228, 2293, 2899, 139, 1105, 3141, 2863, 3244, 754, 1143, 2425,
    1255, 275, 2449, 73, 1200, 3233, 2315, 966, 1400, 3314, 1190, 1952, 76, 2962, 675, 262, 2172,
    140, 3257, 581, 1626, 2362, 3106, 16, 693, 404, 2582, 1844, 2622, 412, 3064, 2813, 2523, 1799,
    2163, 1104, 1277, 2900, 3260, 1127, 534, 2085, 775, 726, 1709, 3284, 3311, 1176, 3324, 1445,
    2602, 1646, 366, 879, 489, 3272, 836, 1014, 1981, 2729, 2871, 3211, 1044, 1288, 2221, 742,
    2320, 1563, 1115, 1459, 3008, 2684, 1641, 2423, 2809, 1131, 945, 3145, 2181, 2190, 2744, 2641,
    849, 1642, 3131, 2610, 551, 393, 243, 3118, 2632, 214, 3266, 1568, 3086, 2766, 1910, 1267,
    2503, 2698, 119, 2023, 2529, 26, 430, 2827, 915, 349, 2084, 191, 2792, 2469, 958, 2398, 1508,
    3185, 2835, 146, 1473, 1932, 1088, 575, 2596, 3228, 1112, 1162, 3059, 32, 1536, 359, 2904,
    3129, 683, 2208, 2471, 2254, 840, 1452, 2614, 2352, 2890, 3292, 2174, 1134, 3271, 2129, 2251,
    2576, 2939, 3126, 2918, 2039, 892, 2262, 2961, 2299, 2783, 3284, 1476, 786, 3297, 2924, 3136,
    3243, 1908, 2068, 717, 52, 1490, 281, 1656, 1609, 1461, 3042, 292, 535, 2396, 1464, 1387, 64,
    2068, 2929, 1087, 1179, 1187, 1375, 2421, 3102, 2593, 1823, 2923, 749, 2905, 2934, 2822, 1572,
    3222, 3157, 807, 104, 1968, 3218, 3118, 3150, 2096, 128, 2530, 2358, 2904, 2875, 1643, 1498,
    3060, 3144, 2154, 208, 2167, 2972, 2336, 1387, 3251, 2996, 2959, 416, 2882, 2774, 2871, 832,
    3272, 1664, 3328,
];

#[derive(Clone, Copy)]
pub struct PrimeField3329 {
    val: usize,
}

impl Debug for PrimeField3329 {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "{:?}", self.val)
    }
}

impl FiniteField for PrimeField3329 {
    fn dimension() -> usize {
        1
    }
    fn is_zero(&self) -> bool {
        self.val == 0
    }

    fn zero() -> Self {
        Self { val: 0 }
    }

    fn one() -> Self {
        Self { val: 1 }
    }

    fn neg(&self) -> Self {
        Self::zero().sub(self)
    }

    fn add(&self, other: &Self) -> Self {
        Self {
            val: (self.val + other.val) % Self::order(),
        }
    }

    fn sub(&self, other: &Self) -> Self {
        let o = Self::order();

        Self {
            val: (self.val + (o - other.val)) % o,
        }
    }

    fn mul(&self, other: &Self) -> Self {
        Self {
            val: (&self.val * &other.val) % Self::order(),
        }
    }

    fn inv(&self) -> Result<Self, String> {
        if self.is_zero() {
            return Err("DIV0".to_string());
        }

        Ok(Self::from_int(INV_3329[self.val]))
    }

    fn div(&self, other: &Self) -> Result<Self, String> {
        Ok(self.mul(&other.inv()?))
    }
}

impl PartialEq for PrimeField3329 {
    fn eq(&self, other: &Self) -> bool {
        self.val == other.val
    }
}

impl Eq for PrimeField3329 {}

impl Default for PrimeField3329 {
    fn default() -> Self {
        Self { val: 0 }
    }
}

impl PrimeField3329 {
    #[inline]
    pub const fn order() -> usize {
        3329
    }

    pub const fn from_int(x: usize) -> Self {
        let o = Self::order();
        Self { val: (x + o) % o } //i can have negative numbers also in the right range
    }

    pub const fn to_int(&self) -> usize {
        self.val
    }
}