#!/usr/bin/env python3

import math
import numpy as np
import paho.mqtt.client as mqtt
import json
from time import sleep, time

# ROS2 libraries
import rclpy
from rclpy.node import Node
from geometry_msgs.msg import Twist
from sensor_msgs.msg import LaserScan

from Rosmaster_Lib import Rosmaster

# Constants
RAD2DEG = 180 / math.pi

class FollowRoad(Node):
    def __init__(self, name):
        super().__init__(name)

        # Initialize STM32 communication
        self.car = Rosmaster()
        self.car.set_car_type(1)  
        self.car.create_receive_threading()

        # Initialize MQTT communication
        self.mqtt_client = mqtt.Client()
        self.mqtt_client.on_connect = self.on_connect
        self.mqtt_client.on_message = self.on_message
        self.mqtt_client.connect("localhost", 1883, 60)
        self.mqtt_client.loop_start()

        # Parameters
        self.linear_speed = 0.4
        self.angular_speed = 0.2  
        self.distance_front= 2.0  # actual distance front to the nearest object/person
        self.response_dist_lat = 0.3  # Obstacle detection distance
        self.response_dist_front = 0.5  # Obstacle detection distance
        self.response_clearance_front = 1.8 # Clear from obstacle distance for speed control
        self.laser_angle = 130.0  # Angle for laser range scanning

        # Flags and states
        self.obstacle_detected = False
        self.obstacle_clearance_front = True
        self.front_stop_area = False
        self.ready = False
        self.follow_line_active = False
        self.error = False
        self.emergency_stopped = False

        self.obstacle_avoid_front = False
        self.obstacle_avoid_left = False
        self.obstacle_avoid_right = False
        self.obstacle_avoid_all = False

        # Initialize LED to blue
        self.set_led_color(0, 0, 255)

        # Lidar data variables
        self.right_warning = 0
        self.left_warning = 0
        self.front_warning = 0

        # Watchdog-related attributes
        self.watchdog_timeout = 3.0  # Timeout in seconds
        self.watchdog_timer = None  # Placeholder for the watchdog timer

        # Start the watchdog timer when the node is initialized
        self.reset_watchdog()
      
        # Subscribers
        self.sub_laser = self.create_subscription(LaserScan, '/scan', self.lidar_callback, 10)
        self.publish_battery_info()
        # Timer for battery voltage publishing 
        self.battery_timer = self.create_timer(5.0, self.publish_battery_info)

        # Timer for regular data publishing
        self.timer = self.create_timer(0.1, self.follow_line_and_avoid_obstacles)
        
        print(f"Autonomous follow line ROS2 node v2")


    def set_led_color(self, r, g, b):
        """Sets the RGB LED color on the robot."""
        self.car.set_colorful_lamps(0xFF, r, g, b)  # 0xFF means all LEDs
        
    def on_connect(self, client, userdata, flags, rc):
        """MQTT connection callback."""
        print(f"Connected to MQTT with result code {rc}")
        self.ready = True
        self.set_led_color(0, 0, 255)  # Blue LED when ready
        client.subscribe("robot/control")
        print("Subscribed to MQTT topic 'robot/control'")

    def publish_battery_info(self):
        """Fetches and publishes the battery voltage."""
        try:
            battery_voltage = self.car.get_battery_voltage()
            print(f"Battery Voltage: {battery_voltage:.1f}V")
        except Exception as e:
            print(f"Cannot get battery info: {e}")

    def reset_watchdog(self):
        """Resets the watchdog timer to stop the robot if no message is received."""
        if self.watchdog_timer is not None:
            self.watchdog_timer.cancel()  # Cancel any existing watchdog timer
        
        # Create a new watchdog timer that stops the robot after timeout duration
        self.watchdog_timer = self.create_timer(self.watchdog_timeout, self.stop_robot_due_to_inactivity)
        #print("Watchdog timer reset.")

    def stop_robot_due_to_inactivity(self):
        """Stops the robot due to inactivity (no messages received)."""
        if self.follow_line_active:
            self.set_led_color(255, 0, 0)  # Red LED for inactivity
            print("No message received within the timeout. Robot stopped.")
            self.follow_line_active = False
        self.car.set_car_motion(0, 0, 0)  # Stop the robot
        
    def on_message(self, client, userdata, msg):
        """MQTT message callback for Follow-Line activation."""
        print(f"Received MQTT message: {msg.payload.decode('utf-8')}")
        try:
            # Decode the payload into a string and then parse the JSON data
            message_str = msg.payload.decode('utf-8')
            message = json.loads(message_str)  # Convert the JSON string to a dictionary
            self.reset_watchdog()
            # Extract the command and angle from the message
            command = message.get("command", "")  # Get the command or default to an empty string
            angle = message.get("angle", 0)  # Get the angle or default to 0

            if command == "activate":
                self.follow_line_active = True
                self.emergency_stopped = False
                self.set_led_color(0, 255, 0)  # Green LED when Follow-Line is activated
                print("Follow-Line activated")

            elif command == "deactivate":
                self.follow_line_active = False
                self.car.set_car_motion(0, 0, 0)
                self.set_led_color(0, 0, 255)  # Blue LED when deactivated
                print("Follow-Line deactivated")

            elif command == "emergency_stop":
                self.handle_emergency_stop()

            elif command == "forward" and self.follow_line_active:
                if not self.obstacle_detected and not self.emergency_stopped:
                    
                    if self.obstacle_clearance_front:
                        """Full forward speed if no obstacle in front"""
                        self.car.set_car_motion(self.linear_speed, 0, 0)
                        print("Moving forward fast")
                        
                    else:
                        """Proportional forward speed wrt to front obstacle/person"""
                        percentage = (self.distance_front - self.response_dist_front) / (self.response_clearance_front - self.response_dist_front)
                        self.car.set_car_motion(self.linear_speed*percentage, 0, 0)
                        print(f"Moving forward proportionally with speed :{self.linear_speed*percentage}")

            elif command == "left" and self.follow_line_active:
                if (angle<=15):
                    """Translating when angle below 15 degrees"""
                    adjusted_angular_speed = self.angular_speed * 0.8
                    adjusted_linear_speed = self.linear_speed * 0.9
                    if not self.obstacle_detected and not self.emergency_stopped:
                        self.car.set_car_motion(adjusted_linear_speed,0, adjusted_angular_speed)
                        print(f"Translating left with angular speed: {adjusted_angular_speed}, linear speed: {adjusted_linear_speed}")
                elif(angle>15 and angle<=28):
                    """Turning slow when angle below 28 degrees"""
                    adjusted_angular_speed = self.angular_speed*0.5
                    adjusted_linear_speed = self.linear_speed * 0.8
                    if not self.obstacle_detected and not self.emergency_stopped:
                        self.car.set_pid_param(0.1,3,3, 0)
                        self.car.set_car_motion(adjusted_linear_speed, adjusted_angular_speed, 0)  
                        print(f"Turning left with angular speed: {adjusted_angular_speed}, linear speed: {adjusted_linear_speed}")
                else:
                    """Turning faster when angle over 28 degrees"""
                    adjusted_angular_speed = self.angular_speed*0.8
                    adjusted_linear_speed = self.linear_speed * 0.7
                    if not self.obstacle_detected and not self.emergency_stopped:
                        self.car.set_pid_param(0.1,3,3, 0) # PID parameter adjustment
                        self.car.set_car_motion(adjusted_linear_speed, adjusted_angular_speed, 0)  
                        print(f"Turning fast left with angular speed: {adjusted_angular_speed}, linear speed: {adjusted_linear_speed}")

            elif command == "right" and self.follow_line_active:
                if (angle<=15):
                    """Translating when angle below 15 degrees"""
                    adjusted_angular_speed = self.angular_speed *0.8
                    adjusted_linear_speed = self.linear_speed * 0.9
                    if not self.obstacle_detected and not self.emergency_stopped:
                        self.car.set_car_motion(adjusted_linear_speed,0, -adjusted_angular_speed)
                        print(f"Translating right with angular speed: {adjusted_angular_speed}, linear speed: {adjusted_linear_speed}------------------")
                elif(angle>15 and angle<=28):
                    """Turning slow when angle below 28 degrees"""
                    adjusted_angular_speed = self.angular_speed*0.5
                    adjusted_linear_speed = self.linear_speed * 0.8
                    if not self.obstacle_detected and not self.emergency_stopped:
                        self.car.set_pid_param(0.1,3,3, 0)
                        self.car.set_car_motion(adjusted_linear_speed, -adjusted_angular_speed, 0)  
                        print(f"Turning right with angular speed: {adjusted_angular_speed}, linear speed: {adjusted_linear_speed}--------------------")
                
                else:
                    """Turning faster when angle over 28 degrees"""
                    adjusted_angular_speed = self.angular_speed*0.8
                    adjusted_linear_speed = self.linear_speed * 0.7
                    if not self.obstacle_detected and not self.emergency_stopped:
                        self.car.set_pid_param(0.1,3,3, 0) # PID parameter adjustment
                        self.car.set_car_motion(adjusted_linear_speed, -adjusted_angular_speed, 0)  
                        print(f"Turning fast right with angular speed: {adjusted_angular_speed}, linear speed: {adjusted_linear_speed}--------------")

            elif command == "stop":
                if self.follow_line_active:
                    if self.front_stop_area:   # Avoid stopping for errors in the bounding box
                        self.car.set_car_motion(0, 0, 0)  # Stop motors
                        print("Robot stopped")
                        self.set_led_color(255, 0, 0)  # Red LED for stop
                        sleep(0.5)
                    else:
                        # Ignore the stop command if the bounding box is faulty but there's no obstacle in front
                        print("Stop command ignored due to clear LiDAR data in front.")

            elif command == "backward":
                if self.follow_line_active:
                    if not self.obstacle_avoid_all and self.front_stop_area:
                        adjusted_linear_speed = self.linear_speed * 0.2
                        self.car.set_car_motion(-adjusted_linear_speed, 0, 0)
                        print("Robot moving backwards")
                        self.set_led_color(255, 0, 0)  # Red LED for moving backwards
                        sleep(0.6)
                    else:
                        # Ignore the stop command if the bounding box is faulty but there's no obstacle in front
                        print("Backward command ignored due to clear LiDAR data in front.")

        except json.JSONDecodeError as e:
            print(f"Failed to decode MQTT message: {e}")
        except KeyError as e:
            print(f"Key error: {e}")
        except Exception as e:
            print(f"Unexpected error: {e}")

    def handle_emergency_stop(self):
        """Handles an emergency stop by stopping the robot """
        self.follow_line_active = False
        self.emergency_stopped = True
        self.car.set_car_motion(0, 0, 0)
        self.set_led_color(255, 0, 0)  # Red LED for emergency stop
        print("Emergency stop received. Robot stopped and follow function deactivated.")

    def lidar_callback(self, scan_data):
        """Processes LiDAR data for obstacle detection."""
        ranges = np.array(scan_data.ranges)
        self.right_warning = 0
        self.left_warning = 0
        self.front_warning = 0


        for i in range(len(ranges)):
                angle = (scan_data.angle_min + scan_data.angle_increment * i) * RAD2DEG
                if 160 > angle > 180 - self.laser_angle:
                    if ranges[i] < self.response_dist_lat:
                        self.right_warning += 1

                if -160 < angle < self.laser_angle - 180:
                    if ranges[i] < self.response_dist_lat:
                        self.left_warning += 1

                if abs(angle) > 160:
                    self.distance_front = ranges[i]
                    if ranges[i] <= self.response_dist_front:
                        self.front_warning += 1
                    if ranges[i]  < self.response_clearance_front:
                        self.obstacle_clearance_front = False
                    if ranges[i]  >= self.response_clearance_front:
                        self.obstacle_clearance_front = True
                    if ranges[i]  < self.response_clearance_front-0.5:
                        self.front_stop_area = True
                    if ranges[i]  >= self.response_clearance_front-0.5:
                        self.front_stop_area = False

        """
        #obstacle left
        if self.left_warning > 10:
            self.obstacle_detected = True
            # self.obstacle_avoid_left = True
            print(f"Lidar data: Front: {self.front_warning}, Left: {self.left_warning}, Right: {self.right_warning}")
            
        #obstacle right
        elif self.right_warning > 10:
            self.obstacle_detected = True
            # self.obstacle_avoid_right =  True
            print(f"Lidar data: Front: {self.front_warning}, Left: {self.left_warning}, Right: {self.right_warning}")
            
        #obstacle front
        elif self.front_warning > 10:
            self.obstacle_detected = True
            # self.obstacle_avoid_front = True
            print(f"Lidar data: Front: {self.front_warning}, Left: {self.left_warning}, Right: {self.right_warning}")

        #obstacle in all directions
        elif self.front_warning > 10 and self.left_warning > 10 and self.right_warning > 10:
            self.obstacle_detected = True
            # self.obstacle_avoid_all = True
            print(f"Lidar data: Front: {self.front_warning}, Left: {self.left_warning}, Right: {self.right_warning}")         
        
        else:
            if self.follow_line_active:
                self.set_led_color(0, 255, 0)  # green LED for obstacle cleared and follow line active
            else:
                self.set_led_color(0, 0, 255)  # blu LED for obstacle cleared and follow line not active
                    
            self.obstacle_detected = False
            self.obstacle_avoid_front = False
            self.obstacle_avoid_left = False
            self.obstacle_avoid_right = False
            self.obstacle_avoid_all = False
        """
            
    def follow_line_and_avoid_obstacles(self):
        """Logic for following the person and avoiding obstacles."""
        
        # Wait if the system is not ready
        if not self.ready:
            print("Waiting for MQTT connection...")
            return

        # If Follow-Line is active and there are no obstacles, follow the person
        if self.follow_line_active and not self.obstacle_detected:
            self.set_led_color(0, 255, 0)  # Green LED when Follow-Line is active and no obstacles
            print("No obstacles. Following line.")
            return

        # If Follow-Line is not active but no obstacles detected, ensure robot stops
        if not self.follow_line_active and not self.obstacle_detected:
            self.set_led_color(0, 0, 255)  # Blue LED for Follow-Line not active and no obstacles
            self.car.set_car_motion(0, 0, 0)  # Ensure the robot fully stops if inactive
            return
        
        # Stop if Follow-Line mode is not activated
        if not self.follow_line_active:
            print("Follow-Line is not activated ------------------.")
            self.car.set_car_motion(0, 0, 0)  # Ensure the robot fully stops if not active
            self.set_led_color(0, 0, 255)  # Blue LED for Follow-Line not active
            return

        if self.follow_line_active and self.obstacle_detected:
            self.set_led_color(255, 0, 0)  # Red LED for obstacle detected
            if self.obstacle_avoid_all:
                """Obstacle in all directions"""
                print("Obstacle detected. Robot is stopped.")
                self.car.set_car_motion(0, 0, 0)  # Ensure the robot fully stops on obstacle detection
            
            elif self.obstacle_avoid_right and self.obstacle_avoid_left:
                """Obstacles on both sides"""
                self.car.set_car_motion(self.linear_speed * 0.5, 0, 0)  # Obstacle both left and right slowly move forward
                print("Obstacles detected on both sides. Moving slowly forward.")
                                
            elif self.obstacle_avoid_left and self.obstacle_avoid_front:
                """Obstacle on the left and front"""
                self.car.set_car_motion(0, 0, -self.angular_speed * 1.2)  # If obstacle also in front, rotate to the right
                print("Obstacle detected on the left. Robot moving right.")
                    
            elif self.obstacle_avoid_right and self.obstacle_avoid_front:
                """Obstacle on the right and front"""
                self.car.set_car_motion(0, 0, self.angular_speed * 1.2) # If obstacle also in front, rotate to the left
                print("Obstacle detected on the right. Robot moving left.")
                    
            elif self.obstacle_avoid_left:
                """Obstacle on the left"""
                self.car.set_car_motion(self.linear_speed * 0.5, 0, -self.angular_speed * 1.2)  # Move forward-right
                print("Obstacle detected on the left. Robot moving forward right.")
                
            elif self.obstacle_avoid_right:
                """Obstacle on the right """
                self.car.set_car_motion(self.linear_speed * 0.5, 0, self.angular_speed * 1.2)   # Move forward-left
                print("Obstacle detected on the right. Robot moving forward left.")
                
            elif self.obstacle_avoid_front:
                """Obstacle in front"""
                self.car.set_car_motion(-self.linear_speed * 0.3, 0, 0) # Move backwards
                print("Obstacle detected in front. Robot moving backwards.")
                
            return
          

def main(args=None):
    rclpy.init(args=args)
    node = FollowRoad("follower_road")
    try:
        rclpy.spin(node)
    except KeyboardInterrupt:
        node.car.set_car_motion(0, 0, 0) # Stop the robot motors
        node.set_led_color(0, 0, 0)  # Turn off LEDs on shutdown
        node.destroy_node()
    finally:
        rclpy.shutdown()

if __name__ == '__main__':
    main()

"""
Riceve i comandi MQTT, verifica se la modalità "Follow Line" è attiva 
e applica le correzioni angolari al robot.
Utilizza il Lidar per rilevare ostacoli e disattiva "Follow Line" se necessario, fermando il robot.
I messaggi di comando vengono inviati tramite MQTT, dove robot/control e robot/commands sono i topic per inviare rispettivamente comandi generici e specifici al nodo di movimento.
"""
