import numpy as np
from scipy.spatial import distance as dist
import paho.mqtt.client as mqtt
from collections import OrderedDict
import json

# MQTT broker details
broker_ip = "localhost"  
broker_port = 1883  # Default MQTT port

# MQTT topic
topic = "robot/control"

def on_connect(client, userdata, flags, rc):
    if rc == 0:
        print("Module connected to MQTT Broker!")
    else:
        print(f"Module failed to connect, return code {rc}")

# Initialize the MQTT client
client = mqtt.Client()

# Set up the connection callback
client.on_connect = on_connect

# Connect to the broker
client.connect(broker_ip, broker_port, 60)

# Start the loop
client.loop_start()

def send_activate():
    command = {"command": "activate"}
    client.publish("robot/control", json.dumps(command))

def send_deactivate():
    command = {"command": "deactivate"}
    client.publish("robot/control", json.dumps(command))


class FollowMe():

	# Class init function (max_disapp_frames is the number of
	# consecutive frames before the centroid is deregistered)
	def __init__(self, sens = 1.4, max_reset_count =3, max_start_stop = 3, sleep_time = 24):
		self.centroids = OrderedDict()
		self.last_sizes  = OrderedDict()
		self.counters = OrderedDict()
		self.flags =OrderedDict()
		self.sensitivity = sens
		self.disappeared = OrderedDict()
		self.max_reset_counter = max_reset_count
		self.max_start_or_stop = max_start_stop
		self.count_sleep = sleep_time
		self.max_sleep = sleep_time

	def register(self, centroidID, dimX):
		# when registering an object we use the next available object
		# ID to store the centroid
		self.centroids[centroidID] = centroidID
		self.last_sizes[centroidID] = dimX
		self.counters[centroidID] = 0
		self.flags[centroidID] = 0
		self.disappeared[centroidID] = 0

	def deregister(self, centroidID):
		# to deregister an object ID we delete the object ID from
		# both of our respective dictionaries
		del self.centroids[centroidID] 
		del self.last_sizes[centroidID]
		del self.counters[centroidID]
		del self.flags[centroidID]
		del self.disappeared[centroidID]
	
	def follow_update(self, centroids, boxes): 		
		# It manages the case in which there are no detected objects (boxes)
		
		if len(boxes) == 0:
			self.empty_case(centroids)
			return -1  

		for (centroidID, centroid) in centroids.items():
			i=0

			for j in range(0, len(boxes)):
				cord = self.center_calculator(boxes[j])
				
				# The centroid match a boxe (there is a corresp. object in the image)
				if ( cord[0] == centroid[0] and cord[1] == centroid[1] ):				
					i=1
					# Checks if the centroid is new or it's already saved
					saved_centroidsID = self.centroids.keys()

					if any( (centroidID == c) for c in saved_centroidsID):
						print("\nupdate info")
						# Updates counter, flag	and last x dimencion
						self.update_info(centroidID,(boxes[j][2] - boxes[j][0]) )			
					else:
						print("register")
						# Registers new centroid
						self.register( centroidID, (boxes[j][2] - boxes[j][0]) )
					break
			
			# The centroid doesn't match any object's box, so it's a pending ID
			if(i==0):

				saved_centroidsID = self.centroids.keys()
				if not any( (centroidID == c) for c in saved_centroidsID):
					# Registers new centroid
					self.register( centroidID, (boxes[j][2] - boxes[j][0]))
				else:
					self.disappeared[centroidID] += 1
					if(self.disappeared[centroidID] == self.max_reset_counter):
						self.counters[centroidID] = 0
						self.disappeared[centroidID] = 0 	

		for (centroidID, centroid) in self.centroids.items():
			if(self.counters[centroidID] > self.max_start_or_stop):
				self.counters[centroidID] = 0
				print("Activated ID:"+str(centroidID))
				self.count_sleep = 0
				send_activate()
				return centroidID
		return -1	

	def unfollow_update(self, centroids, boxes, target):
		
		if len(boxes) == 0:
			self.empty_case(centroids)
			return target  

		# Search among the current boxes, which one corresponds 
		# to the target centroid and in the case of a miss, the
		# control ends
		for (centroidID, centroid) in centroids.items():		
			i=0
			for j in range(0, len(boxes)):
			
				cord = self.center_calculator(boxes[j])
				# The target centroid match a boxe (there is a corresp. object in the image)
				if ( cord[0] == centroid[0] and cord[1] == centroid[1] and centroidID == target ):				
					i=1
					# Checks if the target centroid is not in the list of centroids 
					saved_centroidsID = self.centroids.keys()
					if not any( (centroidID == c) for c in saved_centroidsID):
						# Error case
						print("Error: the target has not been selected before")	
						return -2
					else:
						self.update_info(centroidID, (boxes[j][2] - boxes[j][0]))
					break  
		
			# If the target has been update check the counter and return.
			if( i == 1 ):
				# In this case the target is asking to stop the Follow mode
				if ( self.counters[centroidID] > self.max_start_or_stop):
					self.counters[centroidID] = 0 
					self.count_sleep = 0
					print("Deactivated ID:"+str(centroidID)) 
					send_deactivate()
					return -1
				# Else just return curret the target ID
				return target
			
		return target

	def update_info(self, id, dim):

		print("id")		
		print(id)
		print("flags")
		print(self.flags)
		print("dim")
		print(dim)
		print("last sizes")
		print(self.last_sizes[id])

		if(self.flags[id]==0):
			print("---- primo if")
			print(self.flags[id])
			print(self.last_sizes[id] * self.sensitivity)
			if( dim > (self.last_sizes[id] * self.sensitivity)):
				print("---- secondo if")
				self.counters[id]+=1
				self.flags[id]=1
				
				print(str(id) +"  current: "+ str((dim))+ " - old: "+ str(self.last_sizes[id]) + " count:"+ str(self.counters[id]))
				self.last_sizes[id]=dim
		else:
			print("----- else")
			print(dim * self.sensitivity)
			if( (dim * self.sensitivity) < self.last_sizes[id]):
				print("----- terzo if")
				self.counters[id]+=1
				self.flags[id]=0
				
				print(str(id) + "  current: "+ str((dim))+ " - old: "+ str(self.last_sizes[id]) + " count:"+str(self.counters[id]))				
				self.last_sizes[id]=dim

	# Function that loop over all the bounding box rectangles and 
	# use the bounding box coordinates to derive the centroids
	def center_calculator(self,bxs):

		cX = int((bxs[0] + bxs[2]) / 2.0)
		cY = int((bxs[1] + bxs[3]) / 2.0)
		return (cX, cY)


	def empty_case(self,objects):

		for (centroidID, centroid) in objects.items():
			# It manages the case in which the centroid is present in the list
			# so resets counter
			saved_centroidsID = self.centroids.keys()
			if any( (centroidID == c) for c in saved_centroidsID):
				self.disappeared[centroidID] += 1
				if(self.disappeared[centroidID] == self.max_reset_counter):
					self.counters[centroidID] = 0
					self.disappeared[centroidID] = 0 

			# It manages the case in which the centroid is not present:
			# that means that the centroid has been deleted 

