import jetson_inference
import jetson_utils
import time
import cv2
import numpy as np 
import tkinter as tk
from tkinter import ttk
from tkinter import messagebox
import math
import matplotlib.pyplot as plt
import psutil  # For monitoring resource usage

def Follow_Line_Core(send_activate, send_command, send_stop, truncate, low_b, high_b):

    print("Follow Line started")

    # Initial parameters
    threshold = 10  # Threshold in pixels for considering an "acceptable" error
    errors = []     # List to record the error of each frame
    steering_angles = []  # List to record the steering angle
    processing_times = []  # List to record processing time for each frame
    cpu_usages = []  # List to record CPU usage
    mem_usages = []  # List to record memory usage

    cap = cv2.VideoCapture(0)
    cap.set(3,160)
    cap.set(4,120)

    max_steering_angle = 30  # Limits the maximum steering angle

    try:
       
        while True:
            t0 = time.time()  # Start measuring time for the frame

            ret, frame = cap.read()
            hsv_frame = cv2.cvtColor(frame, cv2.COLOR_BGR2HSV)
            mask = cv2.inRange(hsv_frame, low_b, high_b)
            contours = cv2.findContours(mask, cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE)[-2]  # Prende solo il primo valore

            if len(contours) > 0:
                c = max(contours, key=cv2.contourArea)
                M = cv2.moments(c)
                
                if M["m00"] != 0:
                    cx = int(M["m10"] / M["m00"])
                    cy = int(M["m01"] / M["m00"])
                    #print(f"cx: {cx}")
                    #print(f"cy: {cy}")

                    # Calculate the horizontal error (distance from center)
                    image_center_x = frame.shape[1] / 2
                    error_x = cx - image_center_x
                    errors.append(error_x)

                    # Calculate the angle based on the error
                    steering_angle = (error_x / image_center_x) * max_steering_angle
                    print(f"angle: {steering_angle}")
                    steering_angles.append(steering_angle)  # Save the angle for analysis

                    if abs(error_x) < 10:  # Center
                        print("On track forward!")
                        send_activate()
                        send_command("forward")
                    elif error_x < -10:  # Left
                        print("Turning left!")
                        send_activate()
                        send_command("left", int(-1 * steering_angle))
                    elif error_x > 10:  # Right
                        print("Turning right!")
                        send_activate()
                        send_command("right", int(steering_angle))

                    cv2.circle(frame, (cx,cy), 5, (255,255,255), -1)
            
                cv2.drawContours(frame, [c], -1, (0, 255, 0), 1)

            cv2.imshow("Mask", mask)
            cv2.imshow("Frame", frame)

            # Resource monitoring
            cpu_usages.append(psutil.cpu_percent())  
            mem_usages.append(psutil.virtual_memory().percent)

            # Calculate the processing time for the frame
            processing_time = time.time() - t0
            processing_times.append(processing_time)
            
            if cv2.waitKey(1) & 0xff == ord('q'):
                break

    finally:
        cap.release()
        cv2.destroyAllWindows()

    # Avoid calculations on empty lists
    if len(errors) == 0 or len(processing_times) == 0:
        print("No data collected for metrics.")
        cv2.destroyAllWindows()  # Close OpenCV windows
        plt.close('all')  # Close any open plots
        exit()  # End the script

    # # Statistics
    errors_np = np.array(errors)
    mean_error = np.mean(errors_np)
    std_error = np.std(errors_np)
    percentage_within_threshold = np.sum(np.abs(errors_np) < threshold) / len(errors_np) * 100
    steering_np = np.array(steering_angles)
    processing_times_np = np.array(processing_times)
    mean_processing_time = np.mean(processing_times_np)
    cpu_usages_np = np.array(cpu_usages)
    mem_usages_np = np.array(mem_usages)
    mean_steering = np.mean(steering_np)
    mean_cpu = np.mean(cpu_usages_np)
    mean_mem = np.mean(mem_usages_np)
    fps_values = 1.0 / processing_times_np
    mean_fps = np.mean(fps_values)

    print(f"Mean error: {mean_error:.3f}")
    print(f"Standard deviation: {std_error:.3f}")
    print(f"Percentage of errors within the threshold of {threshold} pixels: {percentage_within_threshold:.3f}%")
    print(f"Mean steering angle: {mean_steering:.3f}")
    print(f"Mean CPU usage: {mean_cpu:.3f} %")
    print(f"Mean Memory usage: {mean_mem:.3f} %")
    print(f"Mean processing time per frame: {mean_processing_time:.3f} sec")
    print(f"Mean FPS: {mean_fps:.3f}")

    # Error over time plot
    plt.figure(figsize=(10, 5))
    plt.plot(errors_np, label='Horizontal error (pixels)')
    plt.axhline(y=mean_error, color='r', linestyle='--', label='Mean error')
    plt.xlabel("Frame")
    plt.ylabel("Error (pixels)")
    plt.legend()
    plt.grid(True)
    plt.title("Error over time")
    #plt.show()

    # Boxplot of the error distribution
    plt.figure(figsize=(5, 7))
    plt.boxplot(errors_np, vert=True, patch_artist=True)
    plt.title("Distribution of horizontal error")
    plt.ylabel("Error (pixels)")
    #plt.show()

    # Steering angle over time plot
    plt.figure(figsize=(10, 5))
    plt.plot(steering_np, label='Steering angle')
    plt.axhline(y=mean_steering, color='r', linestyle='--', label='Mean angle')
    plt.xlabel("Frame")
    plt.ylabel("Steering angle (degrees)")
    plt.legend()
    plt.grid(True)
    plt.title("Steering command over time")
    #plt.show()

    # CPU and Memory usage plot
    plt.figure(figsize=(10, 5))
    plt.plot(cpu_usages_np, label='CPU Usage (%)', color='b')
    plt.plot(mem_usages_np, label='Memory Usage (%)', color='g')
    plt.axhline(y=mean_cpu, color='b', linestyle='--', label='Mean CPU')
    plt.axhline(y=mean_mem, color='g', linestyle='--', label='Mean Memory')
    plt.xlabel("Frame")
    plt.ylabel("Usage (%)")
    plt.legend()
    plt.grid(True)
    plt.title("CPU and Memory Usage")
    plt.show()

    # Close all plots if the user pressed "q"
    if cv2.waitKey(1) & 0xFF == ord('q'):
        plt.close('all')
        cv2.destroyAllWindows()
        exit()

    return
