function [DJ,...
          L,Px,Py,...
          L_x,Px_x,Py_x,...
          L_y,Px_y,Py_y,wi,n]=quad4_iis_sf_v2(x,y,e,n,xa,ya)
%-----------------------------------------------------------------------------------------
% [DJ,...
%  L,Px,Py,...
%  L_x,Px_x,Py_x,...
%  L_y,Px_y,Py_y,wi]=quad4_iis_sf(x,y,e,n,xa,ya);
%-----------------------------------------------------------------------------------------
%                              x=vettore ascisse globali dei vertici del quadrilatero;         
%                              y=vettore ordinate globali dei vertici del quadrilatero;         
%                              e=parametro da porre a:
%                                # 1 per n punti "equispaziati" (vedere n);
%                                # 2 per n nodi di Gauss-Legendre lungo ciascun lato del
%                                    quadrilatero (e quindi con nxn nodi in totale);
%                                # 3 per n nodi di Gauss-Legendre lungo ciascun lato del
%                                    quadrilatero (e quindi con nxn nodi in totale)+il centroide;
%                              n=numero di punti di valutazione delle funzioni di forma:
%                                # se e==1 si hanno i seguenti casi:
%                                  # n=1 per un punto a scelta all'interno dell'elemento
%                                        (vedere input xa e ya);
%                                  # n>1 per nxn punti equispaziati;
%                                # se e=2 gli n punti richiesti (per lato) sono
%                                         necessariamente i nodi di Gauss-Legendre;
%                              xa, ya=coordinate naturali adimensionali comprese in [-1,1]
%                                     del punto di valutazione nel caso e=1 e n=1;
%-----------------------------------------------------------------------------------------
%  DJ=vettore nx1 con il determinante della matrice Jacobiana negli n punti specificati;
%  L=matrice nx4 che ha in ogni colonna gli n valori delle quattro funzioni "lineari"
%    negli n punti specificati;
%  Px=matrice nx4 che ha in ogni colonna gli n valori delle quattro funzioni paraboliche
%     relative alla thetax negli n punti specificati;
%  Py=matrice nx4 che ha in ogni colonna gli n valori delle quattro funzioni paraboliche
%     relative alla thetay negli n punti specificati;
%  L_x, ..., Py_y=matrici nx4 contenenti le derivate (rispetto ad x o ad y) delle funzioni
%                 di cui sopra;
%  wi=vettore nx1 con i pesi per l'integrazione con n punti di Gauss
%-----------------------------------------------------------------------------------------

%-----------------------------------------------------------------------------------------
% CALCOLI PRELIMINARI;
%-----------------------------------------------------------------------------------------

% Coordinate naturali [-1,+1] di valutazione delle funzioni di forma;

switch e
   
   % Punti "equispaziati";
    case 1
        wi=[];
       if n==1
       
          chiv=xa;
          etav=ya;
          
       else
          
          X=linspace(-1,1,n);
          [chiv,etav]=meshgrid(X,X);
          
          chiv=chiv(:);
          etav=etav(:);
          
       end
      
    case 2
   
       % Nodi di Gauss-Legendre secondo un lato;
       
       [Xi,w0]=gaule(n);
       
       % Nodi di Gauss-Legendre sul dominio quadrilatero;
       
       [chiv,etav]=meshgrid(Xi,Xi);
       [w1,w2]=meshgrid(w0,w0);
       wi=(w1.*w2);
       wi=wi(:);
       chiv=chiv(:);
       etav=etav(:);
       n=n*n;

    case 3
       % Nodi di Gauss-Legendre secondo un lato;
       
       [Xi,w0]=gaule(n);
       
       % Nodi di Gauss-Legendre sul dominio quadrilatero;
       
       [chiv,etav]=meshgrid(Xi,Xi);
       [w1,w2]=meshgrid(w0,w0);
       wi=(w1.*w2);
       wi=wi(:);
       chiv=chiv(:);
       etav=etav(:);

       % centoide

       chiv(end+1,1)=0;
       etav(end+1,1)=0;
       n=n*n+1;
end

% Coordinate naturali dei 4 vertici;

chi_i=[-1
        1
        1
       -1];

eta_i=[-1
       -1
        1
        1];

%-----------------------------------------------------------------------------------------
% CALCOLO DELLE FUNZIONI DI FORMA E DELLE LORO DERIVATE NEI PUNTI RICHIESTI;
%-----------------------------------------------------------------------------------------
num_el=length(chiv);
% Funzioni "lineari";
L=zeros(num_el,4);
for r=1:4
   
   L(:,r)=1/4*(1+chi_i(r)*chiv).*(1+eta_i(r)*etav);
   
end
   
% Funzioni paraboliche;

P12=1/2*(1+chiv).*(1-chiv).*(1-etav);
P23=1/2*(1+etav).*(1-etav).*(1+chiv);
P34=1/2*(1+chiv).*(1-chiv).*(1+etav);
P41=1/2*(1+etav).*(1-etav).*(1-chiv);

Px(:,1)=1/8*((x(1)-x(4))*P41+(x(1)-x(2))*P12);
Px(:,2)=1/8*((x(2)-x(1))*P12+(x(2)-x(3))*P23);
Px(:,3)=1/8*((x(3)-x(2))*P23+(x(3)-x(4))*P34);
Px(:,4)=1/8*((x(4)-x(3))*P34+(x(4)-x(1))*P41);

Py(:,1)=1/8*((y(1)-y(4))*P41+(y(1)-y(2))*P12);
Py(:,2)=1/8*((y(2)-y(1))*P12+(y(2)-y(3))*P23);
Py(:,3)=1/8*((y(3)-y(2))*P23+(y(3)-y(4))*P34);
Py(:,4)=1/8*((y(4)-y(3))*P34+(y(4)-y(1))*P41);

% Coefficienti della matrice Jacobiana (e della sua inversa);

J11=1/4*((x(2)-x(1))*(1-etav)+(x(3)-x(4))*(1+etav));
J12=1/4*((y(2)-y(1))*(1-etav)+(y(3)-y(4))*(1+etav));
J21=1/4*((x(4)-x(1))*(1-chiv)+(x(3)-x(2))*(1+chiv));
J22=1/4*((y(4)-y(1))*(1-chiv)+(y(3)-y(2))*(1+chiv));

DJ=J11.*J22-J12.*J21;

IJ11= J22./DJ;
IJ12=-J12./DJ;
IJ21=-J21./DJ;
IJ22= J11./DJ;

% Derivate rispetto a chi e ad eta;
L_chi=zeros(num_el,4);
L_eta=zeros(num_el,4);

for r=1:4
   
   L_chi(:,r)=1/4*chi_i(r).*(1+eta_i(r)*etav);
   L_eta(:,r)=1/4*eta_i(r).*(1+chi_i(r)*chiv);
   
end

P12_chi=-chiv.*(1-etav);
P23_chi= 1/2*(1+etav).*(1-etav);
P34_chi=-chiv.*(1+etav);
P41_chi=-1/2*(1+etav).*(1-etav);

P12_eta=-1/2*(1+chiv).*(1-chiv);
P23_eta=-etav.*(1+chiv);
P34_eta= 1/2*(1+chiv).*(1-chiv);
P41_eta=-etav.*(1-chiv);

Px_chi(:,1)=1/8*((x(1)-x(4))*P41_chi+(x(1)-x(2))*P12_chi);
Px_chi(:,2)=1/8*((x(2)-x(1))*P12_chi+(x(2)-x(3))*P23_chi);
Px_chi(:,3)=1/8*((x(3)-x(2))*P23_chi+(x(3)-x(4))*P34_chi);
Px_chi(:,4)=1/8*((x(4)-x(3))*P34_chi+(x(4)-x(1))*P41_chi);

Px_eta(:,1)=1/8*((x(1)-x(4))*P41_eta+(x(1)-x(2))*P12_eta);
Px_eta(:,2)=1/8*((x(2)-x(1))*P12_eta+(x(2)-x(3))*P23_eta);
Px_eta(:,3)=1/8*((x(3)-x(2))*P23_eta+(x(3)-x(4))*P34_eta);
Px_eta(:,4)=1/8*((x(4)-x(3))*P34_eta+(x(4)-x(1))*P41_eta);

Py_chi(:,1)=1/8*((y(1)-y(4))*P41_chi+(y(1)-y(2))*P12_chi);
Py_chi(:,2)=1/8*((y(2)-y(1))*P12_chi+(y(2)-y(3))*P23_chi);
Py_chi(:,3)=1/8*((y(3)-y(2))*P23_chi+(y(3)-y(4))*P34_chi);
Py_chi(:,4)=1/8*((y(4)-y(3))*P34_chi+(y(4)-y(1))*P41_chi);

Py_eta(:,1)=1/8*((y(1)-y(4))*P41_eta+(y(1)-y(2))*P12_eta);
Py_eta(:,2)=1/8*((y(2)-y(1))*P12_eta+(y(2)-y(3))*P23_eta);
Py_eta(:,3)=1/8*((y(3)-y(2))*P23_eta+(y(3)-y(4))*P34_eta);
Py_eta(:,4)=1/8*((y(4)-y(3))*P34_eta+(y(4)-y(1))*P41_eta);

% Derivate rispetto ad x e ad y;
L_x=zeros(num_el,4);
L_y=zeros(num_el,4);
Px_x=zeros(num_el,4);
Px_y=zeros(num_el,4);
Py_x=zeros(num_el,4);
Py_y=zeros(num_el,4);
for r=1:4
   
   L_x(:,r)=IJ11.*L_chi(:,r)+IJ12.*L_eta(:,r);
   L_y(:,r)=IJ21.*L_chi(:,r)+IJ22.*L_eta(:,r);
   
   Px_x(:,r)=IJ11.*Px_chi(:,r)+IJ12.*Px_eta(:,r);
   Px_y(:,r)=IJ21.*Px_chi(:,r)+IJ22.*Px_eta(:,r);
   
   Py_x(:,r)=IJ11.*Py_chi(:,r)+IJ12.*Py_eta(:,r);
   Py_y(:,r)=IJ21.*Py_chi(:,r)+IJ22.*Py_eta(:,r);
   
end
end
