clear all
close all
clc

% Percorso delle cartelle dei pazienti
baseDir = 'C:\Users\Dell\Desktop\Tesi\Dataset_TOT';

% Mappa dei pazienti con il numero di slice interpolati desiderato per CT e PET
patientInterpolationMap = containers.Map({11, 24, 40, 43, 44, 65, 71, 74}, {5, 5, 5, 5, 5, 5, 5, 3}); % Per CT
patientInterpolationMapPet = containers.Map({11, 24, 40, 43, 44, 65, 71, 74}, {4.25, 4.25, 4.25, 4.25, 4.25, 4.25, 4.25, 4}); % Per PET

%% Cartella CT
ctDir = fullfile(baseDir, 'CT');
ctFiles = dir(fullfile(ctDir, '*.dcm'));

% Cartella di output per CT interpolati
ctOutDir = fullfile(baseDir, 'CT_Interpolated');
if ~exist(ctOutDir, 'dir')
    mkdir(ctOutDir);
end

% Loop attraverso i file DICOM nella cartella CT
for i = 1:length(ctFiles)-1
    ctFilename1 = ctFiles(i).name;
    ctFilename2 = ctFiles(i+1).name;

    % Estrai l'ID del paziente e il numero della slice
    [~, ctFilenameOnly1, ~] = fileparts(ctFilename1);
    parts1 = split(ctFilenameOnly1, '_');
    patientID = str2double(parts1{1});
    sliceNumber1 = str2double(parts1{2});

    [~, ctFilenameOnly2, ~] = fileparts(ctFilename2);
    parts2 = split(ctFilenameOnly2, '_');
    sliceNumber2 = str2double(parts2{2});

    % Normalizza l'ID del paziente
    normalizedPatientID = sprintf('%02d', patientID);

    % Procedi solo se il paziente ha un ID >= 70
    if patientID < 70
        continue;
    end

    % Verifica il numero di slice interpolate desiderato per questo paziente
    if isKey(patientInterpolationMap, patientID)
        numInterpolatedSlices = patientInterpolationMap(patientID);
    else
        numInterpolatedSlices = 3; % Valore predefinito per CT
    end

    % Leggi i file DICOM e le informazioni
    ctFullPath1 = fullfile(ctDir, ctFilename1);
    ctInfo1 = dicominfo(ctFullPath1);
    ctData1 = double(dicomread(ctFullPath1));

    ctFullPath2 = fullfile(ctDir, ctFilename2);
    ctInfo2 = dicominfo(ctFullPath2);
    ctData2 = double(dicomread(ctFullPath2));

    % Controlla se le dimensioni delle immagini sono uguali
    if ~isequal(size(ctData1), size(ctData2))
        continue; % Salta questa coppia se le dimensioni sono diverse
    end

    % Salva il file DICOM originale nella cartella di output CT_Interpolated
    originalCTFilename1 = sprintf('%s_%03d.dcm', normalizedPatientID, sliceNumber1);
    originalCTPath1 = fullfile(ctOutDir, originalCTFilename1);
    dicomwrite(uint16(ctData1), originalCTPath1, ctInfo1, 'CreateMode', 'Copy');

    % Salva anche la seconda fetta originale
    originalCTFilename2 = sprintf('%s_%03d.dcm', normalizedPatientID, sliceNumber2);
    originalCTPath2 = fullfile(ctOutDir, originalCTFilename2);
    dicomwrite(uint16(ctData2), originalCTPath2, ctInfo2, 'CreateMode', 'Copy');

    % Esegui l'interpolazione delle fette
    for j = 1:numInterpolatedSlices
        alpha = j / (numInterpolatedSlices + 1);  % Fattore di interpolazione

        [X, Y] = meshgrid(1:size(ctData1, 2), 1:size(ctData1, 1)); % Griglia per l'interpolazione usando meshgrid

        % Interpolazione con spline cubica tra le due fette
        Z_interp = (1 - alpha) * interp2(X, Y, ctData1, X, Y, 'spline') + ...
                   alpha * interp2(X, Y, ctData2, X, Y, 'spline');

        % Nome progressivo per le fette interpolate
        interpolatedSliceNumber = sliceNumber1 + j;
        interpolatedCTFilename = sprintf('%s_%03d.dcm', normalizedPatientID, interpolatedSliceNumber);
        interpolatedCTPath = fullfile(ctOutDir, interpolatedCTFilename);

        % Aggiorna le informazioni DICOM con la nuova spaziatura e altri tag necessari
        interpolatedCTInfo = ctInfo1;
        interpolatedCTInfo.SliceLocation = ctInfo1.SliceLocation + (ctInfo2.SliceLocation - ctInfo1.SliceLocation) * alpha;
        interpolatedCTInfo.InstanceNumber = ctInfo1.InstanceNumber + j;
        interpolatedCTInfo.SliceThickness = 1;

        % Copia i tag DICOM specificati
        interpolatedCTInfo.RescaleIntercept = ctInfo1.RescaleIntercept;
        interpolatedCTInfo.RescaleSlope = ctInfo1.RescaleSlope;

        % Scrivi l'immagine interpolata come file DICOM
        dicomwrite(uint16(Z_interp), interpolatedCTPath, interpolatedCTInfo, 'CreateMode', 'Copy');
    end
end

%% Cartella PET
petDir = fullfile(baseDir, 'PET');
petFiles = dir(fullfile(petDir, '*.dcm'));

% Cartella di output per PET interpolati
petOutDir = fullfile(baseDir, 'PET_Interpolated');
if ~exist(petOutDir, 'dir')
    mkdir(petOutDir);
end

% Loop attraverso i file DICOM nella cartella PET
for i = 1:length(petFiles)-1
    petFilename1 = petFiles(i).name;
    petFilename2 = petFiles(i+1).name;

    % Estrai l'ID del paziente e il numero della slice
    [~, petFilenameOnly1, ~] = fileparts(petFilename1);
    parts1 = split(petFilenameOnly1, '_');
    patientID = str2double(parts1{1});
    sliceNumber1 = str2double(parts1{2});

    [~, petFilenameOnly2, ~] = fileparts(petFilename2);
    parts2 = split(petFilenameOnly2, '_');
    sliceNumber2 = str2double(parts2{2});

    % Normalizza l'ID del paziente
    normalizedPatientID = sprintf('%02d', patientID);

    % Procedi solo se il paziente ha un ID >= 70
    if patientID < 70
        continue;
    end

    % Verifica il numero di slice interpolate desiderato per questo paziente
    if isKey(patientInterpolationMapPet, patientID)
        numInterpolatedSlices = patientInterpolationMapPet(patientID);
    else
        numInterpolatedSlices = 4; % Valore predefinito per PET
    end

    % Leggi i file DICOM e le informazioni
    petFullPath1 = fullfile(petDir, petFilename1);
    petInfo1 = dicominfo(petFullPath1);
    petData1 = double(dicomread(petFullPath1));

    petFullPath2 = fullfile(petDir, petFilename2);
    petInfo2 = dicominfo(petFullPath2);
    petData2 = double(dicomread(petFullPath2));

    % Controlla se le dimensioni delle immagini sono uguali
    if ~isequal(size(petData1), size(petData2))
        continue; % Salta questa coppia se le dimensioni sono diverse
    end

    % Salva il file DICOM originale nella cartella di output PET_Interpolated
    originalPETFilename1 = sprintf('%s_%03d.dcm', normalizedPatientID, sliceNumber1);
    originalPETPath1 = fullfile(petOutDir, originalPETFilename1);
    dicomwrite(uint16(petData1), originalPETPath1, petInfo1, 'CreateMode', 'Copy');

    % Salva anche la seconda fetta originale
    originalPETFilename2 = sprintf('%s_%03d.dcm', normalizedPatientID, sliceNumber2);
    originalPETPath2 = fullfile(petOutDir, originalPETFilename2);
    dicomwrite(uint16(petData2), originalPETPath2, petInfo2, 'CreateMode', 'Copy');

    % Esegui l'interpolazione delle fette
    for j = 1:numInterpolatedSlices
        alpha = j / (numInterpolatedSlices + 1);  % Fattore di interpolazione

        [X, Y] = meshgrid(1:size(petData1, 2), 1:size(petData1, 1)); % Griglia per l'interpolazione usando meshgrid

        % Interpolazione con spline cubica tra le due fette
        Z_interp = (1 - alpha) * interp2(X, Y, petData1, X, Y, 'spline') + ...
                   alpha * interp2(X, Y, petData2, X, Y, 'spline');

        % Nome progressivo per le fette interpolate
        interpolatedSliceNumber = sliceNumber1 + j;
        interpolatedPETFilename = sprintf('%s_%03d.dcm', normalizedPatientID, interpolatedSliceNumber);
        interpolatedPETPath = fullfile(petOutDir, interpolatedPETFilename);

        % Aggiorna le informazioni DICOM con la nuova spaziatura e altri tag necessari
        interpolatedPETInfo = petInfo1;
        interpolatedPETInfo.SliceLocation = petInfo1.SliceLocation + (petInfo2.SliceLocation - petInfo1.SliceLocation) * alpha;
        interpolatedPETInfo.InstanceNumber = petInfo1.InstanceNumber + j;
        interpolatedPETInfo.SliceThickness = 1;

        % Copia i tag DICOM specificati
        interpolatedPETInfo.RescaleIntercept = petInfo1.RescaleIntercept;
        interpolatedPETInfo.RescaleSlope = petInfo1.RescaleSlope;

        % Scrivi l'immagine interpolata come file DICOM
        dicomwrite(uint16(Z_interp), interpolatedPETPath, interpolatedPETInfo, 'CreateMode', 'Copy');
    end
end

%% Cartella MASK
maskDir = fullfile(baseDir, 'MASK');
maskFiles = dir(fullfile(maskDir, '*.png'));

% Cartella di output per le maschere interpolate
maskOutDir = fullfile(baseDir, 'MASK_Interpolated');
if ~exist(maskOutDir, 'dir')
    mkdir(maskOutDir);
end

% Loop attraverso i file PNG nella cartella MASK
for i = 1:length(maskFiles)-1
    maskFilename1 = maskFiles(i).name;
    maskFilename2 = maskFiles(i+1).name;

    % Estrai l'ID del paziente e il numero della slice
    [~, maskFilenameOnly1, ~] = fileparts(maskFilename1);
    parts1 = split(maskFilenameOnly1, '_');
    patientID = str2double(parts1{1});
    sliceNumber1 = str2double(parts1{2});

    [~, maskFilenameOnly2, ~] = fileparts(maskFilename2);
    parts2 = split(maskFilenameOnly2, '_');
    sliceNumber2 = str2double(parts2{2});

    % Normalizza l'ID del paziente
    normalizedPatientID = sprintf('%02d', patientID);

    % % Procedi solo se il paziente ha un ID >= 70
    % if patientID < 70
    %     continue;
    % end

    % Verifica il numero di slice interpolate desiderato per questo paziente
    if isKey(patientInterpolationMap, patientID)
        numInterpolatedSlices = patientInterpolationMap(patientID);
    else
        numInterpolatedSlices = 3; % Valore predefinito per le maschere
    end

    % Leggi i file PNG
    maskFullPath1 = fullfile(maskDir, maskFilename1);
    maskData1 = imread(maskFullPath1);

    maskFullPath2 = fullfile(maskDir, maskFilename2);
    maskData2 = imread(maskFullPath2);

    % Controlla se le dimensioni delle immagini sono uguali
    if ~isequal(size(maskData1), size(maskData2))
        continue; % Salta questa coppia se le dimensioni sono diverse
    end

    % Salva il file PNG originale nella cartella di output MASK_Interpolated
    originalMaskFilename1 = sprintf('%s_%03d.png', normalizedPatientID, sliceNumber1);
    originalMaskPath1 = fullfile(maskOutDir, originalMaskFilename1);
    imwrite(maskData1, originalMaskPath1);

    % Salva anche la seconda fetta originale
    originalMaskFilename2 = sprintf('%s_%03d.png', normalizedPatientID, sliceNumber2);
    originalMaskPath2 = fullfile(maskOutDir, originalMaskFilename2);
    imwrite(maskData2, originalMaskPath2);

    % Esegui l'interpolazione delle maschere
    for j = 1:numInterpolatedSlices
        alpha = j / (numInterpolatedSlices + 1);  % Fattore di interpolazione

        % Interpolazione lineare tra le due maschere
        maskInterpData = (1 - alpha) * double(maskData1) + alpha * double(maskData2);
        maskInterpData = uint8(maskInterpData);

        % Nome progressivo per le maschere interpolate
        interpolatedSliceNumber = sliceNumber1 + j;
        interpolatedMaskFilename = sprintf('%s_%03d.png', normalizedPatientID, interpolatedSliceNumber);
        interpolatedMaskPath = fullfile(maskOutDir, interpolatedMaskFilename);

        % Scrivi l'immagine maschera interpolata come file PNG
        imwrite(maskInterpData, interpolatedMaskPath);
    end
end