% Step 1: Load data 
close all;
clear;
load('MMF_transferFunctions_OM4_400m.mat');
load('DataBER_106.25G_OM4.mat');
load('DataLmax_106.25G_OM4.mat');

% Step 2: Extract relevant variables
Lambdas = Lambdas;
Laszers = Laser;
frequencies = freq;
Transfer_fct_MMF_OM4 = Hf_MMF_OM4;
EMB_OM4 = EMB_OM4;
Group_delay_OM4 = taug_OM4;
numWavelengths = size(Hf_MMF_OM4, 1);
numLasers = size(Hf_MMF_OM4, 2);
numFibers = size(Hf_MMF_OM4, 4);

%Extract transfer functions for the chosen wavelength and laser


laser_index=1;
wavelength_index = 1;
fiber_index = 1;
% Define lengths
L = 30;
Lref = 400;
K = Lref / L;

% Set reference distance and reference transfer function at 400m for a specific laser and wavelength
Lref = 400; % Reference distance in meters
reference_TF = abs(squeeze(Hf_MMF_OM4(wavelength_index, laser_index, :, 1))); % Absolute value for magnitude
reference_TF_dB = 10*log10(reference_TF/max(reference_TF)); % Convert to dB normalized to max

% Initialize new arrays for the scaled transfer functions
new_Transfer_fct = zeros(size(Transfer_fct_MMF_OM4));
New_HF_30dB = zeros(size(Transfer_fct_MMF_OM4));

% Loop over all wavelengths, lasers, and fibers
for lambda_idx = 1:size(Transfer_fct_MMF_OM4, 1)
    for laser_idx = 1:size(Transfer_fct_MMF_OM4, 2)
        for fiber_idx = 1:size(Transfer_fct_MMF_OM4, 4)
            % Original transfer function for this combination
            TF = abs(squeeze(Transfer_fct_MMF_OM4(lambda_idx, laser_idx, :, fiber_idx)));

            % Scale the frequencies
            freq_scaled = frequencies * K;

            % Interpolate the scaled transfer function
            TF_scaled_30 = interp1(freq_scaled, TF, frequencies, 'linear', 'extrap');
            TF_scaled_30_dB = 10 * log10(TF_scaled_30 / max(reference_TF));

            % Store the scaled transfer function
            new_Transfer_fct_30(lambda_idx, laser_idx, :, fiber_idx) = TF_scaled_30;
            New_HF_30dB(lambda_idx, laser_idx, :, fiber_idx) = TF_scaled_30_dB;
        end
    end
end




numWavelengths = size(Hf_MMF_OM4, 1);
numLasers = size(Hf_MMF_OM4, 2);
numFibers = size(Hf_MMF_OM4, 4);

% Initialize an array to store the equivalent bandwidths
equivalent_bandwidths_all = zeros(numWavelengths, numLasers, numFibers);
bandwidths_3dB_all = zeros(numWavelengths, numLasers, numFibers);
bandwidths_5dB_all = zeros(numWavelengths, numLasers, numFibers);
bandwidths_10dB_all = zeros(numWavelengths, numLasers, numFibers);
% Iterate over all wavelengths, lasers, and fibers
for wavelength_index = 1:numWavelengths
    for laser_index = 1:numLasers
        for fiber_index = 1:numFibers
            % Extract the transfer function for this specific combination
            TF_selected = squeeze(new_Transfer_fct_30(wavelength_index, laser_index, :, fiber_index));

            % Calculate the equivalent bandwidth for the current combination
            equivalent_bandwidths_all(wavelength_index, laser_index, fiber_index) = calculateEquivalentBandwidth(TF_selected, freq);
            bandwidths_3dB_all(wavelength_index, laser_index, fiber_index) = calculateMinus3dBBandwidth(TF_selected, freq);
            bandwidths_5dB_all(wavelength_index, laser_index, fiber_index) = calculateMinus5dBBandwidth(TF_selected, freq);
            bandwidths_10dB_all(wavelength_index, laser_index, fiber_index) = calculateMinus10dBBandwidth(TF_selected, freq);
        end
    end
end
% Step 4: Reshape data into vectors
vector_equiv = reshape(equivalent_bandwidths_all, [], 1);
vector_3dB = reshape(bandwidths_3dB_all, [], 1);
vector_5dB = reshape(bandwidths_5dB_all, [], 1);
vector_10dB = reshape(bandwidths_10dB_all, [], 1);

Lmax_all = squeeze(Lmax_OM4(:, :, 1, :));
Lmax_all_vector = reshape(Lmax_all, [], 1);

% Step 5: Clean data (remove NaNs and Infs)
validIndex = ~(isnan(Lmax_all_vector) | isinf(Lmax_all_vector));

Lmax_all_vector_cleaned = Lmax_all_vector(validIndex);
vector_3dB_cleaned = vector_3dB(validIndex);
vector_5dB_cleaned = vector_5dB(validIndex);
vector_10dB_cleaned = vector_10dB(validIndex);
vector_equiv_cleaned = vector_equiv(validIndex);

% Step 6: Normalize the data
X = [vector_3dB_cleaned, vector_5dB_cleaned, vector_10dB_cleaned, vector_equiv_cleaned];
X = (X - min(X)) ./ (max(X) - min(X));  % Min-max normalization

Y = Lmax_all_vector_cleaned;

% Step 7: Split the data into training (80%) and testing (20%) sets
cv = cvpartition(size(X,1),'HoldOut',0.2);
XTrain = X(training(cv),:);
YTrain = Y(training(cv),:);
XTest = X(test(cv),:);
YTest = Y(test(cv),:);

% Step 8: Define a deeper neural network
net = fitnet([30 20 15 10]);  % Increased complexity with 4 layers

% Configure the training function (Levenberg-Marquardt backpropagation)
net.trainFcn = 'trainlm';

% Adjust the training parameters
net.trainParam.lr = 0.01;  % Learning rate
net.trainParam.epochs = 1000;  % Increase number of epochs
net.performParam.regularization = 0.1;  % L2 regularization

% Train the neural network
[net, tr] = train(net, XTrain', YTrain');

% Step 9: Evaluate the model on the test set
YPred = net(XTest');

% Calculate accuracy metrics
mseError = mse(net, YTest', YPred);
fprintf('Mean Squared Error of the prediction: %f\n', mseError);

maeError = mean(abs(YPred - YTest'));
fprintf('Mean Absolute Error: %f\n', maeError);

rmseError = sqrt(mseError);
fprintf('Root Mean Squared Error: %f\n', rmseError);

mapeError = mean(abs((YPred - YTest') ./ YTest')) * 100;
fprintf('Mean Absolute Percentage Error: %f%%\n', mapeError);

SS_res = sum((YTest' - YPred).^2);
SS_tot = sum((YTest' - mean(YTest')).^2);
R2 = 1 - (SS_res / SS_tot);
fprintf('R² Score: %f\n', R2);

% Calculate the differences and find the maximum difference
differences = abs(YPred - YTest');
maxDifference = max(differences);
meanDifference = mean(differences);
stdDifference = std(differences);

fprintf('Maximum Difference between Predicted and Actual Lmax: %f\n', maxDifference);
fprintf('Mean Difference: %f\n', meanDifference);
fprintf('Standard Deviation of Differences: %f\n', stdDifference);

% Step 10: Visualize the results
figure;
plot(YTest', YTest' - YPred, 'o');
xlabel('True Lmax');
ylabel('Residuals (True - Predicted)');
title('Residual Plot');
grid on;

figure;
histogram(differences, 20); % 20 bins for the histogram
xlabel('Absolute Difference (|True - Predicted|)');
ylabel('Frequency');
title('Histogram of Absolute Differences');

% Additional Visualization: Regression plot
figure;
plotregression(YTest', YPred);

% Step 11: Hyperparameter Tuning Example (Bayesian Optimization)
optimVars = [
    optimizableVariable('LayerSize1', [10, 50], 'Type', 'integer');
    optimizableVariable('LayerSize2', [10, 50], 'Type', 'integer');
    optimizableVariable('LearningRate', [1e-4, 1e-1], 'Transform', 'log');
];

ObjFcn = @(x) trainAndValidateNetwork([x.LayerSize1 x.LayerSize2], x.LearningRate, XTrain', YTrain', XTest', YTest');
results = bayesopt(ObjFcn, optimVars, 'MaxObjectiveEvaluations', 30);

% Extract best hyperparameters
bestLayerSize1 = results.XAtMinObjective.LayerSize1;
bestLayerSize2 = results.XAtMinObjective.LayerSize2;
bestLearningRate = results.XAtMinObjective.LearningRate;

% Re-train with best hyperparameters
net = fitnet([bestLayerSize1 bestLayerSize2]);
net.trainParam.lr = bestLearningRate;
net.performParam.regularization = 0.1;
[net, tr] = train(net, XTrain', YTrain');

% Evaluate the optimized model
YPredOptimized = net(XTest');

% Display optimized model performance
mseOptimized = mse(net, YTest', YPredOptimized);
fprintf('Optimized Mean Squared Error of the prediction: %f\n', mseOptimized);

% Step 12: Ensemble Learning Example
numModels = 5;
nets = cell(1, numModels);
preds = zeros(numModels, length(YTest));

for i = 1:numModels
    nets{i} = fitnet([30 20 15 10]);
    nets{i}.trainParam.lr = 0.01;
    nets{i}.performParam.regularization = 0.1;
    [nets{i}, tr] = train(nets{i}, XTrain', YTrain');
    preds(i, :) = nets{i}(XTest');
end

% Average predictions from all models
ensemblePred = mean(preds, 1);
mseEnsemble = mean((ensemblePred - YTest').^2);
fprintf('Ensemble MSE: %f\n', mseEnsemble);

% Define the trainAndValidateNetwork function here
function [valError] = trainAndValidateNetwork(layerSizes, learningRate, XTrain, YTrain, XVal, YVal)
    % Define the neural network with the given layer sizes
    net = fitnet(layerSizes);
    
    % Set the learning rate and training function
    net.trainFcn = 'trainlm';  % Levenberg-Marquardt
    net.trainParam.lr = learningRate;  % Set the learning rate
    net.trainParam.epochs = 1000;  % Maximum number of epochs
    net.performParam.regularization = 0.1;  % L2 regularization
    
    % Train the network using the training data
    [net, tr] = train(net, XTrain, YTrain);
    
    % Evaluate the network on the validation data
    YValPred = net(XVal);
    valError = mse(YVal, YValPred);  % Return Mean Squared Error as the objective metric
end

% Function to calculate the -3dB bandwidth
function bandwidth_3dB = calculateMinus3dBBandwidth(TF, freq)
    % Convert the transfer function to dB
    TF_dB = 10 * log10(abs(TF));

    % Find the maximum magnitude in dB
    max_TF_dB = max(TF_dB);

    % Find the frequencies where the magnitude is greater than -3dB from the peak
    within_3dB_indices = find(TF_dB >= max_TF_dB - 3);

    % Find the -3dB bandwidth, which is the difference between the first and last frequencies within the -3dB range
    if ~isempty(within_3dB_indices)
        bandwidth_3dB = freq(within_3dB_indices(end)) - freq(within_3dB_indices(1));
    else
        bandwidth_3dB = NaN; % Return NaN if there's no bandwidth within -3dB
    end
end
% Function to calculate the -5dB bandwidth
function bandwidth_5dB = calculateMinus5dBBandwidth(TF, freq)
    % Convert the transfer function to dB
    TF_dB = 10 * log10(abs(TF));

    % Find the maximum magnitude in dB
    max_TF_dB = max(TF_dB);

    % Find the frequencies where the magnitude is greater than -5dB from the peak
    within_5dB_indices = find(TF_dB >= max_TF_dB - 5);

    % Find the -5dB bandwidth, which is the difference between the first and last frequencies within the -5dB range
    if ~isempty(within_5dB_indices)
        bandwidth_5dB = freq(within_5dB_indices(end)) - freq(within_5dB_indices(1));
    else
        bandwidth_5dB = NaN; % Return NaN if there's no bandwidth within -5dB
    end
end

% Function to calculate the -10dB bandwidth
function bandwidth_10dB = calculateMinus10dBBandwidth(TF, freq)
    % Convert the transfer function to dB
    TF_dB = 10 * log10(abs(TF));

    % Find the maximum magnitude in dB
    max_TF_dB = max(TF_dB);

    % Find the frequencies where the magnitude is greater than -10dB from the peak
    within_10dB_indices = find(TF_dB >= max_TF_dB - 10);

    % Find the -10dB bandwidth, which is the difference between the first and last frequencies within the -10dB range
    if ~isempty(within_10dB_indices)
        bandwidth_10dB = freq(within_10dB_indices(end)) - freq(within_10dB_indices(1));
    else
        bandwidth_10dB = NaN; % Return NaN if there's no bandwidth within -10dB
    end
end




% Function to calculate the equivalent bandwidth
function equivalent_bandwidth = calculateEquivalentBandwidth(TF, freq)
    % Square the magnitude of the transfer function
    TF_squared = abs(TF).^2;

    % Integrate over frequency to find total power
    total_power = trapz(freq, TF_squared);

    % Calculate the equivalent bandwidth based on total power
    equivalent_bandwidth = total_power / max(TF_squared);
end


