close all ;
% Step 1: Load data 
load('MMF_transferFunctions_OM4_400m.mat');
load('DataBER_106.25G_OM4.mat');
load('DataLmax_106.25G_OM4.mat');
% Step 2: Extract relevant variables

Lambdas = Lambdas;
Laszers = Laser;
frequencies = freq;
Transfer_fct_MMF_OM4 = Hf_MMF_OM4;
EMB_OM4 = EMB_OM4;
Group_delay_OM4 = taug_OM4;

All_distances=Lmax_all_vector_cleaned ;
% Define reference length
Lref = 400;

% % Extract the relevant subset of Lmax_OM4
% Lmax_selected = squeeze(Lmax_OM4(:, :, 1, :)); % [4 x 8 x 3766]
% % Replace Inf values with NaN
% Lmax_selected(isinf(Lmax_selected)) = NaN;

% Extract the relevant subset of Lmax_OM4
Lmax_selected = squeeze(Lmax_OM4(:, :, 1, :)); % [4 x 8 x 3766]

% Flatten the 3D matrix to a 1D vector
Lmax_selected_flat = Lmax_selected(:);  % 1D vector

% Identify Inf values and replace them with NaN
Lmax_selected_flat(isinf(Lmax_selected_flat)) = NaN;
% % Identify NaN values
% nan_indices = isnan(Lmax_selected_flat);
% 
% % Remove NaN values
% vector_cleaned = Lmax_selected_flat(~nan_indices);
% % Alternatively, to remove Inf values directly:
% % Lmax_selected_flat = Lmax_selected_flat(~isinf(Lmax_selected_flat));
% 
% % Determine the original size
% original_size = size(Lmax_selected);
% num_elements = numel(Lmax_selected_flat);
% num_slices = num_elements / (original_size(1) * original_size(2));
% 
% % Ensure num_slices is an integer
% if mod(num_slices, 1) ~= 0
%     error('The number of slices is not an integer. Check the data and cleaning logic.');
% end
% 
% % Reshape the cleaned 1D vector back to 3D matrix
% Lmax_selected_cleaned = reshape(vector_cleaned, original_size(1), original_size(2), num_slices);
% 
% % Verify the cleaning
% fprintf('Number of elements with Inf values after removal: %d\n', sum(isinf(Lmax_selected_cleaned(:))));
% fprintf('Number of elements with NaN values after removal: %d\n', sum(isnan(Lmax_selected_cleaned(:))));


% Initialize the vector to store equivalent bandwidths
numWavelengths = size(Lmax_selected, 1);
numLasers = size(Lmax_selected, 2);
numFibers = size(Lmax_selected, 3);
equivalent_bandwidths = zeros(numWavelengths, numLasers, numFibers);
B3db=zeros(numWavelengths, numLasers, numFibers);
B5db=zeros(numWavelengths, numLasers, numFibers);
B10db=zeros(numWavelengths, numLasers, numFibers);
% Iterate over all wavelengths, lasers, and fibers to calculate equivalent bandwidths
for wavelength_index = 1:numWavelengths
    for laser_index = 1:numLasers
        for fiber_index = 1:numFibers
            % Get the current Lmax
            L = Lmax_selected_cleaned(wavelength_index, laser_index, fiber_index);

             % Skip this iteration if L is NaN
            if isnan(L)
                continue; % Skip if L is NaN
            end
            
            % Calculate the scaling factor K
            K = Lref / L;

            % Scale the frequency axis
            freq_scaled = frequencies * K;

            % Ensure all elements in freq_scaled are finite
            if any(~isfinite(freq_scaled))
                continue; % Skip this iteration if freq_scaled has non-finite values
            end

            % Extract and scale the transfer function
            TF = abs(squeeze(Hf_MMF_OM4(wavelength_index, laser_index, :, fiber_index)));
            TF_scaled = interp1(freq_scaled, TF, frequencies, "linear", 0);

            % Calculate the equivalent bandwidth for the current combination
            equivalent_bandwidths(wavelength_index, laser_index, fiber_index) = calculateEquivalentBandwidth(TF_scaled, frequencies);
            B3db(wavelength_index, laser_index, fiber_index) = calculateMinus3dBBandwidth(TF_scaled, frequencies);
            B5db(wavelength_index, laser_index, fiber_index) = calculateMinus5dBBandwidth(TF_scaled, frequencies);
            B10db(wavelength_index, laser_index, fiber_index) = calculateMinus10dBBandwidth(TF_scaled, frequencies);
        end
    end
end


% Reshape the 3D arrays into 1D vectors
vector_equiv = reshape(equivalent_bandwidths, [], 1);
vector_3dB = reshape(B3db, [], 1);
vector_5dB = reshape(B5db, [], 1);
vector_10dB = reshape(B10db, [], 1);


% Create a logical index for non-NaN and non-Inf values
validIndex = ~(isnan(All_distances) | isinf(All_distances));

% Filter out NaN and Inf values from Lmax_all_vector
Lmax_all_vector_cleaned = All_distances(validIndex);

% Apply the same valid index to the other vectors
% Remove zero values
vector_3dB_cleaned = vector_3dB(vector_3dB ~= 0);
vector_5dB_cleaned = vector_5dB(vector_5dB ~= 0);
vector_10dB_cleaned = vector_10dB(vector_10dB ~= 0);
vector_equiv_cleaned = vector_equiv(vector_equiv ~= 0);
%Create a matrix of all bandwidths and Lmax
data_matrix = [vector_3dB_cleaned vector_5dB_cleaned vector_10dB_cleaned vector_equiv_cleaned Lmax_all_vector_cleaned];
correlation_matrix = corr(data_matrix, 'Rows', 'complete');  % 'complete' to ignore NaN values
% Display the correlation matrix
disp('Correlation matrix:');
disp(correlation_matrix);
figure;
labels = {'-3dB Bandwidth', '-5dB Bandwidth', '-10dB Bandwidth', 'Equivalent Bandwidth', 'Lmax'};
heatmap(labels, labels, correlation_matrix);
title('Correlation Heatmap between Bandwidth Features and Lmax at Hf@Lmax');
% Save the plot
saveas(gcf, 'Correlation Heatmap between Bandwidth Features and Lmax at Hf@Lmax.png');

% Perform linear regression
p = polyfit(vector_equiv_cleaned, Lmax_all_vector_cleaned, 1);
regression_line = polyval(p, vector_equiv_cleaned);

% Generate scatter plot with regression line and highlight the highest and lowest points
figure;
scatter(vector_equiv_cleaned, Lmax_all_vector_cleaned, 20, 'filled', 'MarkerFaceAlpha', 0.6, 'MarkerEdgeColor', 'w');
hold on;
plot(vector_equiv_cleaned, regression_line, 'r', 'LineWidth', 2);
xlabel('Equivalent Bandwidth (Beq)', 'FontSize', 14);
ylabel('Maximum Length (Lmax)', 'FontSize', 14);
title('Lmax vs Beq', 'FontSize', 16);
grid on;
legend('Data points', sprintf('Regression line (R^2=%.2f)', corr(vector_equiv_cleaned, Lmax_all_vector_cleaned)^2), 'Location', 'best');
hold off;

% Save the plot
saveas(gcf, 'Lmax_vs_Beq_Hf@Lmax.png');











% Function to calculate the -3dB bandwidth
function bandwidth_3dB = calculateMinus3dBBandwidth(TF, freq)
    % Convert the transfer function to dB
    TF_dB = 10 * log10(abs(TF));

    % Find the maximum magnitude in dB
    max_TF_dB = max(TF_dB);

    % Find the frequencies where the magnitude is greater than -3dB from the peak
    within_3dB_indices = find(TF_dB >= max_TF_dB - 3);

    % Find the -3dB bandwidth, which is the difference between the first and last frequencies within the -3dB range
    if ~isempty(within_3dB_indices)
        bandwidth_3dB = freq(within_3dB_indices(end)) - freq(within_3dB_indices(1));
    else
        bandwidth_3dB = NaN; % Return NaN if there's no bandwidth within -3dB
    end
end
% Function to calculate the -5dB bandwidth
function bandwidth_5dB = calculateMinus5dBBandwidth(TF, freq)
    % Convert the transfer function to dB
    TF_dB = 10 * log10(abs(TF));

    % Find the maximum magnitude in dB
    max_TF_dB = max(TF_dB);

    % Find the frequencies where the magnitude is greater than -5dB from the peak
    within_5dB_indices = find(TF_dB >= max_TF_dB - 5);

    % Find the -5dB bandwidth, which is the difference between the first and last frequencies within the -5dB range
    if ~isempty(within_5dB_indices)
        bandwidth_5dB = freq(within_5dB_indices(end)) - freq(within_5dB_indices(1));
    else
        bandwidth_5dB = NaN; % Return NaN if there's no bandwidth within -5dB
    end
end

% Function to calculate the -10dB bandwidth
function bandwidth_10dB = calculateMinus10dBBandwidth(TF, freq)
    % Convert the transfer function to dB
    TF_dB = 10 * log10(abs(TF));

    % Find the maximum magnitude in dB
    max_TF_dB = max(TF_dB);

    % Find the frequencies where the magnitude is greater than -10dB from the peak
    within_10dB_indices = find(TF_dB >= max_TF_dB - 10);

    % Find the -10dB bandwidth, which is the difference between the first and last frequencies within the -10dB range
    if ~isempty(within_10dB_indices)
        bandwidth_10dB = freq(within_10dB_indices(end)) - freq(within_10dB_indices(1));
    else
        bandwidth_10dB = NaN; % Return NaN if there's no bandwidth within -10dB
    end
end

% Function to calculate the equivalent bandwidth
function equivalent_bandwidth = calculateEquivalentBandwidth(TF, freq)
    % Square the magnitude of the transfer function
    TF_squared = abs(TF);

    % Integrate over frequency to find total power
    total_power = trapz(freq, TF_squared);

    % Calculate the equivalent bandwidth based on total power
    equivalent_bandwidth = total_power / max(TF_squared) /1e9;
end

