
%%  QUADRATIC COEFFICIENTS FOR CONSTANT HEAT RATE - [Ref]: Human Spaceflight, Pranke [book]

clc;
close all;
clear;

% Inizializzazioni
v_high = 7800;                      % Velocità Alta - [m/s]
v_low = 5791;                       % Velocità Bassa - [m/s]
v_mean = (v_high + v_low) * 0.5;    % Velocità Media - [m/s]
v_sat = 7837;                       % Velocità di saturazione - [m/s]

% Parametri del veicolo X38
% W = 11340;                          % Peso - [kg]
% S = 21.674;                         % Area di riferimento - [m^2]
% CD = 0.56;                          % Coefficiente di resistenza - [-] 
% beta = W / (S * CD);                % Coefficiente balistico - [kg/m^2]

W = 55000;                          % Weight    [kg]
S = 400;                            % Reference area      [m^2]
Length = 83.1;                      % Reference length    [m]
CD = 0.5;                           % Drag coefficient    [adim] 
CL = 1.5*CD;                        % Lift coefficient    [adim] 
beta = W/(S*CD);    
E = 1.5;     %Skylon

% Diversi valori di Velocità Relativa - [m/s]
v_values = [v_high, v_low, v_mean]; 

% Parametri atmosferici e di riscaldamento
rho0 = 1.225;       % Densità a livello del mare - [kg/m^3]
Q_des = 80;    % Tasso di calore massimo desiderato - [W/cm^2]
BC = beta;          % Coefficiente balistico - [kg/m^2]
R_nose = 0.3048;    % Raggio del naso del veicolo - [m]
tol = Q_des / 100;  % Tolleranza per il tasso di calore - [W/cm^2]
q_const = 11030;    % Costante dell'equazione di riscaldamento

max_iter = 100;     % Numero massimo di iterazioni
D = zeros(3, 1);    % Preallocazione del vettore di decelerazione

% Inizio del ciclo di convergenza
for j = 1:length(v_values)
    i = 1;
    rho = 1e-10;            % Densità iniziale - [kg/m^3]
    DeltaQ = Inf;           % Inizializzazione di DeltaQ per entrare nel ciclo while

    % Calcoli iterativi
    while 1
        % Calcolo del tasso di calore
        v_ratio = v_values(j) / v_sat;
        rhod = rho / rho0;
        Qc = q_const * (1 / sqrt(R_nose)) * sqrt(rhod) * (v_ratio)^3.15;

        % Calcolo della variazione di Q rispetto alla variazione di rho
        dQ_drho = 0.5 * Qc / rho;

        % Calcolo di DeltaQ e Delta_rho
        DeltaQ = Qc - Q_des;
        Delta_rho = DeltaQ / dQ_drho;

        % Aggiornamento della densità
        rho = rho - Delta_rho;

        % Incremento dell'indice
        i = i + 1;

        % Condizioni di uscita dal ciclo
        if abs(DeltaQ) < tol || i > max_iter
            break;
        end
    end

    % Calcolo della pressione dinamica per il valore di velocità relativa
    q = 0.5 * rho * v_values(j)^2; % [Pa]

    % Calcolo di D
    D(j) = q / BC; % [m/s^2]
end

% Calcolo dei coefficienti quadratici utilizzando un fitting quadratico
p = polyfit(v_values, D, 2);

% Coefficienti quadratici
gamma3 = p(1);
gamma2 = p(2);
gamma1 = p(3);

% Stampa i coefficienti
fprintf('gamma1 = %.4f\n', gamma1);
fprintf('gamma2 = %.4f\n', gamma2);
fprintf('gamma3 = %.4f\n', gamma3);

% Verifica dei coefficienti rispetto alla letteratura
expected_gamma1 = 61.1;
expected_gamma2 = -0.015;
expected_gamma3 = 8.97e-7;

fprintf('Expected gamma1 = %.4f\n', expected_gamma1);
fprintf('Expected gamma2 = %.4f\n', expected_gamma2);
fprintf('Expected gamma3 = %.4f\n', expected_gamma3);

% Confronto con i valori attesi
fprintf('Difference in gamma1 = %.4f\n', gamma1 - expected_gamma1);
fprintf('Difference in gamma2 = %.4f\n', gamma2 - expected_gamma2);
fprintf('Difference in gamma3 = %.4f\n', gamma3 - expected_gamma3);

% Plot dei risultati
figure('Name', 'Curve Fit Drag Reference');
set(groot, 'defaultAxesTickLabelInterpreter', 'latex');
set(groot, 'defaulttextinterpreter', 'latex');
set(groot, 'defaultLegendInterpreter', 'latex');
plot(v_values, D, 'o', 'LineWidth', 1.3, 'Color', 'r');
hold on;
v_fit = linspace(min(v_values), max(v_values), 100);
D_fit = gamma1 + gamma2 * v_fit + gamma3 * v_fit.^2;
plot(v_fit, D_fit, 'LineWidth', 1.3, 'Color', 'b');
grid minor;
xlabel('\textbf{Velocity} $\left( \mathrm{m} / \mathrm{s} \right)$');
ylabel('\textbf{Drag deceleration} $\left( \mathrm{m} / \mathrm{s}^2 \right)$');
title('\textbf{Drag Reference Curve Fit}');
legend('Data Points', 'Quadratic Fit');
xlim([min(v_values) max(v_values)]);





%%  2. QUADRATIC COEFFICIENTS FOR CONSTANT HEAT RATE - [Ref]: Human Spaceflight, Pranke [book]

clc;
close all;
clear;

% Inizializzazioni
v_high = 7620;                      % Velocità Alta - [m/s]
v_low = 5791;                       % Velocità Bassa - [m/s]
v_mean = (v_high + v_low) * 0.5;    % Velocità Media - [m/s]
v_sat = 7837;                       % Velocità di saturazione - [m/s]

% Parametri del veicolo X38
% W = 11340;                          % Peso - [kg]
% S = 21.674;                         % Area di riferimento - [m^2]
% CD = 0.56;                          % Coefficiente di resistenza - [-] 
% beta = W / (S * CD);                % Coefficiente balistico - [kg/m^2]

W = 55000;                          % Weight    [kg]
S = 400;                            % Reference area      [m^2]
Length = 83.1;                      % Reference length    [m]
CD = 0.5;                           % Drag coefficient    [adim] 
CL = 1.5*CD;                        % Lift coefficient    [adim] 
beta = W/(S*CD);    
E = 1.5;     %Skylon

% Diversi valori di Velocità Relativa - [m/s]
v_values = [v_high, v_high-40, v_low, v_low+40, v_mean]; 

% Parametri atmosferici e di riscaldamento
rho0 = 1.225;       % Densità a livello del mare - [kg/m^3]
% Q_des = 131.297;    % Tasso di calore massimo desiderato - [W/cm^2] - X38
Q_des = 110; %skylon
BC = beta;          % Coefficiente balistico - [kg/m^2]
R_nose = 0.3048;    % Raggio del naso del veicolo - [m]
tol = Q_des / 100;  % Tolleranza per il tasso di calore - [W/cm^2]
q_const = 11030;    % Costante dell'equazione di riscaldamento

max_iter = 100;     % Numero massimo di iterazioni
D = zeros(5, 1);    % Preallocazione del vettore di decelerazione

% Inizio del ciclo di convergenza
for j = 1:length(v_values)
    i = 1;
    rho = 1e-10;            % Densità iniziale - [kg/m^3]
    DeltaQ = Inf;           % Inizializzazione di DeltaQ per entrare nel ciclo while

    % Calcoli iterativi
    while 1
        % Calcolo del tasso di calore
        v_ratio = v_values(j) / v_sat;
        rhod = rho / rho0;
        Qc = q_const * (1 / sqrt(R_nose)) * sqrt(rhod) * (v_ratio)^3.15;

        % Calcolo della variazione di Q rispetto alla variazione di rho
        dQ_drho = 0.5 * Qc / rho;

        % Calcolo di DeltaQ e Delta_rho
        DeltaQ = Qc - Q_des;
        Delta_rho = DeltaQ / dQ_drho;

        % Aggiornamento della densità
        rho = rho - Delta_rho;

        % Incremento dell'indice
        i = i + 1;

        % Condizioni di uscita dal ciclo
        if abs(DeltaQ) < tol || i > max_iter
            break;
        end
    end

    % Calcolo della pressione dinamica per il valore di velocità relativa
    q = 0.5 * rho * v_values(j)^2; % [Pa]

    % Calcolo di D
    D(j) = q / BC; % [m/s^2]
end

% Calcolo dei coefficienti quadratici utilizzando un fitting quadratico
p = polyfit(v_values, D, 2);

% Coefficienti quadratici
gamma3 = p(1);
gamma2 = p(2);
gamma1 = p(3);

% Stampa i coefficienti
fprintf('gamma1 = %.4f\n', gamma1);
fprintf('gamma2 = %.4f\n', gamma2);
fprintf('gamma3 = %.4f\n', gamma3);

% Verifica dei coefficienti rispetto alla letteratura
expected_gamma1 = 61.1;
expected_gamma2 = -0.015;
expected_gamma3 = 8.97e-7;

fprintf('Expected gamma1 = %.4f\n', expected_gamma1);
fprintf('Expected gamma2 = %.4f\n', expected_gamma2);
fprintf('Expected gamma3 = %.4f\n', expected_gamma3);

% Confronto con i valori attesi
fprintf('Difference in gamma1 = %.4f\n', gamma1 - expected_gamma1);
fprintf('Difference in gamma2 = %.4f\n', gamma2 - expected_gamma2);
fprintf('Difference in gamma3 = %.4f\n', gamma3 - expected_gamma3);

% Plot dei risultati
figure('Name', 'Curve Fit Drag Reference');
set(groot, 'defaultAxesTickLabelInterpreter', 'latex');
set(groot, 'defaulttextinterpreter', 'latex');
set(groot, 'defaultLegendInterpreter', 'latex');
plot(v_values, D, 'o', 'LineWidth', 1.3, 'Color', 'r');
hold on;
v_fit = linspace(min(v_values), max(v_values), 100);
D_fit = gamma1 + gamma2 * v_fit + gamma3 * v_fit.^2;
plot(v_fit, D_fit, 'LineWidth', 1.3, 'Color', 'b');
grid minor;
xlabel('\textbf{Velocity} $\left( \mathrm{m} / \mathrm{s} \right)$');
ylabel('\textbf{Drag deceleration} $\left( \mathrm{m} / \mathrm{s}^2 \right)$');
title('\textbf{Drag Reference Curve Fit}');
legend('Data Points', 'Quadratic Fit');
xlim([min(v_values) max(v_values)]);






