clc;
clear;



%% Temperature at wall - Skylon

clc;
% [1]: TPS Material Properties Database, NASA

% Parameters - RCC [1] 
% epsilon = 0.9;             % Emissivity [-]
% sigma = 5.67e-8;           % Stefan-Boltzmann constant [W/m^2 K^4]
% rho = 1580;                % Wall material density [kg/m^3]
% cp = 1485;                 % Wall material specific heat [J/kg·K]
% k = 1.4;                   % Wall material thermal conductivity [W/m·K]


% TUFI/AETB [1]
% epsilon = 0.873;           % Surface emissivity [-]
% sigma = 5.67e-8;           % Stefan-Boltzmann constant [W/m^2 K^4]
% rho = 8.01e2;              % Wall material density [kg/m^3]
% cp = 7.95e2;               % Wall material specific heat [J/kg·K]
% k = 6.84e-2;               % Wall material thermal conductivity [W/m·K]


% AETB rigid tiles [1]
epsilon = 0.9;               % Surface emissivity [-]
sigma = 5.67e-8;             % Stefan-Boltzmann constant [W/m^2 K^4]
rho = 1.28e2;                % Wall material density [kg/m^3]
cp = 6.28e2;                 % Wall material specific heat [J/kg·K]
k = 6.84e-2;                 % Wall material thermal conductivity [W/m·K]

T_w_initial = 288;           % Initial wall temperature [K]
t_final = 5000;              % Final time [s]
thickness = 0.2;             % Wall thickness [m]


% Heat flux vector and corresponding times
% t_q_conv = [0, 250, 500, 1000, 1700, 2200, 2500, 2850, 3100, 3600, 4000, 4500, 5000]; % Times corresponding to heat flux values
% q_conv_values = [2.2e4, 3.3e4, 7.3e4, 2.3e5, 3.6e5, 4.25e5, 4.7e5, 4.8e5, 4.5e5, 2.5e5, 5.4e4, 1.4e2, 30]; % Heat flux values

t_q_conv = t_i;
q_conv_values = HT_stagnation; 


% Interpolating heat flux with a spline
t_fine = linspace(min(t_q_conv), max(t_q_conv), 1000); % Increase to 1000 points for the spline
q_conv_spline = spline(t_q_conv, q_conv_values, t_fine);

% Function to evaluate the spline
q_conv = @(t) interp1(t_fine, q_conv_spline, t, 'linear', 'extrap');

% Time interval
t_values = linspace(0, t_final, 1000);

% Calculating the mass of the wall per unit area
mass_per_area = rho * thickness; % [kg/m^2]

% ODE function for the time-dependent energy balance with variable flux
odefun = @(t, T_w) (q_conv(t) - epsilon * sigma * T_w^4) / (mass_per_area * cp);

% Solving the ODE
[t, T_w] = ode45(odefun, [0 t_final], T_w_initial);

% Plotting temperature results over time
figure;
set(groot, 'defaultAxesTickLabelInterpreter', 'latex');
set(groot, 'defaulttextinterpreter', 'latex');
set(groot, 'defaultLegendInterpreter', 'latex');
plot(t, T_w, 'r-', 'LineWidth', 1.5, 'DisplayName', 'Wall Temperature');
xlabel('\textbf{Time} $\left( \mathrm{s} \right)$', 'Interpreter', 'latex');
ylabel('\textbf{Temperature} $\left( \mathrm{K} \right)$', 'Interpreter', 'latex');
title('\textbf{TPS Temperature over Time / thickness=20cm}', 'Interpreter', 'latex');
grid on;

% Plotting heat flux
figure;
plot(t_fine, q_conv_spline, 'b-', 'LineWidth', 1.5, 'DisplayName', 'Heat Flux');
xlabel('\textbf{Time} $\left( \mathrm{s} \right)$', 'Interpreter', 'latex');
ylabel('\textbf{Heat rate } $\left( \mathrm{W} / \mathrm{m}^2 \right)$', 'Interpreter', 'latex');
title('\textbf{Stagnation Point Heat Transfer} $\left( \mathrm{W} / \mathrm{m}^2 \right)$');
grid on;
ylim([0 6e5])

fprintf('Thickness :%6.2f cm\n', 100 * thickness);
fprintf('Weight :%6.2f kg/m^2\n', rho * thickness);


%% DIFFERENCE FINITE METHOD - SKYLON 


% AETB-TUFI
rho = 8.01e2;              % Density [kg/m^3]
cp = 7.95e2;               % Specific Heat [J/kg·K]
k = 6.84e-2;               % Thermal Conductivity [W/m·K]
epsilon = 0.873;           % Emissivity  [-]


sigma = 5.67e-8;           % Stefan-Boltzmann costant [W/m^2 K^4]
T_w_initial = 300;         % Initial Temperature [K]
t_final = 5000;            % Final Time [s]
thickness = 0.2;           % Thickness   [m]

% Discretization 
Nx = 50;                   % Number of spatial nodes
Nt = 5000;                 % Number of time steps
dx = thickness / (Nx-1);   % Space step
dt = t_final / (Nt-1);     % Time step

% Diffusion coefficient
alpha = k / (rho * cp);

% Initial condition
T = T_w_initial * ones(Nx, Nt);

% Interpolation of the Heat transfer
t_q_conv = t_i; 
q_conv_values = HT_stagnation;

t_fine = linspace(min(t_q_conv), max(t_q_conv), 1000);
q_conv_spline = spline(t_q_conv, q_conv_values, t_fine);

q_conv = @(t) interp1(t_fine, q_conv_spline, t, 'linear', 'extrap');

% Time interval
t_interpolated = linspace(0, t_final, Nt);

% Time loop with temperature update
for n = 1:Nt-1
    for i = 2:Nx-1
        T(i, n+1) = T(i, n) + alpha * dt / dx^2 * (T(i+1, n) - 2*T(i, n) + T(i-1, n));
    end
    
    % Updating the temperature to the external surface (x = 0)
     q_conv_n = q_conv(t_interpolated(n)); % Flusso di calore convettivo al tempo n
     %T(1, n+1) = T(1, n) + dt * (q_conv_n - epsilon * sigma * T(1, n)^4) / (rho * cp * dx);
     rad_term = epsilon * sigma * (T(1, n)^4);
     cond_term = k * (T(2, n) - T(1, n)) / dx;
     T(1, n+1) = T(1, n) + (2* dt / (rho * cp * dx)) * (-rad_term + cond_term + q_conv_n);

    
    % Temperature update at the inner surface (x = thickness) with adiabatic condition 
    T(Nx, n+1) = T(Nx, n) + alpha * dt / dx^2 * (T(Nx-1, n) - T(Nx, n));
end

% Display of results: Temperature as a function of thickness at different times
figure;
hold on;
set(groot,'defaultAxesTickLabelInterpreter','latex');
set(groot,'defaulttextinterpreter','latex');
set(groot,'defaultLegendInterpreter','latex');
plot(linspace(0, thickness, Nx), T(:, round(Nt*0.25)), 'DisplayName', 't = 1250 s', 'LineWidth',1);
plot(linspace(0, thickness, Nx), T(:, round(Nt*0.5)), 'DisplayName', 't = 2500 s', 'LineWidth',1);
plot(linspace(0, thickness, Nx), T(:, round(Nt*0.75)), 'DisplayName', 't = 3750 s', 'LineWidth',1);
plot(linspace(0, thickness, Nx), T(:, end), 'DisplayName', 't = 5000 s', 'LineWidth',1);
xlabel('\textbf{Thickness} [m]', 'Interpreter', 'latex');
ylabel('\textbf{Temperature} [K]', 'Interpreter', 'latex');
title('\textbf{Temperature Distribution along the Thickness}', 'Interpreter', 'latex');
legend show;
grid on;

% Display of results: Temperature as a function of time at different thicknesses
figure;
hold on;
set(groot,'defaultAxesTickLabelInterpreter','latex');
set(groot,'defaulttextinterpreter','latex');
set(groot,'defaultLegendInterpreter','latex');
plot(t_interpolated, T(1, :), 'DisplayName', 'x = 0 m (superficie esterna)', 'LineWidth',1.2);
xlabel('\textbf{Time} [s]', 'Interpreter', 'latex');
ylabel('\textbf{Temperature} [K]', 'Interpreter', 'latex');
title('\textbf{Evolution of TPS Temperature over Time}', 'Interpreter', 'latex');
grid on;

fprintf('Thickness :%6.2f cm\n', 100 * thickness);
fprintf('Weight :%6.2f kg/m^2\n', rho * thickness);
% plot(t_interpolated, T(round(Nx*0.25), :), 'DisplayName', 'x = 0.05 m');
% plot(t_interpolated, T(round(Nx*0.5), :), 'DisplayName', 'x = 0.1 m');
% plot(t_interpolated, T(round(Nx*0.75), :), 'DisplayName', 'x = 0.15 m');
% plot(t_interpolated, T(end, :), 'DisplayName', 'x = 0.2 m (superficie interna)');
% legend show;
% legend('Location','northwest')
