%% ATMOSPHERIC RE-ENTRY

close all;
clc;
clear

% Bibliography:
% [1] - Dynamics of Atmospheric Re-Entry, Frank J. Regan, Satya M. Anandakrishnanù
% [2] - Human Spaceflight - Pranke
% [3] - Aerodynamic and Aerothermodynamic Analysis of Space Mission Vehicles (A.Viviani, G. Pezzella)
% [4] - Space Vehicle Design - Michael D. Griffin , and James R. French
% [5] - Tommaso Molinari Master Thesis

% Conversion Unit
% 1 [m] = 39.3701 [in]; 1 [J] = [Btu]/1055.056; 1 [m] = 3.2808 [ft]; 1 [kg]=  2.2046 [lb];
% T(°F) = ((T(K) - 273.15) * 1.8) + 32; T(K) = (T(°F) + 459,67) * 5/9;
% 1 [Btu] = 1055,06 [J];
% 1 [ft] = 0.3048 [m]

%% ATMOSPHERE

%CONSTANTS
G = 6.67408e-11;            % Gravitational Constant                [m^3/(kg*s^2)]
R = 8.31432;                % Universal Constant for Ideal Gases    [J/mole*K]
rho0 = 1.225;               % Sea level desnity                     [kg m^-3]
H = 8000;                   % planet atmosphere density scale       [m]


%Earth
R_E = 6378e3;           % Radius of Earth                       [m]
M_E = 5.9724e24;        % Earth's Mass                          [kg]
g0_E = 9.81;            % Acceleration gravity                  [m/s^2]
T0_E = 288.15;          % Temperature at Sea Level              [K]
P0_E = 101325;          % Pressure at Sea Level                 [Pa]
Mm_E = 28.9644*10^-3;   % Molecular Mass                        [kg*mole^-1]


% Air specific heats ratio  [-]
gamma_gas_E = 1.4;      

% Earth's atmospheric layers altitude           [m]
H_layer_E = 1e3*[0 11 20 32 47 52 61 69 79 90 100 110 117.776];    

% Earth's atmopheric layers thermal gradient    [K/m]    
lambda_layer_E = 1e-3*[-6.5 0 1 2.8 0 -2 -4 -3 0 2 4.36 16.4596 0];


% Compute base temperatures and pressures
[Tb, Pb] = BaseTemperaturePressure3D(R, g0_E, T0_E, P0_E, Mm_E, H_layer_E, lambda_layer_E); %Temperatures and pressures at the beginning of each layer

% plotAtmosphere(H_layer_E, lambda_layer_E, Tb, Pb, R, g0_E, Mm_E);

z = 300e3;                           % orbit altitude [m]
z_de = 120e3;                        % h deorbit [m]
R_orbit = R_E + z;                   % orbit radius [m]
R_deorbit = R_E + z_de;              % re-entry radius [m]
Vsat = sqrt(G*M_E/R_orbit);          % orbital velocity [m/s]
v_de = sqrt(G*M_E/R_deorbit);        % velocità deobrit [m/s]

%% LIFTING RE-ENTRY


%Skylon data [5]
W = 56648;                          % Weight    [kg]
S = 382;                            % Reference area      [m^2]
Length = 83.1;                      % Reference length    [m]
CD = 0.5;                           % Drag coefficient    [-] 
CL = 1.5*CD;                        % Lift coefficient    [-] 
beta = (W*g0_E)/(S*CD);    


% Initial conditions
H0=120000;                 % Initial Altitude             [m]
V0=7800;                   % Initial Velocity             [m/s]
lat0 = 45;                 % Initial latitude             [deg]
lon0 = 8;                  % Initial longitude            [deg]
gamma0 = 0.1;              % Initial Flight Path Angle    [deg] - small value - Glide re_entry
heading0 = 0;              % Initial heading angle        [deg]

t_max = 7200;              % Maximum time for numerical integration [s]
Delta_t = 1;               % Time step   [s]


%Integration
[t_i, V_i, gamma_i, heading_i, H_i, r_i,lat_i, lon_i] = ...
    Integration_noRotation(G, M_E, R_E, g0_E, R, Mm_E, Tb, Pb, H_layer_E, lambda_layer_E, ...
     CL, CD, beta, H0, V0, gamma0, lat0, lon0, heading0, t_max, Delta_t);

% Fix vectors to the length
[t_i2, V_i2, gamma_i2, heading_i2, H_i2, r_i2 ,lat_i2, lon_i2] = fixVectors3D(t_i, V_i, gamma_i, heading_i, H_i, r_i, lat_i, lon_i);
t_i = t_i2;
V_i = V_i2;
gamma_i = gamma_i2;
H_i = H_i2;
r_i = r_i2;
lat_i = lat_i2;
lon_i = lon_i2;
heading_i = heading_i2;


% Atmospheric properties
[T_atmo, P_atmo, rho_atmo] = CalculateAtmosphere3D(Tb, Pb, H_layer_E, lambda_layer_E, R, g0_E, Mm_E, H_i); 

B = 1.458e-6;                                % Constant  [kg/(s*m*K^(1/2))]
Su = 110.4;                                  % Sutherland's constant [K]
mu_v = (B*T_atmo.^(3/2))./(T_atmo + Su);     % Dynamic viscosity  [Pa*s] - Sutherland Formula
nu = mu_v./rho_atmo;                         % Cinematic viscosity 

% Mach number [-]
Mach = V_i./(sqrt(gamma_gas_E*(R/Mm_E)*T_atmo)); 

% Dynamic Pressure [Pa]
P_dynamic = (1/2)*rho_atmo.*(V_i.^2);

% Acceleration [g]
a = -P_dynamic/beta + sind(gamma_i*180/pi);

% Reynolds Number [-]
Re = (rho_atmo.*V_i*Length)./(mu_v);  

% Stagnation heat transfer  [W/m^2] - [3] pag. 200, eq. (2.141)
R_n = 0.3048; %bluntness sphere - [m]
K = 11030e4; 
HT_stagnation = K * (1/R_n)^0.5 * (rho_atmo/rho0).^0.5 .* (V_i./Vsat).^3.15;   % Stagnation heat transfer - DKR formula [W/m^2] 
%q_stag_rad = 1.134 * R_n * (rho_atmo/rho0).^1.6 .* (V_i/1e4).^8.5; % VERY LOW - [3, pag 204]

% Heat Load
heat_load = cumtrapz(HT_stagnation);

%% DIFFERENT FPAs

% Skylon data [5]
% W = 56648;                          % Weight    [kg]
% S = 382;                            % Reference area      [m^2]
% Length = 83.1;                      % Reference length    [m]
% CD = 0.5;                           % Drag coefficient    [-] 
% CL = 1.5*CD;                        % Lift coefficient    [-] 
% beta = (W*g0_E)/(S*CD);    
% 
% % Initial conditions
% H0 = 120000;                 % Initial Altitude             [m]
% V0 = 7800;                   % Initial Velocity             [m/s]
% lat0 = 45;                   % Initial latitude             [deg]
% lon0 = 8;                    % Initial longitude            [deg]
% heading0 = 0;                % Initial heading angle        [deg]
% 
% t_max = 7200;                % Maximum time for numerical integration [s]
% Delta_t = 1;                 % Time step   [s]
% 
% gamma_values = [0.1, 0.5, 1.0];  % Array of flight path angles [deg]
% 
% num_gammas = length(gamma_values);
% % Preallocates cell array to store results
% results = cell(length(gamma_values), 1);
% 
% for i = 1:length(gamma_values)
%     gamma0 = gamma_values(i);    % Update initial flight path angle
% 
%     % Integration
%     [t_i, V_i, gamma_i, heading_i, H_i, r_i, lat_i, lon_i] = ...
%         Integration_noRotation(G, M_E, R_E, g0_E, R, Mm_E, Tb, Pb, H_layer_E, lambda_layer_E, ...
%         CL, CD, beta, H0, V0, gamma0, lat0, lon0, heading0, t_max, Delta_t);
% 
%     % Trim vectors to the length
%     [t_i2, V_i2, gamma_i2, heading_i2, H_i2, r_i2, lat_i2, lon_i2] = fixVectors3D(t_i, V_i, gamma_i, heading_i, H_i, r_i, lat_i, lon_i);
% 
%     % Atmospheric properties
%     [T_atmo, P_atmo, rho_atmo] = CalculateAtmosphere3D(Tb, Pb, H_layer_E, lambda_layer_E, R, g0_E, Mm_E, H_i2);
% 
%     B = 1.458e-6;                                % Constant  [kg/(s*m*K^(1/2))]
%     Su = 110.4;                                  % Sutherland's constant [K]
%     mu_v = (B*T_atmo.^(3/2))./(T_atmo + Su);     % Dynamic viscosity  [Pa*s] - Sutherland Formula
%     nu = mu_v./rho_atmo;                         % Cinematic viscosity 
% 
%     % Mach number [-]
%     Mach = V_i2./(sqrt(gamma_gas_E*(R/Mm_E)*T_atmo)); 
% 
%     % Dynamic Pressure [Pa]
%     P_dynamic = (1/2)*rho_atmo.*(V_i2.^2);
% 
%     % Acceleration [g]
%     a = -P_dynamic/beta + sind(gamma_i2*180/pi);
% 
%     % Reynolds Number [-]
%     Re = (rho_atmo.*V_i2*Length)./(mu_v);  
% 
%     % Stagnation heat transfer  [W/m^2] - [3] pag. 200, eq. (2.141)
%     R_n = 0.3048; %bluntness sphere - [m]
%     K = 11030e4; 
%     HT_stagnation = K * (1/R_n)^0.5 * (rho_atmo/rho0).^0.5 .* (V_i2./Vsat).^3.15;   % Stagnation heat transfer - DKR formula [W/m^2] 
% 
%     % Heat Load
%     heat_load = cumtrapz(HT_stagnation);
% 
% 
%     results{i} = struct('t_i', t_i2, 'V_i', V_i2, 'gamma_i', gamma_i2, 'heading_i', heading_i2, ...
%                         'H_i', H_i2, 'r_i', r_i2, 'lat_i', lat_i2, 'lon_i', lon_i2, 'Mach', Mach, ...
%                         'P_dynamic', P_dynamic, 'Re', Re, 'HT_stagnation', HT_stagnation, ...
%                         'heat_load', heat_load, 'a', a);
% end
% 
% 
%  plot_results_gamma(gamma_values, results);


%% PLOT SOLUTIONS

% plotVsTime3D(t_i,H_i,V_i,gamma_i,Mach)
% plot_solutions3D(V_i,H_i,Mach,HT_stagnation,gamma_i,P_dynamic,Re,heat_load,t_i,r_i,a)

%% GT
% lat_i = rad2deg(lat_i);
% lon_i = rad2deg(lon_i);
% 
% % Fix longitud[-180, 180]
% lon_i = mod(lon_i + 180, 360) - 180;
% 
% figure()
% plot(t_i, lat_i);
% xlabel('Time [s]');
% ylabel('Latitude [deg]');
% title('Latitude vs. Time');
% 
% figure()
% plot(t_i, lon_i);
% xlabel('Time [s]');
% ylabel('Longitude [deg]');
% title('Longitude vs. Time');
% 
% % Ground track
% figure;
% worldmap('World');
% load coastlines
% plotm(coastlat, coastlon)
% plotm(lat_i, lon_i, 'r', 'LineWidth', 2);
% title('Ground Track of Re-entry Vehicle');

%% RE-ENTRY CORRIDOR 

phi = 0;  %bank angle[deg]
E=1.5;

% Calculate Dref1 using the current value of phi
Dref1 = (g0_E * (1 - (V_i.^2 / Vsat^2)))/(E*cos(phi*pi/180));

figure('Name','Re-Entry Corridor');
set(groot,'defaultAxesTickLabelInterpreter','latex');
set(groot,'defaulttextinterpreter','latex');
set(groot,'defaultLegendInterpreter','latex');
plot(V_i,Dref1, 'LineWidth',1.3,'Color','r')
hold on
grid minor
xlabel('\textbf{Velocity} $\left( \mathrm{m} / \mathrm{s} \right)$')
ylabel('\textbf{Drag deceleration} $\left( \mathrm{m} / \mathrm{s}^2 \right)$')
title('\textbf{Re-entry Corridor / D-V Map}')
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%% HEAT RATE CONSTRAINT%%%%%%%%%%%%%%%
G1 = 145.7044;
G2 = -0.0353;
G3 = 2.20e-06;


% G1=207.5851;
% G2=-0.0503;
% G3=3.14e-06;
Dref2 = G1 + G2*V_i + G3*V_i.^2;
plot(V_i,Dref2, 'LineWidth',1.3,'Color','b')
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%% ACCELERATION CONSTRAINT %%%%%%%%%%%%%%%
gammae = 2; %flight path angle
Gmax = ((v_de)^2 * sin(gammae*pi/180)) / (2*exp(1)*g0_E*H);
GL_max=(g0_E*exp(2*gammae*pi/180/E))/E;    %Gl/Gmax
Gwing=GL_max*Gmax;     %Gwing=Gl/Gmax*Gmax

Dref3 = Gwing/(sqrt(1 + E^2));
Dref3 = Dref3*ones(length(V_i),1)';
plot(V_i,Dref3,'--', 'LineWidth',1.3,'Color','k')
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%% DYNAMIC PRESSURE CONSTRAINT %%%%%%%%
Nmax = 5e3;     %[Pa] %SKYLON
Dref4 = g0_E*Nmax/(beta);
Dref4 = Dref4*ones(length(V_i),1)';
plot(V_i,Dref4, 'LineWidth',1.3,'Color','g')
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%----------------------------%
%%%%%% NOMINAL PROFILE%%%%%%%%
%----------------------------%

%%%%%%% CONSTANT HEAT RATE [80 kW/m2] %%%%%%%%%%%%%
G1n=73.9640;
G2n=-0.0177; %-0.0165;
G3n=1.0879e-06;
Dnom1 = G1n + G2n*V_i + G3n*V_i.^2;
% plot(V_i,Dnom1, 'LineWidth',1.3)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%% CONSTANT DRAG %%%%%%%%%%%%%%
V_initial = 6000;           % Initial velocity in m/s 
V_final = 2500;             % Final velocity in m/s
V_step = 100;               % Step size in m/s
% Initialize variables
velocities = V_initial:-V_step:V_final;
drag_diff = zeros(size(velocities));
avg_drag = zeros(size(velocities));

% Interpolation
Dref1_interp = interp1(V_i, Dref1, velocities, 'linear', 'extrap');
Dref3_interp = interp1(V_i, Dref3, velocities, 'linear', 'extrap');
Dref4_interp = interp1(V_i, Dref4, velocities, 'linear', 'extrap');

% Compute drag differences and average drag accelerations
for i = 1:length(velocities)
    drag_upper_boundary = min(Dref3_interp(i), Dref4_interp(i));
    drag_lower_boundary = Dref1_interp(i);
    drag_diff(i) = drag_upper_boundary - drag_lower_boundary;
    avg_drag(i) = (drag_upper_boundary + drag_lower_boundary) / 2;
end

% Find the velocity with the minimum drag difference
[~, min_idx] = min(drag_diff);
V_min_diff = velocities(min_idx);
avg_drag_at_min_diff = avg_drag(min_idx);
Dnom2 = ones(1,length(velocities))*avg_drag_at_min_diff;
% plot(velocities, Dnom2, 'LineWidth', 1.3)

% Intersection between Dnom1 e Dnom2
V_interp = linspace(min(V_i), max(V_i), 10000);
Dnom1_interp = interp1(V_i, Dnom1, V_interp, 'spline');
Dnom2_interp = interp1(velocities, Dnom2, V_interp, 'spline');
[~, intersect_idx] = min(abs(Dnom1_interp - Dnom2_interp));
V_intersect = V_interp(intersect_idx);
D_intersect = Dnom1_interp(intersect_idx);
% plot(V_intersect, D_intersect, 'ko', 'MarkerFaceColor', 'k')


idx = (V_i >= V_intersect) & (V_i <= 8000);
V_i_interval = V_i(idx);
Dnom1_interval = Dnom1(idx);
plot(V_i_interval, Dnom1_interval, 'LineWidth', 1.5, 'Color', 'k');
grid on;

% Plot Dnom2 
V_nom2 = linspace(V_final, V_intersect, length(V_i));
Dnom2_segment = ones(1, length(V_nom2)) * avg_drag_at_min_diff;
plot(V_nom2, Dnom2_segment, 'LineWidth', 1.5, 'Color', 'k')
% plot(velocities, Dnom2, 'LineWidth', 1.3)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%PROFILE FOR LINEAR DRAG%%%%%%%%%%%%%
Di = Dnom2_segment(1);                  % Initial Drag - [m/s^2]
vi = V_final;                           % Initial Velocity at the termination of constant drag - [m/s]
hi = -H*log(2*Di*beta/(rho0*vi^2));     % Initial Altitude - [m]
Df = max(Dref1)+0.5;                    % Final Drag - [m/s^2]
vf = 800;                               % Final Velocity - [m/s]
hf = -H*log(2*Df*beta/(rho0*vf^2));     % Final Altitude - [m]
Ei = g0_E*hi + vi^2/2;                  % Initial Energy - [m^2/s^2]
Ef = g0_E*hf + vf^2/2;                  % Final Energy - [m^2/s^2]  
G5 = (Di-Df)/(Ei-Ef);                   % [m^-1]
vlin = linspace(vf,vi,length(V_i));
h = linspace(hf,hi,length(vlin));
E = g0_E.*h+0.5*vlin.^2;
Dref5 = Df + G5.*(E - Ef);              % Drag Reference for linear drag
plot(vlin,Dref5, 'LineWidth',1.5, 'Color', 'k')
text(1200,10,'\textbf{Linear Drag}','Rotation',30)
text(3000,13.5,'\textbf{Constant Drag}')
text(5500,11,'\textbf{Constant Heat Rate}','Rotation',-38)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

legend('Equilibrium Glide','Heat Rate', 'Acceleration', 'Dynamic Pressure','Nominal Profile')
xlim([1000 7800])
ylim([0 30])

%% REENTRY CORRIDOR- Altitude vs Velocity graph


% Equilibrium Glide 
rho_glide = 2*(W*g0_E/(S*CL)) * (1./(V_i.^2) - 1/7801^2);
h_glide = -H*log(rho_glide/1.225);

figure('Name', 'Re-entry corridor - Altitude vs Velocity');
set(groot,'defaultAxesTickLabelInterpreter','latex');
set(groot,'defaulttextinterpreter','latex');
set(groot,'defaultLegendInterpreter','latex');
plot(V_i,h_glide/1e3, 'Color','r', 'LineWidth',1.5)
ylabel("\textbf{Altitude} $\left( \mathrm{km}\right)$");
xlabel("\textbf{Velocity} $\left( \mathrm{m} / \mathrm{s} \right)$");
title('\textbf{Re-entry corridor in h-V domain}')
grid on;
grid minor;
hold on
ylim([0 120])

%Maximum Deceleration
N = sqrt(CD^2+CL^2).*((0.5.*rho_atmo.*V_i.^2*S)/(W*g0_E));
Nmax = 2; %Skylon
rho_dec = 2*Nmax* (W)./(sqrt(CL^2+CD^2)*S*V_i);
h_dec = -H*log(rho_dec/1.225);
plot(V_i,h_dec/1e3, 'Color','b', 'LineWidth',1.5)

%Heat rate
Qmax=100; % [W/cm^2] Skylon
c = 11030;
rho_heat = 1.225 * ( (R_n^0.5 * Qmax)/c * (v_de./V_i).^(3.15) ).^2;
h_heat = -H*log(rho_heat/1.225);
plot(V_i,h_heat/1e3,'LineWidth',1.5)

%Velocity vs. Altitude
% hold on;
% plot(V_i, H_i/1e3, 'LineWidth',1,'Color',[0.4940 0.1840 0.5560]);
% hold off;

legend('Equilibrium Glide', 'Maximum Deceleration','Maximum Heat rate')


%% POST PROCESSING

velocity_astos = readmatrix('exports\velocity.xls');
altitude_astos = readmatrix('exports\altitude.xls');
time_astos = readmatrix('exports\time.xls');
mach_astos = readmatrix('exports\mach.xls');
heat_flux_astos = readmatrix('exports\heat_flux.xls');
dyn_pressure_astos = readmatrix('exports\dyn_pressure.xls');
gamma_astos = readmatrix('exports\gamma.xls');
range_astos = readmatrix('exports\range.xls');

velocity_sim = velocity_astos(:,1);
altitude_sim = altitude_astos(:,1);
time_sim = time_astos(:,1);
mach_sim = mach_astos(:,1);
heat_flux_sim = heat_flux_astos(:,1);
dyn_pressure_sim = dyn_pressure_astos(:,1);
gamma_sim = gamma_astos(:,1);
range_sim = range_astos(:,1);

figure('Name', 'Velocity vs. Altitude - Skylon');
plot(velocity_sim,altitude_sim,'DisplayName','ASTOS simulation','LineWidth',1)
hold on
plot(V_i/1e3, H_i/1e3,'DisplayName','Conceptual design', 'LineWidth',1);
title('\textbf{Velocity vs. Altitude - Skylon}')
ylabel("\textbf{Altitude} $\left( \mathrm{km} \right)$");
xlabel("\textbf{Velocity} $\left( \mathrm{km} / \mathrm{s} \right)$");
legend show

figure('Name', ' Altitude vs Time - Skylon');
plot(time_sim, altitude_sim,'DisplayName','ASTOS simulation','LineWidth',1)
hold on
plot(t_i, H_i/1e3,'DisplayName','Conceptual design', 'LineWidth',1);
title('\textbf{Altitude vs Time - Skylon}')
ylabel("\textbf{Altitude} $\left( \mathrm{km} \right)$");
xlabel('\textbf{Time} $\left( \mathrm{s} \right)$')
legend show

figure()
plot(time_sim,velocity_sim,'DisplayName','ASTOS simulation','LineWidth',1)
hold on 
plot(t_i, V_i/1e3,'DisplayName','Conceptual design', 'LineWidth',1)
title('\textbf{Velocity vs Time - Skylon}')
xlabel('\textbf{Time} $\left( \mathrm{s} \right)$')
ylabel("\textbf{Velocity} $\left( \mathrm{m} / \mathrm{s} \right)$");
legend show

figure()
plot(time_sim,mach_sim,'DisplayName','ASTOS simulation','LineWidth',1)
hold on
plot(t_i,Mach,'DisplayName','Conceptual design', 'LineWidth',1)
title('\textbf{Mach vs Time - Skylon}')
xlabel('\textbf{Time} $\left( \mathrm{s} \right)$')
ylabel('Mach number')
legend show

figure()
plot(heat_flux_sim,altitude_sim,'DisplayName','ASTOS simulation','LineWidth',1)
hold on
% plot(qprova/1e3,H_i/1e3,'DisplayName','Conceptual design ', 'LineWidth',1)
plot(HT_stagnation/1e3,H_i/1e3,'DisplayName','Conceptual design - DKR formula', 'LineWidth',1)
title('\textbf{Heat transfer at stagnation point - Skylon}')
xlabel('\textbf{Stagnation Point Heat Transfer} $\left( \mathrm{kW} / \mathrm{m}^2 \right)$')
ylabel("\textbf{Altitude} $\left( \mathrm{km} \right)$");
legend show

figure()
plot(dyn_pressure_sim,altitude_sim,'DisplayName','ASTOS simulation','LineWidth',1)
hold on
plot(P_dynamic,H_i/1e3,'DisplayName','Conceptual design', 'LineWidth',1)
title('\textbf{Dynamic Pressure at stagnation point - Skylon}')
xlabel("\textbf{Dynamic Pressure} $\left( \mathrm{Pa} \right)$");
ylabel("\textbf{Altitude} $\left( \mathrm{km} \right)$");
legend show

figure()
plot(gamma_sim,altitude_sim,'DisplayName','ASTOS simulation','LineWidth',1)
hold on
plot(-gamma_i*180/pi,H_i/1e3,'DisplayName','Conceptual design', 'LineWidth',1)
title('\textbf{Flight path angle - Skylon}')


figure()
plot(mach_sim,altitude_sim,'DisplayName','ASTOS simulation','LineWidth',1)
hold on
plot(Mach, H_i/1e3,'DisplayName','Conceptual design', 'LineWidth',1)
title('\textbf{Mach number during mission time - Skylon}')
xlabel('\textbf{Mach number}')
ylabel("\textbf{Altitude} $\left( \mathrm{km} \right)$");
legend show

figure()
plot(range_sim, altitude_sim,'DisplayName','ASTOS simulation','LineWidth',1)
hold on
plot(r_i/1e3, H_i/1e3,'DisplayName','Conceptual design', 'LineWidth',1)
title('\textbf{Drownrange - Skylon}')
xlabel('\textbf{Range} $\left( \mathrm{km} \right)$')
ylabel("\textbf{Altitude} $\left( \mathrm{km} \right)$");
legend show
xlim([0 30000])


